/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.notification;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Hashtable;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.notification.PropertySubstituter;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;

/**
 * Property substituter unit tests.
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 */
public class PropertySubstituterUnitTest extends TestCase {
	private static final String FILE_NAME = "foo.txt";
	private static final String NAME_TWO = "two.csv";
	private static final String MESSAGE_ID = "urn:1234";
	
	private Message createMessage() {
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
		assertNotNull("Message is null", msg);
		
		try {
			msg.getHeader().getCall().setMessageID(new URI(MESSAGE_ID));
			msg.getAttachment().addItemAt(0, FILE_NAME);
		} catch (URISyntaxException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		return msg;
	}
	
	public void testSubstitution() {
		Message msg = createMessage();
		msg.getProperties().setProperty("filename", FILE_NAME);
		String foo = PropertySubstituter.replaceArguments("{filename}", msg);
		assertEquals("Property substitution should have changed "+ foo +" to " + FILE_NAME, foo, FILE_NAME);
		
		String notfound = PropertySubstituter.replaceArguments("{notfound}", msg);
		assertEquals("Result of property substitution for " + foo + " should be the original value, not " + notfound, "{notfound}", notfound);

		msg.getBody().add("filename", NAME_TWO);
		String result = PropertySubstituter.replaceArguments("body.filename", msg);
		assertEquals("Could not find the string in the body of the message at location body.filename", result, NAME_TWO);
		
		
		String resultTwo = PropertySubstituter.replaceArguments("{body.filename}", msg);
		assertEquals("Could not find the string in the body of the message at location body.filename", resultTwo, NAME_TWO);
		
		String resultThree = PropertySubstituter.replaceArguments("{body.filename}-{body.hashtable}", msg);
		assertEquals("Could not find the string in the body of the message at location body.filename", resultThree, NAME_TWO + "-{body.hashtable}");


		String resultFour = PropertySubstituter.replaceArguments("{body.filename}-{header.call.MessageID}", msg);
		assertEquals("Could not find MessageID", resultFour, NAME_TWO + "-" + MESSAGE_ID);

		
		msg.getBody().add("hashtable", new Hashtable());
		String hashResult = PropertySubstituter.replaceArguments("body.hashtable", msg);
		assertEquals(hashResult, "body.hashtable");
		
		String propFromObjectMapper = PropertySubstituter.replaceArguments("properties.filename", msg);
		assertEquals("Could not find property from ObjectMapper.properties - expected " + FILE_NAME + ", found " + propFromObjectMapper, 
				propFromObjectMapper, FILE_NAME);
		
		String propNotInObjectMapper = PropertySubstituter.replaceArguments("properties.notfound", msg);
		assertEquals("Property substitution for " + propNotInObjectMapper + " should find no property on the message", "properties.notfound", propNotInObjectMapper);				
		
		String header = PropertySubstituter.replaceArguments("header.call.MessageID", msg);
		assertEquals("Could not find MessageID", header, MESSAGE_ID);
		
		String notFoundHeader = PropertySubstituter.replaceArguments("header.call.NotFound", msg);
		assertEquals("Could not find MessageID", notFoundHeader, "header.call.NotFound");
		
		String attachment = PropertySubstituter.replaceArguments("attachment.call.MessageID", msg);
		assertEquals("Could not find MessageID", header, MESSAGE_ID);
		
		String notFoundAttachment = PropertySubstituter.replaceArguments("attachment.call.NotFound", msg);
		assertEquals("Could not find MessageID", notFoundAttachment, "attachment.call.NotFound");
	}
}