/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.services.rules.RuleServiceException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.mapping.ObjectMapper;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;

/**
 * Utit methods for handling Rules related configuration operations.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class RuleConfigUtil
{
    private RuleConfigUtil()
    {
    }
    
    /**
     * Extracts the Object instances specified in the object names of the entryPointMap argument from the ESB Message Object.
     * 
     * @param entryPointMap Map of that contains object names as it's values. These are the objects that should be extracted from the ESB Message.
     * @param message The ESB Message object.
     * @return Map<String, List<Object> The same contents as the passed in Map but instead of a List of Strings the values are a List of Objects which
     *                                 have been extracted from the ESB Message object.
     * @throws RuleServiceException
     */
    public static Map<String, List<Object>> extractObjectsFromMessage(final Map<String, List<String>> entryPointMap, final Message message) throws RuleServiceException
    {
        AssertArgument.isNotNull(entryPointMap, "entryPointMap");
        AssertArgument.isNotNull(message, "message");
        try
        {
            final Map<String, List<Object>> newEntryPointMap = new HashMap<String, List<Object>>();
            
            ObjectMapper objectMapper = new ObjectMapper();
            for (Entry<String, List<String>> entry : entryPointMap.entrySet())
            {
                final String entryPointName = entry.getKey();
                final List<Object> extractedObjects = new ArrayList<Object>();
                for (String objectName : entry.getValue())
                {
                    Object objectFromMessage = objectMapper.getObjectFromMessage(message, objectName);
                    extractedObjects.add(objectFromMessage);
                }
                newEntryPointMap.put(entryPointName, extractedObjects);
            }
            return newEntryPointMap;
        } 
        catch (final ObjectMappingException e)
        {
            throw new RuleServiceException("Exception while trying to extract the objects '" + entryPointMap + "' from the message", e);
        }
    }
    
}
