/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2009
 */
package org.jboss.soa.esb.actions.soap.proxy;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.routing.http.HttpRouter;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.http.HttpClientFactory;
import org.jboss.soa.esb.http.configurators.AuthBASIC;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;

/**
 * Implements http-specific routing of the SOAP request to the target webservice.  All config properties are passed through to a
 * wrapped instance of {@link HttpRouter}, so any of it's config properties can be added to the {@link SOAPProxy} configuration.
 * 
 * @author dward at jboss.org
 */
class HttpSOAPProxyTransport extends SOAPProxyTransport
{
	
	private static Logger logger = Logger.getLogger(HttpSOAPProxyTransport.class);
	
	private String endpointAddress;
	private HttpRouter httpRouter;
	
	HttpSOAPProxyTransport(ConfigTree config, final MessagePayloadProxy payloadProxy, String defaultEndpointAddress) throws ConfigurationException
	{
		ConfigTree cloned_config = null;
		ConfigTree parent_config = config.getParent();
		if (parent_config != null) {
			final String maxThreads = parent_config.getAttribute(ListenerTagNames.MAX_THREADS_TAG);
			if (maxThreads != null) {
				parent_config = new ConfigTree("parent");
				parent_config.setAttribute(ListenerTagNames.MAX_THREADS_TAG, maxThreads);
				cloned_config = config.cloneObj(parent_config);
			}
		}
		if (cloned_config == null) {
			cloned_config = config.cloneObj();
		}
		
		setAttribute(cloned_config, "MappedHeaderList", "Content-Type, Accept, Authorization, SOAPAction");
		setAttribute(cloned_config, "method", "POST");
		setAttribute(cloned_config, "responseType", "STRING");
		
		endpointAddress = setAttribute(cloned_config, "endpointUrl", defaultEndpointAddress);
		
		String file = cloned_config.getAttribute("file");
		if (file != null)
		{
			boolean found_file_child = false;
			ConfigTree[] children = cloned_config.getChildren(HttpClientFactory.HTTP_CLIENT_PROPERTY);
			for (ConfigTree child : children)
			{
				if ( "file".equals(child.getName()) )
				{
					found_file_child = true;
					break;
				}
			}
			if (!found_file_child)
			{
				ConfigTree file_child = new ConfigTree(HttpClientFactory.HTTP_CLIENT_PROPERTY, cloned_config);
				file_child.setAttribute("name", "file");
				file_child.setAttribute("value", file);
				if ( logger.isDebugEnabled() )
				{
					logger.debug("added HttpRouter http-client-property name [file] value [" + file + "]");
				}
			}
		}
		
		if ( cloned_config.getBooleanAttribute("clientCredentialsRequired", true) )
		{
			for (String name : new String[]{AuthBASIC.class.getName(), AuthBASIC.class.getSimpleName()})
			{
				ConfigTree skip_child = new ConfigTree(HttpClientFactory.HTTP_CLIENT_PROPERTY, cloned_config);
				skip_child.setAttribute("name", name + ".skip");
				skip_child.setAttribute("value", "true");
			}
		}
		
		httpRouter = new HttpRouter(cloned_config) {
			@Override
			public MessagePayloadProxy getPayloadProxy() {
				return payloadProxy;
			}
		};
	}
	
	// package-protected for testing purposes, see SOAPProxyUnitTest.test_maxThreads*()
	HttpRouter getHttpRouter() {
		return httpRouter;
	}
	
	public String getEndpointAddress()
	{
		return endpointAddress;
	}
	
	public void initialise() throws ActionLifecycleException
	{
		httpRouter.initialise();
	}
	
	public Message process(Message message) throws ActionProcessingException
	{
		return httpRouter.process(message);
	}
	
	public void destroy() throws ActionLifecycleException
	{
		httpRouter.destroy();
	}
	
	private String setAttribute(ConfigTree config, String name, String defaultValue)
	{
		boolean defaulting = false;
		String value = config.getAttribute(name);
		if (value == null)
		{
			defaulting = true;
			value = defaultValue;
			config.setAttribute(name, value);
		}
		if ( logger.isDebugEnabled() )
		{
			logger.debug(
				(defaulting ? "defaulted" : "respected") + " HttpRouter attribute [" + name + "] " +
				(defaulting ? "to" : "of") + " [" + value + "]" );
		}
		return value;
	}
	
}
