/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * (C) 2005-2008, JBoss Inc.
 */
 	package org.jboss.soa.esb.actions.soap;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;
import java.util.Map;

import junit.framework.TestCase;

import org.jboss.soa.esb.actions.soap.SchemaDeployer;
import org.jboss.soa.esb.actions.soap.SchemaDeployerImpl;
import org.jboss.soa.esb.actions.soap.SchemaLocationParser;
import org.jboss.soa.esb.util.XPathUtil;

public class SchemaLocatorUnitTest extends TestCase {
	public SchemaLocatorUnitTest() {
		
	}
	
	public void testSchemaLocationExternal() throws Exception {
    	
		StringBuffer fileData = getFileData("/test-schema-ext.wsdl");
		
		SchemaDeployer sd = (SchemaDeployer) new SchemaDeployerImpl();
    	SchemaLocationParser slp = new SchemaLocationParser(sd);

    	String data = slp.parse(fileData.toString());

    	assertEquals(slp.getSoapAddress(), "http://localhost:8080/jaxws-jbws2526");
		assertEquals(
				"http://localhost:8080/jaxws-jbws2526?wsdl&resource=jbws2526.xsd",
				XPathUtil.getNodeFromXPathExpression(data, "/definitions/types/schema/import").getAttribute("schemaLocation"));
    	
    	Map<String, String> resources = sd.getResources();
    	assertEquals(1, resources.size());
    	
		for (String key : resources.keySet()) {
			assertEquals(key, "jbws2526.xsd");
		}

	}
	
	public void testSchemaLocationInternal() throws Exception{
		StringBuffer fileData = getFileData("/test-schema-int.wsdl");
		SchemaDeployer sd = new SchemaDeployerImpl() {
			// mock method to eschew actual network access
			@Override
			public String getXSDString(URL url, File tempFile) throws IOException {
				return "!!! Dummy XSD data !!!";
			}
		};
		SchemaLocationParser slp = new SchemaLocationParser(sd);
		String data = slp.parse(fileData.toString());
		
		assertEquals(slp.getSoapAddress(), "http://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS");
		assertEquals(
				"http://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS?wsdl&resource=HelloWorldService.xsd",
				XPathUtil.getNodeFromXPathExpression(data, "/definitions/types/schema/import").getAttribute("schemaLocation"));
		
		Map<String, String> resources = sd.getResources();
		assertEquals(1, resources.size());
		assertEquals("!!! Dummy XSD data !!!", resources.get("HelloWorldService.xsd"));
	}
	
	public void testSchemaLocationInternal_cxf() throws Exception {
		StringBuffer fileData = getFileData("/test-schema-int-cxf.wsdl");
		SchemaDeployer sd = new SchemaDeployerImpl() {
			// mock method to eschew actual network access
			@Override
			public String getXSDString(URL url, File tempFile) throws IOException {
				return "!!! Dummy XSD data !!!";
			}
		};
		SchemaLocationParser slp = new SchemaLocationParser(sd);
		String data = slp.parse(fileData.toString());

		assertEquals(slp.getSoapAddress(), "http://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS");
		assertEquals(
				"http://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS?wsdl&resource=HelloWorldService.xsd",
				XPathUtil.getNodeFromXPathExpression(data, "/definitions/types/schema/import").getAttribute("schemaLocation"));

		Map<String, String> resources = sd.getResources();
		assertEquals(1, resources.size());
		assertEquals("!!! Dummy XSD data !!!", resources.get("HelloWorldService.xsd"));
	}
	
	public void testSchemaLocationInternal_wsdlPrefixed() throws Exception {
		StringBuffer fileData = getFileData("/test-schema-int-prefixed.wsdl");
		SchemaDeployer sd = new SchemaDeployerImpl() {
			// mock method to eschew actual network access
			@Override
			public String getXSDString(URL url, File tempFile) throws IOException {
				return "!!! Dummy XSD data !!!";
			}
		};
		SchemaLocationParser slp = new SchemaLocationParser(sd);
		String data = slp.parse(fileData.toString());

		assertEquals(slp.getSoapAddress(), "http://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS");
		assertEquals(
				"http://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS?wsdl&resource=HelloWorldService.xsd",
				XPathUtil.getNodeFromXPathExpression(data, "/definitions/types/schema/import").getAttribute("schemaLocation"));

		Map<String, String> resources = sd.getResources();
		assertEquals(1, resources.size());
		assertEquals("!!! Dummy XSD data !!!", resources.get("HelloWorldService.xsd"));
	}
	
	public void testSchemaLocationInternal_https() throws Exception {
		StringBuffer fileData = getFileData("/test-schema-int-https.wsdl");
		SchemaDeployer sd = new SchemaDeployerImpl() {
			// mock method to eschew actual network access
			@Override
			public String getXSDString(URL url, File tempFile) throws IOException {
				return "!!! Dummy XSD data !!!";
			}
		};
		SchemaLocationParser slp = new SchemaLocationParser(sd);
		String data = slp.parse(fileData.toString());

		assertEquals(slp.getSoapAddress(), "https://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS");
		assertEquals(
				"https://127.0.0.1:8080/HelloWorld_ESB/http/HelloWorldWS?wsdl&resource=HelloWorldService.xsd",
				XPathUtil.getNodeFromXPathExpression(data, "/definitions/types/schema/import").getAttribute("schemaLocation"));

		Map<String, String> resources = sd.getResources();
		assertEquals(1, resources.size());
		assertEquals("!!! Dummy XSD data !!!", resources.get("HelloWorldService.xsd"));
	}

	private StringBuffer getFileData(String filePath) throws FileNotFoundException,
			IOException {
		StringBuffer fileData = new StringBuffer(1000);
		URL testSchema = getClass().getResource(filePath);
		
		BufferedReader reader = new BufferedReader(new FileReader(testSchema.getPath()));
		char[] buffer = new char[1024];
		int length=0;
		while ((length=reader.read(buffer)) != -1) {
			String readData = String.valueOf(buffer, 0, length);
			fileData.append(readData);
			buffer = new char[1024];
		}
		reader.close();
		return fileData;
	}
	
}
