/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2010
 */
package org.jboss.soa.esb.services.soapui;

import org.jboss.aop.joinpoint.MethodInvocation;

import com.eviware.soapui.impl.wsdl.support.wsdl.WsdlLoader;

/**
 * Aspect used to override soapUI's WsdlContext$Loader.getWsdlLoader() so that an
 * InheritableThreadLocal's EsbWsdlLoader is always used instead of a UrlWsdlLoader,
 * which sometimes would erroneously get used by a SwingWorker Thread even though
 * the EsbWsdlLoader is passed into the WsdlContext's load method by WsdlImporter.
 * Setting a WsdlContext (or even a shared DefinitionCache) after the WsdlInterfaces
 * are returned by the WsdlProject is unfortunately too late, so this fix is necessary.
 * 
 * <p>The associated jira issue is <a href="https://jira.jboss.org/browse/JBESB-3276">JBESB-3276</a></p>
 * 
 * @see {@link SoapUIClientService#getWsdlInterfaces(java.lang.String,java.util.Properties)} for setting the appropriate EsbWsdlLoader
 * 
 * @author dward at jboss.org
 */
public class WsdlLoaderAspect {
	
	private static final InheritableThreadLocal<WsdlLoader> wsdlLoaderITL = new InheritableThreadLocal<WsdlLoader>();
	
	public static WsdlLoader get() {
		return wsdlLoaderITL.get();
	}
	
	public static void set(WsdlLoader wsdlLoader) {
		wsdlLoaderITL.set(wsdlLoader);
	}
	
	public static void unset() {
		set(null);
	}
	
	public WsdlLoaderAspect() {}

	public Object getWsdlLoader(final MethodInvocation invocation) throws Throwable {
		WsdlLoader wsdlLoader = get();
		return (wsdlLoader != null) ? wsdlLoader : invocation.invokeNext();
	}

}
