/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.util;

import junit.framework.TestCase;

import java.util.zip.ZipOutputStream;
import java.util.zip.ZipEntry;
import java.util.Map;
import java.util.LinkedHashMap;
import java.util.Set;
import java.io.*;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.apache.log4j.Logger;

/**
 * Deployment archive creator.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class TestDeploymentArchive {

    private static Logger logger = Logger.getLogger(TestDeploymentArchive.class);
    private static final String PROP_ORG_JBOSS_ESB_DEPLOY_DIR = "jbosstest.deploy.dir";

    private File archiveFile;
    private LinkedHashMap<String, byte[]> entries = new LinkedHashMap<String, byte[]>();

    /**
     * Create the named archive in the folder denoted by the "jbosstest.deploy.dir"
     * System property.
     *
     * @param name Archive name.
     */
    public TestDeploymentArchive(String name) {
        this(getParentFolder(name), name);
    }

    /**
     * Create the named archive in the folder specified by the parentFolder arg.
     *
     * @param parentFolder Archive parent folder.
     * @param name Archive name.
     */
    public TestDeploymentArchive(File parentFolder, String name) {
        AssertArgument.isNotNull(parentFolder, "parentFolder");
        AssertArgument.isNotNullAndNotEmpty(name, "name");

        if(!parentFolder.exists()) {
            TestCase.fail("Cannot create archive '" + name + "'.  Parent folder '" + parentFolder.getAbsolutePath() + "' does not exist.");
        }
        archiveFile = new File(parentFolder, name);
    }

    private static File getParentFolder(String name) {
        String deployTargetFolder = System.getProperty(PROP_ORG_JBOSS_ESB_DEPLOY_DIR);
        if(deployTargetFolder == null) {
            TestCase.fail("Cannot create archive '" + name + "'.  Parent folder must be specified in System property '" + PROP_ORG_JBOSS_ESB_DEPLOY_DIR + "'.");
        }
        return new File(deployTargetFolder);
    }

    public void addEntry(String path, InputStream data) {
        AssertArgument.isNotNullAndNotEmpty(path, "path");
        AssertArgument.isNotNull(data, "data");
        
        try {
            entries.put(path, StreamUtils.readStream(data));
        } finally {
            try {
                data.close();
            } catch (IOException e) {
                TestCase.fail("Unable to close input stream for archive entry '" + path + "'.");
            }
        }
    }

    public void addEntry(String path, String resClasspathLocation) {
        InputStream resource = getClass().getResourceAsStream(resClasspathLocation);
        if(resource == null) {
            TestCase.fail("Unable to close input stream for archive entry '" + path + "'.");
        } else {
            addEntry(path, resource);
        }
    }

    public void addEntry(Class clazz) {
        String className = clazz.getName();

        className = className.replace('.', '/') + ".class";
        addEntry(className, "/" + className);
    }

    /**
     * Create an archive of the specified name and containing entries
     * for the data contained in the streams supplied entries arg. 
     * specifying the entry name and the value is a InputStream containing
     * the entry data. 
     */
    public void create() {
        ZipOutputStream archiveStream = null;
        try {
            archiveStream = new ZipOutputStream(new FileOutputStream(archiveFile));
        } catch (FileNotFoundException e) {
            TestCase.fail("Unable to open output stream to archive file '" + archiveFile.getAbsolutePath() + "'.");
        }

        try {
            writeEntriesToArchive(archiveStream);
            logger.info("Deployment archive '" + archiveFile.getAbsolutePath() + "' created.");
        } finally {
            try {
                archiveStream.close();
            } catch (IOException e) {
                TestCase.fail("Unable to close output stream to archive file '" + archiveFile.getAbsolutePath() + "'.");
            }
        }
    }

    /**
     * Remove the archive.
     */
    public void remove() {
        if(archiveFile.exists()) {
            archiveFile.delete();
        }
    }

    private void writeEntriesToArchive(ZipOutputStream archiveStream) {
        Set<Map.Entry<String, byte[]>> entrySet = entries.entrySet();
        for (Map.Entry<String, byte[]> entry : entrySet) {
            try {
                archiveStream.putNextEntry(new ZipEntry(entry.getKey()));
                archiveStream.write(entry.getValue());
                archiveStream.closeEntry();
            } catch (IOException e) {
                TestCase.fail("Unable to create archive entry for '" + archiveFile.getAbsolutePath() + "': " + e.getMessage());
            }
        }
    }
}
