package twitter4j;

import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;

/**
 * @author Cedric Meury - cedric at meury.com
 */
public class StatusJSONImplTest {

    @Test
    void testReturningNullForMissingIsoLanguageCode() throws Exception {
        // given
        String json = "{\"contributors\":null,\"text\":\"@belbeer After four years of study in radiotechnical college I realized that the language skills are much more useful than fucking math.\",\"geo\":null,\"retweeted\":false,\"in_reply_to_screen_name\":\"belbeer\",\"truncated\":false,\"entities\":{\"urls\":[],\"hashtags\":[],\"user_mentions\":[{\"id\":74987101,\"name\":\"belbeer\",\"indices\":[0,8],\"screen_name\":\"belbeer\",\"id_str\":\"74987101\"}]},\"in_reply_to_status_id_str\":\"297250346298904576\",\"id\":297265580690513920,\"source\":\"web\",\"in_reply_to_user_id_str\":\"74987101\",\"favorited\":false,\"in_reply_to_status_id\":297250346298904576,\"retweet_count\":0,\"created_at\":\"Fri Feb 01 08:50:12 +0000 2013\",\"in_reply_to_user_id\":74987101,\"id_str\":\"297265580690513920\",\"place\":null,\"user\":{\"location\":\"where am i?\",\"default_profile\":true,\"profile_background_tile\":false,\"statuses_count\":1100,\"lang\":\"ru\",\"profile_link_color\":\"0084B4\",\"id\":386522307,\"following\":null,\"protected\":false,\"favourites_count\":13,\"profile_text_color\":\"333333\",\"description\":\"160 ????????\",\"verified\":false,\"contributors_enabled\":false,\"profile_sidebar_border_color\":\"C0DEED\",\"name\":\"???????? ??????\",\"profile_background_color\":\"C0DEED\",\"created_at\":\"Fri Oct 07 12:44:01 +0000 2011\",\"default_profile_image\":false,\"followers_count\":46,\"profile_image_url_https\":\"https://si0.twimg.com/profile_images/2785804589/b0b355c75d3f77658fc94ec05a7bb5af_normal.jpeg\",\"geo_enabled\":true,\"profile_background_image_url\":\"http://a0.twimg.com/images/themes/theme1/bg.png\",\"profile_background_image_url_https\":\"https://si0.twimg.com/images/themes/theme1/bg.png\",\"follow_request_sent\":null,\"entities\":{\"description\":{\"urls\":[]},\"url\":{\"urls\":[{\"expanded_url\":null,\"indices\":[0,29],\"url\":\"http://goodfoto.blogspot.com/\"}]}},\"url\":\"http://goodfoto.blogspot.com/\",\"utc_offset\":7200,\"time_zone\":\"Athens\",\"notifications\":null,\"profile_use_background_image\":true,\"friends_count\":46,\"profile_sidebar_fill_color\":\"DDEEF6\",\"screen_name\":\"LonliLokli2000\",\"id_str\":\"386522307\",\"profile_image_url\":\"http://a0.twimg.com/profile_images/2785804589/b0b355c75d3f77658fc94ec05a7bb5af_normal.jpeg\",\"listed_count\":1,\"is_translator\":false},\"coordinates\":null}";
        Status status = TwitterObjectFactory.createStatus(json);

        // when
        String lang = status.getLang();

        // then
        assertNull(lang);
    }

    @Test
    void testIsRetweetedWhenStatusIsRetweeted() throws JSONException, TwitterException {
        String rawJson = "{ \"created_at\":\"Mon Mar 11 19:37:00 +0000 2013\", \"id\":311199093852618752, \"id_str\":\"311199093852618752\", \"text\":\"Introducing application-only authentication for the Twitter REST API v1.1 https:\\/\\/t.co\\/BrLLpVyuCe ^TS\", \"source\":\"web\", \"truncated\":false, \"in_reply_to_status_id\":null, \"in_reply_to_status_id_str\":null, \"in_reply_to_user_id\":null, \"in_reply_to_user_id_str\":null, \"in_reply_to_screen_name\":null, \"user\":{ \"id\":6253282, \"id_str\":\"6253282\", \"name\":\"Twitter API\", \"screen_name\":\"twitterapi\", \"location\":\"San Francisco, CA\", \"description\":\"The Real Twitter API. I tweet about API changes, service issues and happily answer questions about Twitter and our API. Don't get an answer? It's on my website.\", \"url\":\"http:\\/\\/dev.twitter.com\", \"entities\":{ \"url\":{ \"urls\":[ { \"url\":\"http:\\/\\/dev.twitter.com\", \"expanded_url\":null, \"indices\":[ 0, 22 ] } ] }, \"description\":{ \"urls\":[ ] } }, \"protected\":false, \"followers_count\":1533137, \"friends_count\":33, \"listed_count\":11369, \"created_at\":\"Wed May 23 06:01:13 +0000 2007\", \"favourites_count\":25, \"utc_offset\":-28800, \"time_zone\":\"Pacific Time (US & Canada)\", \"geo_enabled\":true, \"verified\":true, \"statuses_count\":3392, \"lang\":\"en\", \"contributors_enabled\":true, \"is_translator\":false, \"profile_background_color\":\"C0DEED\", \"profile_background_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_tile\":true, \"profile_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_banner_url\":\"https:\\/\\/si0.twimg.com\\/profile_banners\\/6253282\\/1347394302\", \"profile_link_color\":\"0084B4\", \"profile_sidebar_border_color\":\"C0DEED\", \"profile_sidebar_fill_color\":\"DDEEF6\", \"profile_text_color\":\"333333\", \"profile_use_background_image\":true, \"default_profile\":false, \"default_profile_image\":false, \"following\":null, \"follow_request_sent\":false, \"notifications\":null }, \"geo\":null, \"coordinates\":null, \"place\":null, \"contributors\":[ 819797 ], \"retweet_count\":131, \"entities\":{ \"hashtags\":[ ], \"urls\":[ { \"url\":\"https:\\/\\/t.co\\/BrLLpVyuCe\", \"expanded_url\":\"https:\\/\\/dev.twitter.com\\/docs\\/auth\\/application-only-auth\", \"display_url\":\"dev.twitter.com\\/docs\\/auth\\/appl…\", \"indices\":[ 74, 97 ] } ], \"user_mentions\":[ ] }, \"favorited\":false, \"retweeted\":true, \"possibly_sensitive\":false, \"lang\":\"en\" }";
        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        assertEquals(true, status.isRetweeted());
    }

    @Test
    void testIsRetweetedWhenStatusIsNotRetweeted() throws JSONException, TwitterException {
        String rawJson = "{ \"created_at\":\"Tue Mar 05 23:57:32 +0000 2013\", \"id\":309090333021581313, \"id_str\":\"309090333021581313\", \"text\":\"As announced, @anywhere has been retired per https:\\/\\/t.co\\/bWXjhurvwp The js file now logs a message to the console and exits quietly. ^ARK\", \"source\":\"web\", \"truncated\":false, \"in_reply_to_status_id\":null, \"in_reply_to_status_id_str\":null, \"in_reply_to_user_id\":null, \"in_reply_to_user_id_str\":null, \"in_reply_to_screen_name\":null, \"user\":{ \"id\":6253282, \"id_str\":\"6253282\", \"name\":\"Twitter API\", \"screen_name\":\"twitterapi\", \"location\":\"San Francisco, CA\", \"description\":\"The Real Twitter API. I tweet about API changes, service issues and happily answer questions about Twitter and our API. Don't get an answer? It's on my website.\", \"url\":\"http:\\/\\/dev.twitter.com\", \"entities\":{ \"url\":{ \"urls\":[ { \"url\":\"http:\\/\\/dev.twitter.com\", \"expanded_url\":null, \"indices\":[ 0, 22 ] } ] }, \"description\":{ \"urls\":[ ] } }, \"protected\":false, \"followers_count\":1533137, \"friends_count\":33, \"listed_count\":11369, \"created_at\":\"Wed May 23 06:01:13 +0000 2007\", \"favourites_count\":25, \"utc_offset\":-28800, \"time_zone\":\"Pacific Time (US & Canada)\", \"geo_enabled\":true, \"verified\":true, \"statuses_count\":3392, \"lang\":\"en\", \"contributors_enabled\":true, \"is_translator\":false, \"profile_background_color\":\"C0DEED\", \"profile_background_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_tile\":true, \"profile_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_banner_url\":\"https:\\/\\/si0.twimg.com\\/profile_banners\\/6253282\\/1347394302\", \"profile_link_color\":\"0084B4\", \"profile_sidebar_border_color\":\"C0DEED\", \"profile_sidebar_fill_color\":\"DDEEF6\", \"profile_text_color\":\"333333\", \"profile_use_background_image\":true, \"default_profile\":false, \"default_profile_image\":false, \"following\":null, \"follow_request_sent\":false, \"notifications\":null }, \"geo\":null, \"coordinates\":null, \"place\":null, \"contributors\":[ 7588892 ], \"retweet_count\":74, \"entities\":{ \"hashtags\":[ ], \"urls\":[ { \"url\":\"https:\\/\\/t.co\\/bWXjhurvwp\", \"expanded_url\":\"https:\\/\\/dev.twitter.com\\/blog\\/sunsetting-anywhere\", \"display_url\":\"dev.twitter.com\\/blog\\/sunsettin…\", \"indices\":[ 45, 68 ] } ], \"user_mentions\":[ { \"screen_name\":\"anywhere\", \"name\":\"Anywhere\", \"id\":9576402, \"id_str\":\"9576402\", \"indices\":[ 14, 23 ] } ] }, \"favorited\":false, \"retweeted\":false, \"possibly_sensitive\":false, \"lang\":\"en\" }";
        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        assertEquals(false, status.isRetweeted());
    }

    @Test
    void testIsRetweetedWhenStatusHasNoRetweetedProperty() throws JSONException, TwitterException {
        String rawJson = "{ \"created_at\":\"Tue Mar 05 23:57:32 +0000 2013\", \"id\":309090333021581313, \"id_str\":\"309090333021581313\", \"text\":\"As announced, @anywhere has been retired per https:\\/\\/t.co\\/bWXjhurvwp The js file now logs a message to the console and exits quietly. ^ARK\", \"source\":\"web\", \"truncated\":false, \"in_reply_to_status_id\":null, \"in_reply_to_status_id_str\":null, \"in_reply_to_user_id\":null, \"in_reply_to_user_id_str\":null, \"in_reply_to_screen_name\":null, \"user\":{ \"id\":6253282, \"id_str\":\"6253282\", \"name\":\"Twitter API\", \"screen_name\":\"twitterapi\", \"location\":\"San Francisco, CA\", \"description\":\"The Real Twitter API. I tweet about API changes, service issues and happily answer questions about Twitter and our API. Don't get an answer? It's on my website.\", \"url\":\"http:\\/\\/dev.twitter.com\", \"entities\":{ \"url\":{ \"urls\":[ { \"url\":\"http:\\/\\/dev.twitter.com\", \"expanded_url\":null, \"indices\":[ 0, 22 ] } ] }, \"description\":{ \"urls\":[ ] } }, \"protected\":false, \"followers_count\":1533137, \"friends_count\":33, \"listed_count\":11369, \"created_at\":\"Wed May 23 06:01:13 +0000 2007\", \"favourites_count\":25, \"utc_offset\":-28800, \"time_zone\":\"Pacific Time (US & Canada)\", \"geo_enabled\":true, \"verified\":true, \"statuses_count\":3392, \"lang\":\"en\", \"contributors_enabled\":true, \"is_translator\":false, \"profile_background_color\":\"C0DEED\", \"profile_background_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_tile\":true, \"profile_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_banner_url\":\"https:\\/\\/si0.twimg.com\\/profile_banners\\/6253282\\/1347394302\", \"profile_link_color\":\"0084B4\", \"profile_sidebar_border_color\":\"C0DEED\", \"profile_sidebar_fill_color\":\"DDEEF6\", \"profile_text_color\":\"333333\", \"profile_use_background_image\":true, \"default_profile\":false, \"default_profile_image\":false, \"following\":null, \"follow_request_sent\":false, \"notifications\":null }, \"geo\":null, \"coordinates\":null, \"place\":null, \"contributors\":[ 7588892 ], \"retweet_count\":74, \"entities\":{ \"hashtags\":[ ], \"urls\":[ { \"url\":\"https:\\/\\/t.co\\/bWXjhurvwp\", \"expanded_url\":\"https:\\/\\/dev.twitter.com\\/blog\\/sunsetting-anywhere\", \"display_url\":\"dev.twitter.com\\/blog\\/sunsettin…\", \"indices\":[ 45, 68 ] } ], \"user_mentions\":[ { \"screen_name\":\"anywhere\", \"name\":\"Anywhere\", \"id\":9576402, \"id_str\":\"9576402\", \"indices\":[ 14, 23 ] } ] }, \"favorited\":false, \"possibly_sensitive\":false, \"lang\":\"en\" }";
        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        assertEquals(false, status.isRetweeted());
    }

    @Test
    void testLoadingGeoLocationWithCoordinatesField() throws Exception {

        //given
        String json = "{\"filter_level\":\"medium\",\"contributors\":null,\"text\":\"@Livvy_Scott1 because I am a clever  boy.\",\"geo\":{\"type\":\"Point\",\"coordinates\":[52.25604116,0.70928444]},\"retweeted\":false,\"in_reply_to_screen_name\":\"Livvy_Scott1\",\"truncated\":false,\"lang\":\"en\",\"entities\":{\"urls\":[],\"hashtags\":[],\"user_mentions\":[{\"id\":476669159,\"name\":\"livs ?\",\"indices\":[0,13],\"screen_name\":\"Livvy_Scott1\",\"id_str\":\"476669159\"}]},\"in_reply_to_status_id_str\":\"320934680662794241\",\"id\":320936613498744832,\"source\":\"<a href=\\\"http://twitter.com/download/android\\\" rel=\\\"nofollow\\\">Twitter for Android<\\/a>\",\"in_reply_to_user_id_str\":\"476669159\",\"favorited\":false,\"in_reply_to_status_id\":320934680662794241,\"retweet_count\":0,\"created_at\":\"Sun Apr 07 16:30:26 +0000 2013\",\"in_reply_to_user_id\":476669159,\"favorite_count\":0,\"id_str\":\"320936613498744832\",\"place\":{\"id\":\"55c6bcd3013a0607\",\"bounding_box\":{\"type\":\"Polygon\",\"coordinates\":[[[0.38178,52.055592],[0.38178,52.400796],[0.967452,52.400796],[0.967452,52.055592]]]},\"place_type\":\"city\",\"contained_within\":[],\"name\":\"St. Edmundsbury\",\"attributes\":{},\"country_code\":\"GB\",\"url\":\"http://api.twitter.com/1/geo/id/55c6bcd3013a0607.json\",\"polylines\":[],\"geometry\":null,\"country\":\"United Kingdom\",\"full_name\":\"St. Edmundsbury, Suffolk\"},\"user\":{\"location\":\"Attleborough\",\"default_profile\":false,\"profile_background_tile\":true,\"statuses_count\":2520,\"lang\":\"en\",\"profile_link_color\":\"009999\",\"profile_banner_url\":\"https://si0.twimg.com/profile_banners/448404395/1365018601\",\"id\":448404395,\"following\":null,\"protected\":false,\"favourites_count\":203,\"profile_text_color\":\"333333\",\"description\":\"Born at a very young age. Top Jock - DJ Bookings-maxwellDJ18@hotmail.co.uk \",\"verified\":false,\"contributors_enabled\":false,\"profile_sidebar_border_color\":\"EEEEEE\",\"name\":\"DJ MaxwellJ \",\"profile_background_color\":\"131516\",\"created_at\":\"Tue Dec 27 23:49:48 +0000 2011\",\"default_profile_image\":false,\"followers_count\":309,\"profile_image_url_https\":\"https://si0.twimg.com/profile_images/3472432483/07133836faedec0252f17d691cb7eb5d_normal.jpeg\",\"geo_enabled\":true,\"profile_background_image_url\":\"http://a0.twimg.com/images/themes/theme14/bg.gif\",\"profile_background_image_url_https\":\"https://si0.twimg.com/images/themes/theme14/bg.gif\",\"follow_request_sent\":null,\"url\":null,\"utc_offset\":0,\"time_zone\":\"Casablanca\",\"notifications\":null,\"profile_use_background_image\":true,\"friends_count\":342,\"profile_sidebar_fill_color\":\"EFEFEF\",\"screen_name\":\"maxwellDJ18\",\"id_str\":\"448404395\",\"profile_image_url\":\"http://a0.twimg.com/profile_images/3472432483/07133836faedec0252f17d691cb7eb5d_normal.jpeg\",\"listed_count\":0,\"is_translator\":false},\"coordinates\":{\"type\":\"Point\",\"coordinates\":[0.70928444,52.25604116]}}\n";

        //when
        Status status = TwitterObjectFactory.createStatus(json);
        GeoLocation geoLocation = status.getGeoLocation();
        //then
        assertNotNull(geoLocation);
        assertEquals(geoLocation.getLongitude(), 0.70928444, 0.00000001);
        assertEquals(geoLocation.getLatitude(), 52.25604116, 0.00000001);
    }

    @Test
    void testLangWithLangFieldDefined() throws Exception {
        //given
        String rawJson = "{ \"created_at\":\"Mon Mar 11 19:37:00 +0000 2013\", \"id\":311199093852618752, \"id_str\":\"311199093852618752\", \"text\":\"Introducing application-only authentication for the Twitter REST API v1.1 https:\\/\\/t.co\\/BrLLpVyuCe ^TS\", \"source\":\"web\", \"truncated\":false, \"in_reply_to_status_id\":null, \"in_reply_to_status_id_str\":null, \"in_reply_to_user_id\":null, \"in_reply_to_user_id_str\":null, \"in_reply_to_screen_name\":null, \"user\":{ \"id\":6253282, \"id_str\":\"6253282\", \"name\":\"Twitter API\", \"screen_name\":\"twitterapi\", \"location\":\"San Francisco, CA\", \"description\":\"The Real Twitter API. I tweet about API changes, service issues and happily answer questions about Twitter and our API. Don't get an answer? It's on my website.\", \"url\":\"http:\\/\\/dev.twitter.com\", \"entities\":{ \"url\":{ \"urls\":[ { \"url\":\"http:\\/\\/dev.twitter.com\", \"expanded_url\":null, \"indices\":[ 0, 22 ] } ] }, \"description\":{ \"urls\":[ ] } }, \"protected\":false, \"followers_count\":1533137, \"friends_count\":33, \"listed_count\":11369, \"created_at\":\"Wed May 23 06:01:13 +0000 2007\", \"favourites_count\":25, \"utc_offset\":-28800, \"time_zone\":\"Pacific Time (US & Canada)\", \"geo_enabled\":true, \"verified\":true, \"statuses_count\":3392, \"lang\":\"en\", \"contributors_enabled\":true, \"is_translator\":false, \"profile_background_color\":\"C0DEED\", \"profile_background_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_background_images\\/656927849\\/miyt9dpjz77sc0w3d4vj.png\", \"profile_background_tile\":true, \"profile_image_url\":\"http:\\/\\/a0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_image_url_https\":\"https:\\/\\/si0.twimg.com\\/profile_images\\/2284174872\\/7df3h38zabcvjylnyfe3_normal.png\", \"profile_banner_url\":\"https:\\/\\/si0.twimg.com\\/profile_banners\\/6253282\\/1347394302\", \"profile_link_color\":\"0084B4\", \"profile_sidebar_border_color\":\"C0DEED\", \"profile_sidebar_fill_color\":\"DDEEF6\", \"profile_text_color\":\"333333\", \"profile_use_background_image\":true, \"default_profile\":false, \"default_profile_image\":false, \"following\":null, \"follow_request_sent\":false, \"notifications\":null }, \"geo\":null, \"coordinates\":null, \"place\":null, \"contributors\":[ 819797 ], \"retweet_count\":131, \"entities\":{ \"hashtags\":[ ], \"urls\":[ { \"url\":\"https:\\/\\/t.co\\/BrLLpVyuCe\", \"expanded_url\":\"https:\\/\\/dev.twitter.com\\/docs\\/auth\\/application-only-auth\", \"display_url\":\"dev.twitter.com\\/docs\\/auth\\/appl…\", \"indices\":[ 74, 97 ] } ], \"user_mentions\":[ ] }, \"favorited\":false, \"retweeted\":true, \"possibly_sensitive\":false, \"lang\":\"en\" }";

        //when
        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        //then
        assertNotNull(status.getLang());
        assertEquals("en", status.getLang());
    }

    @Test
    void testExtendedMediaEntities() throws Exception {
        //given from https://dev.twitter.com/docs/api/multiple-media-extended-entities
        String rawJson = "{  \"created_at\": \"Fri Mar 28 21:34:45 +0000 2014\",  \"id\": 449660889793581060,  \"id_str\": \"449660889793581056\",  \"text\": \"\\\"I hope you'll keep...building bonds of friendship that will enrich your lives & enrich our world\\\" —FLOTUS in China, http://t.co/fxmuQN9JL9\",  \"source\": \"<a href=\\\"http://twitter.com/download/iphone\\\" rel=\\\"nofollow\\\">Twitter for iPhone</a>\",  \"truncated\": false,  \"in_reply_to_status_id\": null,  \"in_reply_to_status_id_str\": null,  \"in_reply_to_user_id\": null,  \"in_reply_to_user_id_str\": null,  \"in_reply_to_screen_name\": null,  \"user\":  {    \"id\": 1093090866,    \"id_str\": \"1093090866\",    \"name\": \"The First Lady\",    \"screen_name\": \"FLOTUS\",    \"location\": \"Washington, DC\",    \"description\": \"This account is run by the Office of First Lady Michelle Obama. Tweets from the First Lady are signed –mo. Tweets may be archived. More at http://t.co/9DxP65hB\",    \"url\": \"http://t.co/RvRZspIO8c\",    \"entities\":  {      \"url\":  {        \"urls\":  [           {            \"url\": \"http://t.co/RvRZspIO8c\",            \"expanded_url\": \"http://www.whitehouse.gov\",            \"display_url\": \"whitehouse.gov\",            \"indices\":  [              0,              22            ]          }        ]      },      \"description\":  {        \"urls\":  [           {            \"url\": \"http://t.co/9DxP65hB\",            \"expanded_url\": \"http://wh.gov/privacy\",            \"display_url\": \"wh.gov/privacy\",            \"indices\":  [              139,              159            ]          }        ]      }    },    \"protected\": false,    \"followers_count\": 878857,    \"friends_count\": 21,    \"listed_count\": 5349,    \"created_at\": \"Tue Jan 15 20:03:17 +0000 2013\",    \"favourites_count\": 0,    \"utc_offset\": -14400,    \"time_zone\": \"Eastern Time (US & Canada)\",    \"geo_enabled\": false,    \"verified\": true,    \"statuses_count\": 813,    \"lang\": \"en\",    \"contributors_enabled\": false,    \"is_translator\": false,    \"is_translation_enabled\": false,    \"profile_background_color\": \"0084B4\",    \"profile_background_image_url\": \"http://pbs.twimg.com/profile_background_images/766168824/6d7e9bac6f5904ab0843bd1e9d935695.jpeg\",    \"profile_background_image_url_https\": \"https://pbs.twimg.com/profile_background_images/766168824/6d7e9bac6f5904ab0843bd1e9d935695.jpeg\",    \"profile_background_tile\": false,    \"profile_image_url\": \"http://pbs.twimg.com/profile_images/453354757499781120/WsSISQK__normal.jpeg\",    \"profile_image_url_https\": \"https://pbs.twimg.com/profile_images/453354757499781120/WsSISQK__normal.jpeg\",    \"profile_banner_url\": \"https://pbs.twimg.com/profile_banners/1093090866/1396923217\",    \"profile_link_color\": \"0084B4\",    \"profile_sidebar_border_color\": \"FFFFFF\",    \"profile_sidebar_fill_color\": \"DDEEF6\",    \"profile_text_color\": \"333333\",    \"profile_use_background_image\": true,    \"default_profile\": false,    \"default_profile_image\": false,    \"following\": false,    \"follow_request_sent\": false,    \"notifications\": false  },  \"geo\": null,  \"coordinates\": null,  \"place\": null,  \"contributors\": null,  \"retweet_count\": 222,  \"favorite_count\": 295,  \"entities\": {      \"hashtags\": [],      \"media\": [          {              \"display_url\": \"pic.twitter.com/fxmuQN9JL9\",              \"expanded_url\": \"http://twitter.com/FLOTUS/status/449660889793581056/photo/1\",              \"id\": 449660809380380673,              \"id_str\": \"449660809380380673\",              \"indices\": [                  121,                  143              ],              \"media_url\": \"http://pbs.twimg.com/media/Bj2EH6yIQAEYvxu.jpg\",              \"media_url_https\": \"https://pbs.twimg.com/media/Bj2EH6yIQAEYvxu.jpg\",              \"sizes\": {                  \"large\": {                      \"h\": 426,                      \"resize\": \"fit\",                      \"w\": 640                  },                  \"medium\": {                      \"h\": 399,                      \"resize\": \"fit\",                      \"w\": 600                  },                  \"small\": {                      \"h\": 226,                      \"resize\": \"fit\",                      \"w\": 340                  },                  \"thumb\": {                      \"h\": 150,                      \"resize\": \"crop\",                      \"w\": 150                  }              },              \"type\": \"photo\",              \"url\": \"http://t.co/fxmuQN9JL9\"          }      ],      \"symbols\": [],      \"urls\": [],      \"user_mentions\": []  },  \"extended_entities\": {      \"media\": [          {              \"display_url\": \"pic.twitter.com/fxmuQN9JL9\",              \"expanded_url\": \"http://twitter.com/FLOTUS/status/449660889793581056/photo/1\",              \"id\": 449660809380380673,              \"id_str\": \"449660809380380673\",              \"indices\": [                  121,                  143              ],              \"media_url\": \"http://pbs.twimg.com/media/Bj2EH6yIQAEYvxu.jpg\",              \"media_url_https\": \"https://pbs.twimg.com/media/Bj2EH6yIQAEYvxu.jpg\",              \"sizes\": {                  \"large\": {                      \"h\": 426,                      \"resize\": \"fit\",                      \"w\": 640                  },                  \"medium\": {                      \"h\": 399,                      \"resize\": \"fit\",                      \"w\": 600                  },                  \"small\": {                      \"h\": 226,                      \"resize\": \"fit\",                      \"w\": 340                  },                  \"thumb\": {                      \"h\": 150,                      \"resize\": \"crop\",                      \"w\": 150                  }              },              \"type\": \"photo\",              \"url\": \"http://t.co/fxmuQN9JL9\"          },          {              \"display_url\": \"pic.twitter.com/fxmuQN9JL9\",              \"expanded_url\": \"http://twitter.com/FLOTUS/status/449660889793581056/photo/1\",              \"id\": 449660806754738177,              \"id_str\": \"449660806754738177\",              \"indices\": [                  121,                  143              ],              \"media_url\": \"http://pbs.twimg.com/media/Bj2EHxAIIAE8dtg.jpg\",              \"media_url_https\": \"https://pbs.twimg.com/media/Bj2EHxAIIAE8dtg.jpg\",              \"sizes\": {                  \"large\": {                      \"h\": 426,                      \"resize\": \"fit\",                      \"w\": 640                  },                  \"medium\": {                      \"h\": 399,                      \"resize\": \"fit\",                      \"w\": 600                  },                  \"small\": {                      \"h\": 226,                      \"resize\": \"fit\",                      \"w\": 340                  },                  \"thumb\": {                      \"h\": 150,                      \"resize\": \"crop\",                      \"w\": 150                  }              },              \"type\": \"photo\",              \"url\": \"http://t.co/fxmuQN9JL9\"          },          {              \"display_url\": \"pic.twitter.com/fxmuQN9JL9\",              \"expanded_url\": \"http://twitter.com/FLOTUS/status/449660889793581056/photo/1\",              \"id\": 449660808537333761,              \"id_str\": \"449660808537333761\",              \"indices\": [                  121,                  143              ],              \"media_url\": \"http://pbs.twimg.com/media/Bj2EH3pIYAE4LQn.jpg\",              \"media_url_https\": \"https://pbs.twimg.com/media/Bj2EH3pIYAE4LQn.jpg\",              \"sizes\": {                  \"large\": {                      \"h\": 427,                      \"resize\": \"fit\",                      \"w\": 640                  },                  \"medium\": {                      \"h\": 400,                      \"resize\": \"fit\",                      \"w\": 600                  },                  \"small\": {                      \"h\": 227,                      \"resize\": \"fit\",                      \"w\": 340                  },                  \"thumb\": {                      \"h\": 150,                      \"resize\": \"crop\",                      \"w\": 150                  }              },              \"type\": \"photo\",              \"url\": \"http://t.co/fxmuQN9JL9\"          },          {              \"display_url\": \"pic.twitter.com/fxmuQN9JL9\",              \"expanded_url\": \"http://twitter.com/FLOTUS/status/449660889793581056/photo/1\",              \"id\": 449660877097406464,              \"id_str\": \"449660877097406464\",              \"indices\": [                  121,                  143              ],              \"media_url\": \"http://pbs.twimg.com/media/Bj2EL3DIEAAzGAX.jpg\",              \"media_url_https\": \"https://pbs.twimg.com/media/Bj2EL3DIEAAzGAX.jpg\",              \"sizes\": {                  \"large\": {                      \"h\": 426,                      \"resize\": \"fit\",                      \"w\": 640                  },                  \"medium\": {                      \"h\": 399,                      \"resize\": \"fit\",                      \"w\": 600                  },                  \"small\": {                      \"h\": 226,                      \"resize\": \"fit\",                      \"w\": 340                  },                  \"thumb\": {                      \"h\": 150,                      \"resize\": \"crop\",                      \"w\": 150                  }              },              \"type\": \"photo\",              \"url\": \"http://t.co/fxmuQN9JL9\"          }      ]  },  \"favorited\": false,  \"retweeted\": false,  \"possibly_sensitive\": false,  \"lang\": \"en\"}";

        //when
        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        //then
        assertEquals(4, status.getMediaEntities().length);
    }

    @Test
    void testQuotedStatus() throws Exception {

        //given from https://twittercommunity.com/t/api-payloads-to-include-original-quoted-tweet-objects/38184
        String rawJson = "{\"created_at\":\"Thu May 21 20:03:48 +0000 2015\",\"id\":601478517394636801,\"id_str\":\"601478517394636801\",\"text\":\"Awesome news! https://t.co/plSZZhcvkP\",\"source\":\"<a href=\\\"http://twitter.com\\\" rel=\\\"nofollow\\\">Twitter Web Client</a>\",\"truncated\":false,\"in_reply_to_status_id\":null,\"in_reply_to_status_id_str\":null,\"in_reply_to_user_id\":null,\"in_reply_to_user_id_str\":null,\"in_reply_to_screen_name\":null,\"user\":{\"id\":3001969357,\"id_str\":\"3001969357\",\"name\":\"Jordan Brinks\",\"screen_name\":\"ilyanshe\",\"location\":\"Madagascar\",\"url\":\"http://spacejam.com\",\"description\":\"Alter Ego\",\"protected\":false,\"verified\":false,\"followers_count\":8,\"friends_count\":37,\"listed_count\":0,\"favourites_count\":1,\"statuses_count\":118,\"created_at\":\"Thu Jan 29 18:27:49 +0000 2015\",\"utc_offset\":3600,\"time_zone\":\"West Central Africa\",\"geo_enabled\":true,\"lang\":\"en\",\"contributors_enabled\":false,\"is_translator\":false,\"profile_background_color\":\"C0DEED\",\"profile_background_image_url\":\"http://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_image_url_https\":\"https://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_tile\":false,\"profile_link_color\":\"0084B4\",\"profile_sidebar_border_color\":\"C0DEED\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_text_color\":\"333333\",\"profile_use_background_image\":true,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/601155672395227136/qakfE9EU_normal.jpg\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/601155672395227136/qakfE9EU_normal.jpg\",\"profile_banner_url\":\"https://pbs.twimg.com/profile_banners/3001969357/1432161817\",\"default_profile\":true,\"default_profile_image\":false,\"following\":null,\"follow_request_sent\":null,\"notifications\":null},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"quoted_status_id\":560101531794821120,\"quoted_status_id_str\":\"560101531794821120\",\"quoted_status\":{\"created_at\":\"Tue Jan 27 15:46:26 +0000 2015\",\"id\":560101531794821120,\"id_str\":\"560101531794821120\",\"text\":\"Native video support in REST and Streaming API https://t.co/VrFm1qGZnF\",\"source\":\"<a href=\\\"http://itunes.apple.com/us/app/twitter/id409789998?mt=12\\\" rel=\\\"nofollow\\\">Twitter for Mac</a>\",\"truncated\":false,\"in_reply_to_status_id\":null,\"in_reply_to_status_id_str\":null,\"in_reply_to_user_id\":null,\"in_reply_to_user_id_str\":null,\"in_reply_to_screen_name\":null,\"user\":{\"id\":6253282,\"id_str\":\"6253282\",\"name\":\"Twitter API\",\"screen_name\":\"twitterapi\",\"location\":\"San Francisco, CA\",\"url\":\"http://dev.twitter.com\",\"description\":\"The Real Twitter API. I tweet about API changes, service issues and happily answer questions about Twitter and our API. Don't get an answer? It's on my website.\",\"protected\":false,\"verified\":true,\"followers_count\":3331390,\"friends_count\":48,\"listed_count\":12956,\"favourites_count\":27,\"statuses_count\":3534,\"created_at\":\"Wed May 23 06:01:13 +0000 2007\",\"utc_offset\":-25200,\"time_zone\":\"Pacific Time (US & Canada)\",\"geo_enabled\":true,\"lang\":\"en\",\"contributors_enabled\":false,\"is_translator\":false,\"profile_background_color\":\"C0DEED\",\"profile_background_image_url\":\"http://pbs.twimg.com/profile_background_images/656927849/miyt9dpjz77sc0w3d4vj.png\",\"profile_background_image_url_https\":\"https://pbs.twimg.com/profile_background_images/656927849/miyt9dpjz77sc0w3d4vj.png\",\"profile_background_tile\":true,\"profile_link_color\":\"0084B4\",\"profile_sidebar_border_color\":\"C0DEED\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_text_color\":\"333333\",\"profile_use_background_image\":true,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/2284174872/7df3h38zabcvjylnyfe3_normal.png\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/2284174872/7df3h38zabcvjylnyfe3_normal.png\",\"profile_banner_url\":\"https://pbs.twimg.com/profile_banners/6253282/1431474710\",\"default_profile\":false,\"default_profile_image\":false,\"following\":null,\"follow_request_sent\":null,\"notifications\":null,\"place\":{\"id\":\"96683cc9126741d1\"}},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"retweet_count\":0,\"favorite_count\":0,\"entities\":{\"hashtags\":[],\"trends\":[],\"urls\":[{\"url\":\"https://t.co/VrFm1qGZnF\",\"expanded_url\":\"https://twittercommunity.com/t/twitter-video-now-supported-in-rest-and-streaming-api/31258\",\"display_url\":\"twittercommunity.com/t/twitter-vide…\",\"indices\":[47,70]}],\"user_mentions\":[],\"symbols\":[]},\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"filter_level\":\"low\",\"lang\":\"en\"},\"retweet_count\":0,\"favorite_count\":0,\"entities\":{\"hashtags\":[],\"trends\":[],\"urls\":[{\"url\":\"https://t.co/plSZZhcvkP\",\"expanded_url\":\"https://twitter.com/twitterapi/status/560101531794821120\",\"display_url\":\"twitter.com/twitterapi/sta…\",\"indices\":[14,37]}],\"user_mentions\":[],\"symbols\":[]},\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"filter_level\":\"low\",\"lang\":\"en\",\"timestamp_ms\":\"1432238628134\"}";

        JSONObject json = new JSONObject(rawJson);
        Status status = new StatusJSONImpl(json);

        assertEquals(560101531794821120L, status.getQuotedStatusId());

        Status quotedStatus = status.getQuotedStatus();
        assertNotNull(quotedStatus);
        assertEquals(560101531794821120L, quotedStatus.getId());

        assertNull(status.getQuotedStatusPermalink());
    }

    public void testQuotedStatusPermalink() throws Exception {

        String rawJson = "{\"quoted_status\":{\"in_reply_to_status_id_str\":null,\"in_reply_to_status_id\":null,\"coordinates\":null,\"created_at\":\"Tue May 22 06:25:58 +0000 2018\",\"truncated\":false,\"in_reply_to_user_id_str\":null,\"source\":\"<a href=\\\"https://sites.google.com/site/tweentwitterclient/\\\" rel=\\\"nofollow\\\">Tween</a>\",\"retweet_count\":0,\"retweeted\":false,\"geo\":null,\"in_reply_to_screen_name\":null,\"is_quote_status\":false,\"entities\":{\"urls\":[],\"hashtags\":[],\"user_mentions\":[],\"symbols\":[]},\"text\":\"quoted_status_permalink が配信されるようになったので Twitter4J の対応しよう\",\"id_str\":\"998812193834418177\",\"in_reply_to_user_id\":null,\"display_text_range\":[0,55],\"favorite_count\":2,\"id\":998812193834418200,\"place\":null,\"contributors\":null,\"lang\":\"ja\",\"user\":{\"utc_offset\":32400,\"friends_count\":991,\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/423153841505193984/yGKSJu78_normal.jpeg\",\"listed_count\":134,\"profile_background_image_url\":\"http://abs.twimg.com/images/themes/theme1/bg.png\",\"default_profile_image\":false,\"favourites_count\":6826,\"description\":\"TwitterクライアントTwitPane、FacebookブラウザTafView、mixiブラウザTkMixiViewer、英単語学習ソフト P-Study System 、MZ3/4 などを開発。「ちょっぴり使いやすい」アプリを日々開発しています。ペーンクラフト代表\",\"created_at\":\"Thu Aug 23 10:06:53 +0000 2007\",\"is_translator\":false,\"profile_background_image_url_https\":\"https://abs.twimg.com/images/themes/theme1/bg.png\",\"protected\":false,\"screen_name\":\"takke\",\"id_str\":\"8379212\",\"profile_link_color\":\"1B95E0\",\"is_translation_enabled\":false,\"translator_type\":\"none\",\"id\":8379212,\"geo_enabled\":true,\"profile_background_color\":\"000000\",\"lang\":\"ja\",\"has_extended_profile\":false,\"profile_image_extensions_alt_text\":null,\"profile_sidebar_border_color\":\"000000\",\"profile_text_color\":\"000000\",\"verified\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/423153841505193984/yGKSJu78_normal.jpeg\",\"time_zone\":\"Tokyo\",\"url\":\"https://t.co/8al23JWsCZ\",\"contributors_enabled\":false,\"profile_background_tile\":false,\"profile_banner_url\":\"https://pbs.twimg.com/profile_banners/8379212/1390720101\",\"entities\":{\"description\":{\"urls\":[]},\"url\":{\"urls\":[{\"display_url\":\"panecraft.net\",\"indices\":[0,23],\"expanded_url\":\"http://www.panecraft.net\",\"url\":\"https://t.co/8al23JWsCZ\"}]}},\"statuses_count\":48447,\"follow_request_sent\":false,\"followers_count\":1639,\"profile_use_background_image\":false,\"default_profile\":false,\"following\":false,\"name\":\"竹内裕昭\",\"location\":\"北海道\",\"profile_sidebar_fill_color\":\"000000\",\"profile_banner_extensions_alt_text\":null,\"notifications\":false},\"favorited\":false},\"in_reply_to_status_id_str\":null,\"in_reply_to_status_id\":null,\"created_at\":\"Tue May 22 06:27:55 +0000 2018\",\"in_reply_to_user_id_str\":null,\"source\":\"<a href=\\\"https://sites.google.com/site/tweentwitterclient/\\\" rel=\\\"nofollow\\\">Tween</a>\",\"quoted_status_id\":998812193834418200,\"retweet_count\":0,\"retweeted\":false,\"geo\":null,\"in_reply_to_screen_name\":null,\"is_quote_status\":true,\"text\":\"てすてす https://t.co/8FWawCNvz9\",\"id_str\":\"998812681736830976\",\"in_reply_to_user_id\":null,\"favorite_count\":0,\"id\":998812681736831000,\"place\":null,\"quoted_status_permalink\":{\"expanded\":\"https://twitter.com/takke/status/998812193834418177\",\"display\":\"twitter.com/takke/status/9…\",\"url\":\"https://t.co/8FWawCNvz9\"},\"lang\":\"ja\",\"favorited\":false,\"possibly_sensitive\":false,\"coordinates\":null,\"truncated\":false,\"entities\":{\"urls\":[{\"display_url\":\"twitter.com/takke/status/9…\",\"indices\":[5,28],\"expanded_url\":\"https://twitter.com/takke/status/998812193834418177\",\"url\":\"https://t.co/8FWawCNvz9\"}],\"hashtags\":[],\"user_mentions\":[],\"symbols\":[]},\"display_text_range\":[0,4],\"quoted_status_id_str\":\"998812193834418177\",\"contributors\":null,\"user\":{\"utc_offset\":32400,\"friends_count\":3213,\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/3785115381/319a239dfbdbe3bec7afdacbfada91c8_normal.jpeg\",\"listed_count\":369,\"profile_background_image_url\":\"http://abs.twimg.com/images/themes/theme1/bg.png\",\"default_profile_image\":false,\"favourites_count\":1380,\"description\":\"Linux+育児+http://t.co/KdHyQJu0。埼玉に生息する３児のパパ。Win用TwitterクライアントTween公開してます。https://t.co/JVgcF73g\",\"created_at\":\"Fri Jun 15 08:06:06 +0000 2007\",\"is_translator\":false,\"profile_background_image_url_https\":\"https://abs.twimg.com/images/themes/theme1/bg.png\",\"protected\":false,\"screen_name\":\"kiri_feather\",\"id_str\":\"6828542\",\"profile_link_color\":\"0084B4\",\"is_translation_enabled\":false,\"translator_type\":\"regular\",\"id\":6828542,\"geo_enabled\":true,\"profile_background_color\":\"C0DEED\",\"lang\":\"ja\",\"has_extended_profile\":false,\"profile_image_extensions_alt_text\":null,\"profile_sidebar_border_color\":\"FFFFFF\",\"profile_text_color\":\"333333\",\"verified\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/3785115381/319a239dfbdbe3bec7afdacbfada91c8_normal.jpeg\",\"time_zone\":\"Tokyo\",\"url\":\"https://t.co/lKyjpmASMN\",\"contributors_enabled\":false,\"profile_background_tile\":false,\"profile_banner_url\":\"https://pbs.twimg.com/profile_banners/6828542/1348013722\",\"entities\":{\"description\":{\"urls\":[{\"display_url\":\"VB.net\",\"indices\":[9,29],\"expanded_url\":\"http://VB.net\",\"url\":\"http://t.co/KdHyQJu0\"},{\"display_url\":\"sites.google.com/site/tweentwit…\",\"indices\":[72,93],\"expanded_url\":\"https://sites.google.com/site/tweentwitterclient/\",\"url\":\"https://t.co/JVgcF73g\"}]},\"url\":{\"urls\":[{\"display_url\":\"d.hatena.ne.jp/Kiri_Feather/\",\"indices\":[0,23],\"expanded_url\":\"http://d.hatena.ne.jp/Kiri_Feather/\",\"url\":\"https://t.co/lKyjpmASMN\"}]}},\"statuses_count\":74523,\"follow_request_sent\":false,\"followers_count\":3991,\"profile_use_background_image\":true,\"default_profile\":false,\"following\":true,\"name\":\"きり(ΦωΦ)\",\"location\":\"Saitama, Japan\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_banner_extensions_alt_text\":null,\"notifications\":false}}";

        JSONObject json = new JSONObject(rawJson);
        Status status = new StatusJSONImpl(json);

        assertEquals(998812193834418200L, status.getQuotedStatusId());

        URLEntity quotedStatusPermalink = status.getQuotedStatusPermalink();
        assertEquals("https://twitter.com/takke/status/998812193834418177", quotedStatusPermalink.getExpandedURL());
        assertEquals("twitter.com/takke/status/9…", quotedStatusPermalink.getDisplayURL());
        assertEquals("https://t.co/8FWawCNvz9", quotedStatusPermalink.getURL());
    }

    @Test
    void testAcitivityWithheldInCountries() throws Exception {
        String rawJson = "{\"created_at\":\"Tue Nov 12 14:15:41 +0000 2013\",\"id\":400265648674467840,\"id_str\":\"400265648674467840\",\"text\":\"@ADecherf On t\\u2019a siffl\\u00e9 ?? Non, alors va te faire enculer au lieu de casser les couilles aux gens ! Adieu gros #PayDay !!\\n@fallenangels667\",\"source\":\"\\u003ca href=\\\"http:\\/\\/tapbots.com\\/tweetbot\\\" rel=\\\"nofollow\\\"\\u003eTweetbot for iOS\\u003c\\/a\\u003e\",\"truncated\":false,\"in_reply_to_status_id\":400250694789103616,\"in_reply_to_status_id_str\":\"400250694789103616\",\"in_reply_to_user_id\":1339289664,\"in_reply_to_user_id_str\":\"1339289664\",\"in_reply_to_screen_name\":\"ADecherf\",\"user\":{\"id\":587870719,\"id_str\":\"587870719\",\"name\":\"il Padrino\",\"screen_name\":\"DulezC\",\"location\":\"France (34) - Sicilia (CT)\",\"description\":\"Contre la bien-pensance. En guerre contre les collabos et les socialauds qui ruinent notre pays et le d\\u00e9g\\u00e9n\\u00e8rent ! Honneur et Patrie. \\u270bMEMENTO AUDERE SEMPER\\u270b\",\"url\":null,\"entities\":{\"description\":{\"urls\":[]}},\"protected\":false,\"followers_count\":2909,\"friends_count\":2525,\"listed_count\":16,\"created_at\":\"Wed May 23 00:25:40 +0000 2012\",\"favourites_count\":244,\"utc_offset\":7200,\"time_zone\":\"Paris\",\"geo_enabled\":false,\"verified\":false,\"statuses_count\":22124,\"lang\":\"fr\",\"contributors_enabled\":false,\"is_translator\":false,\"is_translation_enabled\":false,\"profile_background_color\":\"C0DEED\",\"profile_background_image_url\":\"http:\\/\\/abs.twimg.com\\/images\\/themes\\/theme1\\/bg.png\",\"profile_background_image_url_https\":\"https:\\/\\/abs.twimg.com\\/images\\/themes\\/theme1\\/bg.png\",\"profile_background_tile\":false,\"profile_image_url\":\"http:\\/\\/pbs.twimg.com\\/profile_images\\/3069773085\\/f483e45f226bea36fff5b67f300a52f9_normal.jpeg\",\"profile_image_url_https\":\"https:\\/\\/pbs.twimg.com\\/profile_images\\/3069773085\\/f483e45f226bea36fff5b67f300a52f9_normal.jpeg\",\"profile_banner_url\":\"https:\\/\\/pbs.twimg.com\\/profile_banners\\/587870719\\/1406034390\",\"profile_link_color\":\"0084B4\",\"profile_sidebar_border_color\":\"C0DEED\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_text_color\":\"333333\",\"profile_use_background_image\":true,\"default_profile\":true,\"default_profile_image\":false,\"following\":false,\"follow_request_sent\":false,\"notifications\":false},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"withheld_in_countries\":[\"FR\"],\"retweet_count\":0,\"favorite_count\":0,\"entities\":{\"hashtags\":[{\"text\":\"PayDay\",\"indices\":[111,118]}],\"symbols\":[],\"urls\":[],\"user_mentions\":[{\"screen_name\":\"ADecherf\",\"name\":\"(*\\u00a5*)\",\"id\":1339289664,\"id_str\":\"1339289664\",\"indices\":[0,9]},{\"screen_name\":\"FaLLenAngeLs667\",\"name\":\"Fallen Angel\",\"id\":1219663764,\"id_str\":\"1219663764\",\"indices\":[122,138]}]},\"favorited\":false,\"retweeted\":false,\"lang\":\"fr\"}";

        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        String[] withheldInCountries = status.getWithheldInCountries();
        assertEquals(1, withheldInCountries.length);
        assertEquals("FR", withheldInCountries[0]);
    }

    @Test
    void testExtendedTweetCompatibilityMode() throws Exception {

        // http://qiita.com/takke/items/c80a687718fc65b0b5b9#extended-tweet-compatibility-mode
        String rawJson = "{\"created_at\":\"Mon Mar 28 14:39:13 +0000 2016\",\"id\":714461850188926976,\"id_str\":\"714461850188926976\",\"text\":\"@jeremycloud It's neat to have owls and raccoons around until you realize that raccoons will eat the eggs from the … https://t.co/OY7qmdJQnO\",\"entities\":{\"hashtags\":[],\"symbols\":[],\"user_mentions\":[{\"screen_name\":\"jeremycloud\",\"name\":\"/dev/cloud/jeremy\",\"id\":15062340,\"id_str\":\"15062340\",\"indices\":[0,12]}],\"urls\":[{\"url\":\"https://t.co/OY7qmdJQnO\",\"expanded_url\":\"https://twitter.com/i/web/status/714461850188926976\",\"display_url\":\"twitter.com/i/web/status/7…\",\"indices\":[117,140]}]},\"truncated\":true,\"source\":\"<a href=\\\"http://twitter.com\\\" rel=\\\"nofollow\\\">Twitter Web Client</a>\",\"in_reply_to_status_id\":706860403981099008,\"in_reply_to_status_id_str\":\"706860403981099008\",\"in_reply_to_user_id\":15062340,\"in_reply_to_user_id_str\":\"15062340\",\"in_reply_to_screen_name\":\"jeremycloud\",\"user\":{\"id\":4449621923,\"id_str\":\"4449621923\",\"name\":\"Mr Bones\",\"screen_name\":\"MrBonesDroid\",\"location\":\"\",\"profile_location\":null,\"description\":\"\",\"url\":null,\"entities\":{\"description\":{\"urls\":[]}},\"protected\":true,\"followers_count\":5,\"friends_count\":7,\"listed_count\":0,\"created_at\":\"Fri Dec 11 15:18:02 +0000 2015\",\"favourites_count\":7,\"utc_offset\":-25200,\"time_zone\":\"Pacific Time (US &amp; Canada)\",\"geo_enabled\":false,\"verified\":false,\"statuses_count\":35,\"lang\":\"en-gb\",\"contributors_enabled\":false,\"is_translator\":false,\"is_translation_enabled\":false,\"profile_background_color\":\"F5F8FA\",\"profile_background_image_url\":null,\"profile_background_image_url_https\":null,\"profile_background_tile\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/677097663288860672/zZxWCPSI_normal.jpg\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/677097663288860672/zZxWCPSI_normal.jpg\",\"profile_link_color\":\"2B7BB9\",\"profile_sidebar_border_color\":\"C0DEED\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_text_color\":\"333333\",\"profile_use_background_image\":true,\"has_extended_profile\":false,\"default_profile\":true,\"default_profile_image\":false,\"following\":true,\"follow_request_sent\":false,\"notifications\":false},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"quoted_status_id\":704059336788606976,\"quoted_status_id_str\":\"704059336788606976\",\"quoted_status\":{\"created_at\":\"Sun Feb 28 21:43:21 +0000 2016\",\"id\":704059336788606976,\"id_str\":\"704059336788606976\",\"text\":\"My favorite photographic subject, up closer than ever before. https://t.co/K958bKh9Sd\",\"entities\":{\"hashtags\":[],\"symbols\":[],\"user_mentions\":[],\"urls\":[],\"media\":[{\"id\":704059330149031936,\"id_str\":\"704059330149031936\",\"indices\":[62,85],\"media_url\":\"http://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"media_url_https\":\"https://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"url\":\"https://t.co/K958bKh9Sd\",\"display_url\":\"pic.twitter.com/K958bKh9Sd\",\"expanded_url\":\"http://twitter.com/jeremycloud/status/704059336788606976/photo/1\",\"type\":\"photo\",\"sizes\":{\"medium\":{\"w\":600,\"h\":600,\"resize\":\"fit\"},\"thumb\":{\"w\":150,\"h\":150,\"resize\":\"crop\"},\"large\":{\"w\":871,\"h\":871,\"resize\":\"fit\"},\"small\":{\"w\":340,\"h\":340,\"resize\":\"fit\"}}}]},\"extended_entities\":{\"media\":[{\"id\":704059330149031936,\"id_str\":\"704059330149031936\",\"indices\":[62,85],\"media_url\":\"http://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"media_url_https\":\"https://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"url\":\"https://t.co/K958bKh9Sd\",\"display_url\":\"pic.twitter.com/K958bKh9Sd\",\"expanded_url\":\"http://twitter.com/jeremycloud/status/704059336788606976/photo/1\",\"type\":\"photo\",\"sizes\":{\"medium\":{\"w\":600,\"h\":600,\"resize\":\"fit\"},\"thumb\":{\"w\":150,\"h\":150,\"resize\":\"crop\"},\"large\":{\"w\":871,\"h\":871,\"resize\":\"fit\"},\"small\":{\"w\":340,\"h\":340,\"resize\":\"fit\"}}}]},\"truncated\":false,\"source\":\"<a href=\\\"http://twitter.com/download/iphone\\\" rel=\\\"nofollow\\\">Twitter for iPhone</a>\",\"in_reply_to_status_id\":null,\"in_reply_to_status_id_str\":null,\"in_reply_to_user_id\":null,\"in_reply_to_user_id_str\":null,\"in_reply_to_screen_name\":null,\"user\":{\"id\":15062340,\"id_str\":\"15062340\",\"name\":\"/dev/cloud/jeremy\",\"screen_name\":\"jeremycloud\",\"location\":\"Madison, Wisconsin\",\"description\":\"Professional yak shaver. Amateur bike shedder.\",\"url\":\"https://t.co/FcYeBkOpVY\",\"entities\":{\"url\":{\"urls\":[{\"url\":\"https://t.co/FcYeBkOpVY\",\"expanded_url\":\"http://about.me/jeremy.cloud\",\"display_url\":\"about.me/jeremy.cloud\",\"indices\":[0,23]}]},\"description\":{\"urls\":[]}},\"protected\":false,\"followers_count\":4324,\"friends_count\":410,\"listed_count\":103,\"created_at\":\"Mon Jun 09 17:00:58 +0000 2008\",\"favourites_count\":815,\"utc_offset\":-18000,\"time_zone\":\"Central Time (US &amp; Canada)\",\"geo_enabled\":true,\"verified\":false,\"statuses_count\":2218,\"lang\":\"en\",\"contributors_enabled\":false,\"is_translator\":false,\"is_translation_enabled\":false,\"profile_background_color\":\"000000\",\"profile_background_image_url\":\"http://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_image_url_https\":\"https://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_tile\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/436903139183054849/i_MbCcoW_normal.jpeg\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/436903139183054849/i_MbCcoW_normal.jpeg\",\"profile_banner_url\":\"https://pbs.twimg.com/profile_banners/15062340/1447451621\",\"profile_link_color\":\"4A913C\",\"profile_sidebar_border_color\":\"000000\",\"profile_sidebar_fill_color\":\"000000\",\"profile_text_color\":\"000000\",\"profile_use_background_image\":false,\"has_extended_profile\":true,\"default_profile\":false,\"default_profile_image\":false,\"following\":true,\"follow_request_sent\":false,\"notifications\":false},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"is_quote_status\":false,\"retweet_count\":0,\"favorite_count\":11,\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"possibly_sensitive_appealable\":false,\"lang\":\"en\"},\"is_quote_status\":true,\"retweet_count\":0,\"favorite_count\":0,\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"possibly_sensitive_appealable\":false,\"lang\":\"en\"}";

        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        assertEquals(true, status.isTruncated());
        assertEquals(140, status.getText().length());
        assertEquals(-1, status.getDisplayTextRangeStart());
        assertEquals(-1, status.getDisplayTextRangeEnd());
    }

    @Test
    void testExtendedTweetCompatibilityModeStreaming() throws Exception {

        // http://qiita.com/takke/items/c80a687718fc65b0b5b9#extended-tweet-compatibility-mode-streaming-with-extended_tweet
        String rawJson = "{\"created_at\":\"Mon Mar 28 14:39:13 +0000 2016\",\"id\":714461850188926976,\"id_str\":\"714461850188926976\",\"text\":\"@jeremycloud It's neat to have owls and raccoons around until you realize that raccoons will eat the eggs from the … https://t.co/OY7qmdJQnO\",\"entities\":{\"hashtags\":[],\"symbols\":[],\"user_mentions\":[{\"screen_name\":\"jeremycloud\",\"name\":\"/dev/cloud/jeremy\",\"id\":15062340,\"id_str\":\"15062340\",\"indices\":[0,12]}],\"urls\":[{\"url\":\"https://t.co/OY7qmdJQnO\",\"expanded_url\":\"https://twitter.com/i/web/status/714461850188926976\",\"display_url\":\"twitter.com/i/web/status/7…\",\"indices\":[117,140]}]},\"truncated\":true,\"extended_tweet\":{\"full_text\":\"@jeremycloud It's neat to have owls and raccoons around until you realize that raccoons will eat the eggs from the owl's nest https://t.co/Q0pkaU4ORH\",\"display_text_range\":[13,125],\"entities\":{\"hashtags\":[],\"symbols\":[],\"user_mentions\":[{\"screen_name\":\"jeremycloud\",\"name\":\"/dev/cloud/jeremy\",\"id\":15062340,\"id_str\":\"15062340\",\"indices\":[0,12]}],\"urls\":[{\"url\":\"https://t.co/Q0pkaU4ORH\",\"expanded_url\":\"https://twitter.com/jeremycloud/status/704059336788606976\",\"display_url\":\"twitter.com/jeremycloud/st…\",\"indices\":[126,149]}]}},\"source\":\"<a href=\\\"http://twitter.com\\\" rel=\\\"nofollow\\\">Twitter Web Client</a>\",\"in_reply_to_status_id\":706860403981099008,\"in_reply_to_status_id_str\":\"706860403981099008\",\"in_reply_to_user_id\":15062340,\"in_reply_to_user_id_str\":\"15062340\",\"in_reply_to_screen_name\":\"jeremycloud\",\"user\":{\"id\":4449621923,\"id_str\":\"4449621923\",\"name\":\"Mr Bones\",\"screen_name\":\"MrBonesDroid\",\"location\":\"\",\"profile_location\":null,\"description\":\"\",\"url\":null,\"entities\":{\"description\":{\"urls\":[]}},\"protected\":true,\"followers_count\":5,\"friends_count\":7,\"listed_count\":0,\"created_at\":\"Fri Dec 11 15:18:02 +0000 2015\",\"favourites_count\":7,\"utc_offset\":-25200,\"time_zone\":\"Pacific Time (US & Canada)\",\"geo_enabled\":false,\"verified\":false,\"statuses_count\":35,\"lang\":\"en-gb\",\"contributors_enabled\":false,\"is_translator\":false,\"is_translation_enabled\":false,\"profile_background_color\":\"F5F8FA\",\"profile_background_image_url\":null,\"profile_background_image_url_https\":null,\"profile_background_tile\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/677097663288860672/zZxWCPSI_normal.jpg\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/677097663288860672/zZxWCPSI_normal.jpg\",\"profile_link_color\":\"2B7BB9\",\"profile_sidebar_border_color\":\"C0DEED\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_text_color\":\"333333\",\"profile_use_background_image\":true,\"has_extended_profile\":false,\"default_profile\":true,\"default_profile_image\":false,\"following\":true,\"follow_request_sent\":false,\"notifications\":false},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"quoted_status_id\":704059336788606976,\"quoted_status_id_str\":\"704059336788606976\",\"quoted_status\":{\"created_at\":\"Sun Feb 28 21:43:21 +0000 2016\",\"id\":704059336788606976,\"id_str\":\"704059336788606976\",\"text\":\"My favorite photographic subject, up closer than ever before. https://t.co/K958bKh9Sd\",\"entities\":{\"hashtags\":[],\"symbols\":[],\"user_mentions\":[],\"urls\":[],\"media\":[{\"id\":704059330149031936,\"id_str\":\"704059330149031936\",\"indices\":[62,85],\"media_url\":\"http://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"media_url_https\":\"https://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"url\":\"https://t.co/K958bKh9Sd\",\"display_url\":\"pic.twitter.com/K958bKh9Sd\",\"expanded_url\":\"http://twitter.com/jeremycloud/status/704059336788606976/photo/1\",\"type\":\"photo\",\"sizes\":{\"medium\":{\"w\":600,\"h\":600,\"resize\":\"fit\"},\"thumb\":{\"w\":150,\"h\":150,\"resize\":\"crop\"},\"large\":{\"w\":871,\"h\":871,\"resize\":\"fit\"},\"small\":{\"w\":340,\"h\":340,\"resize\":\"fit\"}}}]},\"extended_entities\":{\"media\":[{\"id\":704059330149031936,\"id_str\":\"704059330149031936\",\"indices\":[62,85],\"media_url\":\"http://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"media_url_https\":\"https://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"url\":\"https://t.co/K958bKh9Sd\",\"display_url\":\"pic.twitter.com/K958bKh9Sd\",\"expanded_url\":\"http://twitter.com/jeremycloud/status/704059336788606976/photo/1\",\"type\":\"photo\",\"sizes\":{\"medium\":{\"w\":600,\"h\":600,\"resize\":\"fit\"},\"thumb\":{\"w\":150,\"h\":150,\"resize\":\"crop\"},\"large\":{\"w\":871,\"h\":871,\"resize\":\"fit\"},\"small\":{\"w\":340,\"h\":340,\"resize\":\"fit\"}}}]},\"truncated\":false,\"source\":\"<a href=\\\"http://twitter.com/download/iphone\\\" rel=\\\"nofollow\\\">Twitter for iPhone</a>\",\"in_reply_to_status_id\":null,\"in_reply_to_status_id_str\":null,\"in_reply_to_user_id\":null,\"in_reply_to_user_id_str\":null,\"in_reply_to_screen_name\":null,\"user\":{\"id\":15062340,\"id_str\":\"15062340\",\"name\":\"/dev/cloud/jeremy\",\"screen_name\":\"jeremycloud\",\"location\":\"Madison, Wisconsin\",\"description\":\"Professional yak shaver. Amateur bike shedder.\",\"url\":\"https://t.co/FcYeBkOpVY\",\"entities\":{\"url\":{\"urls\":[{\"url\":\"https://t.co/FcYeBkOpVY\",\"expanded_url\":\"http://about.me/jeremy.cloud\",\"display_url\":\"about.me/jeremy.cloud\",\"indices\":[0,23]}]},\"description\":{\"urls\":[]}},\"protected\":false,\"followers_count\":4324,\"friends_count\":410,\"listed_count\":103,\"created_at\":\"Mon Jun 09 17:00:58 +0000 2008\",\"favourites_count\":815,\"utc_offset\":-18000,\"time_zone\":\"Central Time (US & Canada)\",\"geo_enabled\":true,\"verified\":false,\"statuses_count\":2218,\"lang\":\"en\",\"contributors_enabled\":false,\"is_translator\":false,\"is_translation_enabled\":false,\"profile_background_color\":\"000000\",\"profile_background_image_url\":\"http://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_image_url_https\":\"https://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_tile\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/436903139183054849/i_MbCcoW_normal.jpeg\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/436903139183054849/i_MbCcoW_normal.jpeg\",\"profile_banner_url\":\"https://pbs.twimg.com/profile_banners/15062340/1447451621\",\"profile_link_color\":\"4A913C\",\"profile_sidebar_border_color\":\"000000\",\"profile_sidebar_fill_color\":\"000000\",\"profile_text_color\":\"000000\",\"profile_use_background_image\":false,\"has_extended_profile\":true,\"default_profile\":false,\"default_profile_image\":false,\"following\":true,\"follow_request_sent\":false,\"notifications\":false},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"is_quote_status\":false,\"retweet_count\":0,\"favorite_count\":11,\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"possibly_sensitive_appealable\":false,\"lang\":\"en\"},\"is_quote_status\":true,\"retweet_count\":0,\"favorite_count\":0,\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"possibly_sensitive_appealable\":false,\"lang\":\"en\"}";

        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        assertEquals(true, status.isTruncated());
        assertEquals(149, status.getText().length());

        assertEquals(13, status.getDisplayTextRangeStart());
        assertEquals(125, status.getDisplayTextRangeEnd());
        assertEquals(0, status.getHashtagEntities().length);
        assertEquals(0, status.getSymbolEntities().length);
        assertEquals(1, status.getUserMentionEntities().length);
        assertEquals(1, status.getURLEntities().length);
        assertEquals(0, status.getMediaEntities().length);
    }

    @Test
    void testExtendedTweetCompatibilityModeStreaming_gif() throws Exception {

        // https://gist.github.com/takke/230f1a293071676083b2db90554b9a91
        String rawJson = "{\"coordinates\":null,\"in_reply_to_user_id_str\":null,\"source\":\"<a href=\\\"http:\\/\\/twitter.com\\\" rel=\\\"nofollow\\\">Twitter Web Client<\\/a>\",\"geo\":null,\"in_reply_to_user_id\":null,\"user\":{\"default_profile_image\":false,\"screen_name\":\"takke\",\"profile_link_color\":\"0084B4\",\"profile_background_color\":\"C0DEED\",\"is_translator\":false,\"url\":\"http:\\/\\/www.panecraft.net\",\"profile_background_image_url\":\"http:\\/\\/abs.twimg.com\\/images\\/themes\\/theme1\\/bg.png\",\"statuses_count\":42218,\"profile_image_url_https\":\"https:\\/\\/pbs.twimg.com\\/profile_images\\/423153841505193984\\/yGKSJu78_normal.jpeg\",\"profile_text_color\":\"333333\",\"following\":null,\"profile_sidebar_border_color\":\"C0DEED\",\"verified\":false,\"id\":8379212,\"lang\":\"ja\",\"profile_background_tile\":false,\"listed_count\":133,\"id_str\":\"8379212\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_background_image_url_https\":\"https:\\/\\/abs.twimg.com\\/images\\/themes\\/theme1\\/bg.png\",\"protected\":false,\"utc_offset\":32400,\"profile_use_background_image\":true,\"notifications\":null,\"profile_banner_url\":\"https:\\/\\/pbs.twimg.com\\/profile_banners\\/8379212\\/1390720101\",\"geo_enabled\":true,\"friends_count\":959,\"default_profile\":true,\"followers_count\":1603,\"profile_image_url\":\"http:\\/\\/pbs.twimg.com\\/profile_images\\/423153841505193984\\/yGKSJu78_normal.jpeg\",\"follow_request_sent\":null,\"contributors_enabled\":false,\"description\":\"TwitterクライアントTwitPane、FacebookブラウザTafView、mixiブラウザTkMixiViewer、英単語学習ソフト P-Study System 、MZ3\\/4 などを開発。「ちょっぴり使いやすい」アプリを日々開発しています。ペーンクラフト代表\",\"time_zone\":\"Tokyo\",\"location\":\"北海道\",\"name\":\"竹内裕昭\",\"created_at\":\"Thu Aug 23 10:06:53 +0000 2007\",\"favourites_count\":5592},\"in_reply_to_status_id_str\":null,\"timestamp_ms\":\"1475025189939\",\"favorite_count\":0,\"id\":780938364719542272,\"is_quote_status\":false,\"lang\":\"ja\",\"possibly_sensitive\":false,\"entities\":{\"urls\":[{\"indices\":[117,140],\"display_url\":\"twitter.com\\/i\\/web\\/status\\/7…\",\"expanded_url\":\"https:\\/\\/twitter.com\\/i\\/web\\/status\\/780938364719542272\",\"url\":\"https:\\/\\/t.co\\/3K9rqDsNmV\"}],\"hashtags\":[],\"user_mentions\":[],\"symbols\":[]},\"extended_tweet\":{\"entities\":{\"urls\":[],\"hashtags\":[],\"user_mentions\":[],\"media\":[{\"indices\":[141,164],\"id_str\":\"780938106451009538\",\"expanded_url\":\"https:\\/\\/twitter.com\\/takke\\/status\\/780938364719542272\\/photo\\/1\",\"media_url\":\"http:\\/\\/pbs.twimg.com\\/tweet_video_thumb\\/CtZzEfqUIAIvbOJ.jpg\",\"id\":780938106451009538,\"url\":\"https:\\/\\/t.co\\/8EVW3OeIeN\",\"display_url\":\"pic.twitter.com\\/8EVW3OeIeN\",\"media_url_https\":\"https:\\/\\/pbs.twimg.com\\/tweet_video_thumb\\/CtZzEfqUIAIvbOJ.jpg\",\"type\":\"animated_gif\",\"sizes\":{\"small\":{\"resize\":\"fit\",\"h\":212,\"w\":212},\"thumb\":{\"resize\":\"crop\",\"h\":150,\"w\":150},\"large\":{\"resize\":\"fit\",\"h\":212,\"w\":212},\"medium\":{\"resize\":\"fit\",\"h\":212,\"w\":212}},\"video_info\":{\"aspect_ratio\":[1,1],\"variants\":[{\"bitrate\":0,\"url\":\"https:\\/\\/pbs.twimg.com\\/tweet_video\\/CtZzEfqUIAIvbOJ.mp4\",\"content_type\":\"video\\/mp4\"}]}}],\"symbols\":[]},\"full_text\":\"StreamingのGIF動画のテスト、140文字超の場合：01234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 https:\\/\\/t.co\\/8EVW3OeIeN\",\"display_text_range\":[0,140]},\"id_str\":\"780938364719542272\",\"retweet_count\":0,\"favorited\":false,\"in_reply_to_status_id\":null,\"truncated\":true,\"contributors\":null,\"text\":\"StreamingのGIF動画のテスト、140文字超の場合：0123456789012345678901234567890123456789012345678901234567890123456789012345678901234… https:\\/\\/t.co\\/3K9rqDsNmV\",\"created_at\":\"Wed Sep 28 01:13:09 +0000 2016\",\"filter_level\":\"low\",\"place\":null,\"in_reply_to_screen_name\":null,\"retweeted\":false}";

        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        assertEquals(true, status.isTruncated());
        assertEquals(164, status.getText().length());

        assertEquals(0, status.getDisplayTextRangeStart());
        assertEquals(140, status.getDisplayTextRangeEnd());

        assertEquals(1, status.getMediaEntities().length);

        MediaEntity mediaEntity = status.getMediaEntities()[0];
        assertEquals("animated_gif", mediaEntity.getType());
        assertEquals(1, mediaEntity.getVideoAspectRatioWidth());
        assertEquals(1, mediaEntity.getVideoAspectRatioHeight());
        assertEquals(0, mediaEntity.getVideoDurationMillis());
        assertEquals(1, mediaEntity.getVideoVariants().length);
        assertEquals(0, mediaEntity.getVideoVariants()[0].getBitrate());
        assertEquals("https://pbs.twimg.com/tweet_video/CtZzEfqUIAIvbOJ.mp4", mediaEntity.getVideoVariants()[0].getUrl());
        assertEquals("video/mp4", mediaEntity.getVideoVariants()[0].getContentType());
    }

    @Test
    void testExtendedTweetExtendedMode() throws Exception {

        // http://qiita.com/takke/items/c80a687718fc65b0b5b9#extended-tweet-extended-mode
        String rawJson = "{\"created_at\":\"Mon Mar 28 14:39:13 +0000 2016\",\"id\":714461850188926976,\"id_str\":\"714461850188926976\",\"full_text\":\"@jeremycloud It's neat to have owls and raccoons around until you realize that raccoons will eat the eggs from the owl's nest https://t.co/Q0pkaU4ORH\",\"display_text_range\":[13,125],\"entities\":{\"hashtags\":[],\"symbols\":[],\"user_mentions\":[{\"screen_name\":\"jeremycloud\",\"name\":\"/dev/cloud/jeremy\",\"id\":15062340,\"id_str\":\"15062340\",\"indices\":[0,12]}],\"urls\":[{\"url\":\"https://t.co/Q0pkaU4ORH\",\"expanded_url\":\"https://twitter.com/jeremycloud/status/704059336788606976\",\"display_url\":\"twitter.com/jeremycloud/st…\",\"indices\":[126,149]}]},\"truncated\":false,\"source\":\"<a href=\\\"http://twitter.com\\\" rel=\\\"nofollow\\\">Twitter Web Client</a>\",\"in_reply_to_status_id\":706860403981099008,\"in_reply_to_status_id_str\":\"706860403981099008\",\"in_reply_to_user_id\":15062340,\"in_reply_to_user_id_str\":\"15062340\",\"in_reply_to_screen_name\":\"jeremycloud\",\"user\":{\"id\":4449621923,\"id_str\":\"4449621923\",\"name\":\"Mr Bones\",\"screen_name\":\"MrBonesDroid\",\"location\":\"\",\"profile_location\":null,\"description\":\"\",\"url\":null,\"entities\":{\"description\":{\"urls\":[]}},\"protected\":true,\"followers_count\":5,\"friends_count\":7,\"listed_count\":0,\"created_at\":\"Fri Dec 11 15:18:02 +0000 2015\",\"favourites_count\":7,\"utc_offset\":-25200,\"time_zone\":\"Pacific Time (US & Canada)\",\"geo_enabled\":false,\"verified\":false,\"statuses_count\":35,\"lang\":\"en-gb\",\"contributors_enabled\":false,\"is_translator\":false,\"is_translation_enabled\":false,\"profile_background_color\":\"F5F8FA\",\"profile_background_image_url\":null,\"profile_background_image_url_https\":null,\"profile_background_tile\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/677097663288860672/zZxWCPSI_normal.jpg\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/677097663288860672/zZxWCPSI_normal.jpg\",\"profile_link_color\":\"2B7BB9\",\"profile_sidebar_border_color\":\"C0DEED\",\"profile_sidebar_fill_color\":\"DDEEF6\",\"profile_text_color\":\"333333\",\"profile_use_background_image\":true,\"has_extended_profile\":false,\"default_profile\":true,\"default_profile_image\":false,\"following\":true,\"follow_request_sent\":false,\"notifications\":false},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"quoted_status_id\":704059336788606976,\"quoted_status_id_str\":\"704059336788606976\",\"quoted_status\":{\"created_at\":\"Sun Feb 28 21:43:21 +0000 2016\",\"id\":704059336788606976,\"id_str\":\"704059336788606976\",\"full_text\":\"My favorite photographic subject, up closer than ever before. https://t.co/K958bKh9Sd\",\"display_text_range\":[0,85],\"entities\":{\"hashtags\":[],\"symbols\":[],\"user_mentions\":[],\"urls\":[],\"media\":[{\"id\":704059330149031936,\"id_str\":\"704059330149031936\",\"indices\":[62,85],\"media_url\":\"http://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"media_url_https\":\"https://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"url\":\"https://t.co/K958bKh9Sd\",\"display_url\":\"pic.twitter.com/K958bKh9Sd\",\"expanded_url\":\"http://twitter.com/jeremycloud/status/704059336788606976/photo/1\",\"type\":\"photo\",\"sizes\":{\"medium\":{\"w\":600,\"h\":600,\"resize\":\"fit\"},\"thumb\":{\"w\":150,\"h\":150,\"resize\":\"crop\"},\"large\":{\"w\":871,\"h\":871,\"resize\":\"fit\"},\"small\":{\"w\":340,\"h\":340,\"resize\":\"fit\"}}}]},\"extended_entities\":{\"media\":[{\"id\":704059330149031936,\"id_str\":\"704059330149031936\",\"indices\":[62,85],\"media_url\":\"http://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"media_url_https\":\"https://pbs.twimg.com/media/CcVSOwJVIAAKwE6.jpg\",\"url\":\"https://t.co/K958bKh9Sd\",\"display_url\":\"pic.twitter.com/K958bKh9Sd\",\"expanded_url\":\"http://twitter.com/jeremycloud/status/704059336788606976/photo/1\",\"type\":\"photo\",\"sizes\":{\"medium\":{\"w\":600,\"h\":600,\"resize\":\"fit\"},\"thumb\":{\"w\":150,\"h\":150,\"resize\":\"crop\"},\"large\":{\"w\":871,\"h\":871,\"resize\":\"fit\"},\"small\":{\"w\":340,\"h\":340,\"resize\":\"fit\"}}}]},\"truncated\":false,\"source\":\"<a href=\\\"http://twitter.com/download/iphone\\\" rel=\\\"nofollow\\\">Twitter for iPhone</a>\",\"in_reply_to_status_id\":null,\"in_reply_to_status_id_str\":null,\"in_reply_to_user_id\":null,\"in_reply_to_user_id_str\":null,\"in_reply_to_screen_name\":null,\"user\":{\"id\":15062340,\"id_str\":\"15062340\",\"name\":\"/dev/cloud/jeremy\",\"screen_name\":\"jeremycloud\",\"location\":\"Madison, Wisconsin\",\"description\":\"Professional yak shaver. Amateur bike shedder.\",\"url\":\"https://t.co/FcYeBkOpVY\",\"entities\":{\"url\":{\"urls\":[{\"url\":\"https://t.co/FcYeBkOpVY\",\"expanded_url\":\"http://about.me/jeremy.cloud\",\"display_url\":\"about.me/jeremy.cloud\",\"indices\":[0,23]}]},\"description\":{\"urls\":[]}},\"protected\":false,\"followers_count\":4324,\"friends_count\":410,\"listed_count\":103,\"created_at\":\"Mon Jun 09 17:00:58 +0000 2008\",\"favourites_count\":815,\"utc_offset\":-18000,\"time_zone\":\"Central Time (US & Canada)\",\"geo_enabled\":true,\"verified\":false,\"statuses_count\":2218,\"lang\":\"en\",\"contributors_enabled\":false,\"is_translator\":false,\"is_translation_enabled\":false,\"profile_background_color\":\"000000\",\"profile_background_image_url\":\"http://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_image_url_https\":\"https://abs.twimg.com/images/themes/theme1/bg.png\",\"profile_background_tile\":false,\"profile_image_url\":\"http://pbs.twimg.com/profile_images/436903139183054849/i_MbCcoW_normal.jpeg\",\"profile_image_url_https\":\"https://pbs.twimg.com/profile_images/436903139183054849/i_MbCcoW_normal.jpeg\",\"profile_banner_url\":\"https://pbs.twimg.com/profile_banners/15062340/1447451621\",\"profile_link_color\":\"4A913C\",\"profile_sidebar_border_color\":\"000000\",\"profile_sidebar_fill_color\":\"000000\",\"profile_text_color\":\"000000\",\"profile_use_background_image\":false,\"has_extended_profile\":true,\"default_profile\":false,\"default_profile_image\":false,\"following\":true,\"follow_request_sent\":false,\"notifications\":false},\"geo\":null,\"coordinates\":null,\"place\":null,\"contributors\":null,\"is_quote_status\":false,\"retweet_count\":0,\"favorite_count\":11,\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"possibly_sensitive_appealable\":false,\"lang\":\"en\"},\"is_quote_status\":true,\"retweet_count\":0,\"favorite_count\":0,\"favorited\":false,\"retweeted\":false,\"possibly_sensitive\":false,\"possibly_sensitive_appealable\":false,\"lang\":\"en\"}";

        JSONObject json = new JSONObject(rawJson);
        StatusJSONImpl status = new StatusJSONImpl(json);

        assertEquals(false, status.isTruncated());
        assertEquals(149, status.getText().length());
        assertEquals(13, status.getDisplayTextRangeStart());
        assertEquals(125, status.getDisplayTextRangeEnd());
    }

}
