/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.modcluster.demo.servlet;

import java.io.IOException;
import java.util.Collections;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.methods.HeadMethod;

/**
 * @author Paul Ferraro
 * 
 */
public class BusyConnectorsLoadServlet extends LoadServlet {
    /** The serialVersionUID */
    private static final long serialVersionUID = -946741803216943778L;

    private static final String END = "end";

    /**
     * @{inheritDoc
     * @see javax.servlet.http.HttpServlet#service(javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void service(HttpServletRequest request, HttpServletResponse response) throws IOException {
        String parameter = request.getParameter(END);

        if (parameter == null) {
            int duration = Integer.parseInt(this.getParameter(request, DURATION, "15")) * 1000;

            long end = System.currentTimeMillis() + duration;

            String url = this.createLocalURL(request, Collections.singletonMap(END, String.valueOf(end)));
            Runnable task = new ExecuteMethodTask(url);

            int count = Integer.parseInt(this.getParameter(request, COUNT, "50"));

            this.log("Sending " + count + " concurrent requests to: " + url);

            Thread[] threads = new Thread[count];

            for (int i = 0; i < count; ++i) {
                threads[i] = new Thread(task);
            }

            for (int i = 0; i < count; ++i) {
                threads[i].start();
            }

            for (int i = 0; i < count; ++i) {
                try {
                    threads[i].join();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }

            this.writeLocalName(request, response);
        } else {
            long end = Long.parseLong(parameter);

            if (end > System.currentTimeMillis()) {
                String url = this.createLocalURL(request, Collections.singletonMap(END, String.valueOf(end)));
                response.setStatus(307);
                response.setHeader("location", response.encodeRedirectURL(url));
            }
        }
    }

    private class ExecuteMethodTask implements Runnable {
        private final String url;

        ExecuteMethodTask(String url) {
            this.url = url;
        }

        @Override
        public void run() {
            HttpClient client = new HttpClient();
            String url = this.url;

            try {
                while (url != null) {
                    HttpMethod method = new HeadMethod(url);
                    // Disable auto redirect following, to allow circular redirect
                    method.setFollowRedirects(false);

                    int code = client.executeMethod(method);

                    url = (code == 307) ? method.getResponseHeader("location").getValue() : null;
                }
            } catch (HttpException e) {
                BusyConnectorsLoadServlet.this.log(e.getMessage(), e);
            } catch (IOException e) {
                BusyConnectorsLoadServlet.this.log(e.getMessage(), e);
            }
        }
    }
}
