__author__ = 'slaskawi@redhat.com'

import re
import logging
from string import lower, replace


class JDGReleaseConfiguration:
    REGEXP_PRODUCT_GROUP = 1
    REGEXP_VERSION_GROUP = 2
    REGEXP_RELEASE_TYPE_GROUP = 3
    REGEXP_MODIFIER_GROUP = 5
    RELEASE_NAME_VALIDATION_REGEXP = r"(JDG)_(\d\.\d\.\d)\.((DR|ER|GA|CR)\d)-(redhat-\d)"

    def __init__(self, release_name, productization_branch_name, release_directory, releasing_user_name, keytab_file,
                 dry_run):
        """Creates new instance of release configuration.
        :param release_name:
            Full release name, for example: JDG_6.4.0.DR1-redhat-1"""

        self._release_commit_message = release_name
        match = re.search(JDGReleaseConfiguration.RELEASE_NAME_VALIDATION_REGEXP, release_name)
        if (match is None) or match.end() != len(release_name):
            raise AttributeError(
                "Release name must match regexp: " + JDGReleaseConfiguration.RELEASE_NAME_VALIDATION_REGEXP)

        self._product = match.group(JDGReleaseConfiguration.REGEXP_PRODUCT_GROUP)
        self._version = match.group(JDGReleaseConfiguration.REGEXP_VERSION_GROUP)
        self._releaseType = match.group(JDGReleaseConfiguration.REGEXP_RELEASE_TYPE_GROUP)
        self._modifier = match.group(JDGReleaseConfiguration.REGEXP_MODIFIER_GROUP)
        self._release_directory = release_directory
        self._releasing_user_name = releasing_user_name
        self._upstream_branch = productization_branch_name
        self._packaging_branch = self._upstream_branch
        self._productization_branch = self._upstream_branch + "-prod"
        self._productization_base_tag = self._product + "_" + self._version + "." + self._releaseType + "-base"
        self._release_commit_message = self._product + " " + self._version + "." + self._releaseType
        self._release_tag_name = self._product + "_" + self._version + "." + self._releaseType
        self._dry_run = dry_run
        self._keytab_file = keytab_file

        logging.info("#### JDG Release configuration ####")
        logging.info("Product Name:                  " + self._product)
        logging.info("Product Version:               " + self._version)
        logging.info("Product Release Type:          " + self._releaseType)
        logging.info("Product Modifier:              " + self._modifier)
        logging.info("Release directory:             " + self._release_directory)
        logging.info("Releasing user:                " + self._releasing_user_name)
        logging.info("Product repo address:          " + self.get_product_repo_address())
        logging.info("Packaging repo address:        " + self.get_packaging_repo_address())
        logging.info("Product repo location:         " + self.get_productization_repo_location())
        logging.info("Packaging repo location:       " + self.get_packaging_repo_location())
        logging.info("Product upstream branch:       " + self._upstream_branch)
        logging.info("Packaging branch:              " + self._packaging_branch)
        logging.info("Product productization branch: " + self._productization_branch)
        logging.info("Productization commit message: " + self._release_commit_message)
        logging.info("Productization base tag:       " + self._productization_base_tag)
        logging.info("Productization release tag:    " + self._release_tag_name)
        logging.info("Dry run:                       " + str(self._dry_run))
        logging.info("Keytab file:                   " + str(self._keytab_file))
        logging.info("#### /JDG Release configuration ####")

    def get_productization_repo_location(self):
        return "jdg"

    def get_packaging_repo_location(self):
        return "packaging"

    def get_product_repo_address(self):
        return "git+ssh://" + self._releasing_user_name + "@git.app.eng.bos.redhat.com/srv/git/infinispan/infinispan.git"

    def get_packaging_repo_address(self):
        # return "git+ssh://" + self._releasing_user_name + "@git.app.eng.bos.redhat.com/srv/git/jdg-packaging.git"
        return "ssh://" + self._releasing_user_name + "@code.engineering.redhat.com:22/jdg-packaging"

    def get_upstream_repo_address(self):
        return "git@github.com:infinispan/jdg.git"

    def get_upstream_branch(self):
        return self._upstream_branch

    def get_packaging_branch(self):
        return self._packaging_branch

    def get_productization_branch(self):
        return self._productization_branch

    def get_productization_base_tag(self):
        return self._productization_base_tag

    def get_release_version_major(self):
        """
        :return: Major part of the version
        """
        return self._version[0:1]

    def get_release_version_minor(self):
        """
        :return: Minor part of the version
        """
        return self._version[2:3]

    def get_release_version_micro(self):
        """
        :return: Micro part of the version
        """
        return self._version[4:5]

    def get_release_version_modifier(self):
        """
        :return: Version modifier, e.g. redhat-1
        """
        return self._modifier

    def get_full_release_version(self):
        """
        :return: Full version identifies, e.g. 6.2.0.DR1.redhat-1.
        """
        return self._version + "." + self._releaseType + "." + self._modifier

    def get_release_version(self):
        """
        :return: Version identifies, e.g. 6.2.0.
        """
        return self._version

    def get_release_tag_name(self):
        return self._release_tag_name

    def get_release_commit_message(self):
        return self._release_commit_message

    def get_dry_run(self):
        return self._dry_run

    def get_release_directory(self):
        return self._release_directory

    def get_keytab_file(self):
        return self._keytab_file

    def get_releasing_username(self):
        return self._releasing_user_name

    def get_brew_repo_for_regeneration(self):
        return "jb-edg-6-rhel-6-build"

    def get_brew_repo_for_build(self):
        return "jb-edg-6-jdk7-rhel-6-candidate"

    def get_dependency_information_directory(self):
        return self.get_productization_repo_location() + "/target/dependency-report"

    def get_brew_dependency_information_destination(self):
        return self._releasing_user_name + "@x86-014.build.bos.redhat.com:/mnt/redhat/devel/jdg/stage"