__author__ = 'slaskawi@redhat.com'

import re
import logging
from string import lower, replace


class JDGReleaseConfiguration:
    REGEXP_PRODUCT_GROUP = 1
    REGEXP_VERSION_GROUP = 2
    REGEXP_RELEASE_TYPE_GROUP = 3
    REGEXP_MODIFIER_GROUP = 5
    RELEASE_MODIFIER_VALIDATION_REGEXP = r"((DR|ER|GA|CR)\d+)-(redhat-\d+)"
    RELEASE_VERSION_VALIDATION_REGEXP = r"\d+\.\d+\.\d+"
    VERSION_MODIFIER_REGEXP = r"[\.|-]redhat-\d+"

    def __init__(self, product_version, infinispan_version, release_modifier, productization_branch_name, release_directory, releasing_user_name, keytab_file,
                 dry_run, skip_clients):
        """Creates new instance of release configuration.
        :param product_version:
            Version of the product e.g. 6.4.0
        :param infinispan_version:
            Version of the Infinispan e.g. 6.2.0
        :param release_modifier:
            Release modifier e.g. ER1-redhat-1
            """

        self._validateRegexp(JDGReleaseConfiguration.RELEASE_MODIFIER_VALIDATION_REGEXP, release_modifier, "Modifier does not match regexp")
        self._validateRegexp(JDGReleaseConfiguration.RELEASE_VERSION_VALIDATION_REGEXP, product_version, "Product version does match regexp")
        self._validateRegexp(JDGReleaseConfiguration.RELEASE_VERSION_VALIDATION_REGEXP, infinispan_version, "Infinispan version does match regexp")

        self._infinispan_version = infinispan_version
        self._camel_version = product_version
        self._product_version = product_version
        self._release_modifier = release_modifier
        self._modifier = re.sub(JDGReleaseConfiguration.VERSION_MODIFIER_REGEXP,"",release_modifier)
        self._release_directory = release_directory
        self._releasing_user_name = releasing_user_name
        self._upstream_branch = productization_branch_name
        self._productization_branch = self._upstream_branch + "-prod"
        self._release_commit_message = "JDG_" + product_version + "." + release_modifier[0:3]
        self._productization_base_tag = self._release_commit_message + "-base"
        self._release_tag_name = self._release_commit_message
        self._dry_run = dry_run
        self._skip_clients = skip_clients
        self._keytab_file = keytab_file

        logging.info("#### JDG Release configuration ####")
        logging.info("Product Version:               " + self._product_version)
        logging.info("Infinispan Version:            " + self._infinispan_version)
        logging.info("Camel Version:                 " + self._camel_version)
        logging.info("Product Modifier:              " + self._release_modifier)
        logging.info("Release directory:             " + self._release_directory)
        logging.info("Releasing user:                " + self._releasing_user_name)
        logging.info("Product repo address:          " + self.get_product_repo_address())
        logging.info("Packaging repo address:        " + self.get_packaging_repo_address())
        logging.info("Product repo location:         " + self.get_productization_repo_location())
        logging.info("Packaging repo location:       " + self.get_packaging_repo_location())
        logging.info("Quickstarts repo location:     " + self.get_quickstarts_repo_location())       
        logging.info("C++ client repo address:       " + self.get_cpp_upstream_repo_address())
        logging.info("C++ client repo location:      " + self.get_cpp_repo_location())
        logging.info("C# client repo address:        " + self.get_dotnet_upstream_repo_address())
        logging.info("C# client repo location:       " + self.get_dotnet_repo_location())
        logging.info("Quickstarts repo location:     " + self.get_quickstarts_repo_location())
        logging.info("Product upstream branch:       " + self._upstream_branch)
        logging.info("Product productization branch: " + self._productization_branch)
        logging.info("Productization commit message: " + self._release_commit_message)
        logging.info("Productization base tag:       " + self._productization_base_tag)
        logging.info("Productization release tag:    " + self._release_tag_name)
        logging.info("Dry run:                       " + str(self._dry_run))
        logging.info("Keytab file:                   " + str(self._keytab_file))
        logging.info("#### /JDG Release configuration ####")

    def get_productization_repo_location(self):
        return "jdg"

    def get_packaging_repo_location(self):
        return "packaging"

    def get_camel_productization_repo_location(self):
        return "camel"
    
    def get_cpp_repo_location(self):
        return "cpp-client"
    
    def get_dotnet_repo_location(self):
        return "dotnet-client" 

    def get_quickstarts_repo_location(self):
        return "jdg-quickstarts" 

    def get_product_repo_address(self):
        return "git+ssh://" + self._releasing_user_name + "@git.app.eng.bos.redhat.com/srv/git/infinispan/infinispan.git"

    def get_packaging_repo_address(self):
        return "ssh://" + self._releasing_user_name + "@code.engineering.redhat.com:22/jdg-packaging"

    def get_camel_productization_repo_address(self):
        return "git@github.com:infinispan/jdg-camel.git"

    def get_upstream_repo_address(self):
        return "git@github.com:infinispan/jdg.git"

    def get_cpp_upstream_repo_address(self):
        return "git@github.com:infinispan/jdg-cpp-client.git"
    
    def get_dotnet_upstream_repo_address(self):
        return "git@github.com:infinispan/jdg-dotnet-client.git"
    
    def get_quickstarts_repo_address(self):
        return "git@github.com:jboss-developer/jboss-jdg-quickstarts.git"

    def get_upstream_branch(self):
        return self._upstream_branch

    def get_productization_branch(self):
        return self._productization_branch

    def get_productization_base_tag(self):
        return self._productization_base_tag

    def get_full_infinispan_release_version(self):
        """
        :return: Full version identifies, e.g. 6.2.0.DR1-redhat-1.
        """
        return self._infinispan_version + "." + self._release_modifier

    def get_full_camel_snapshot_version(self):
        """
        :return: Full snapshot version, e.g. 2.15.0-redhat-SNAPSHOT.
        """
        return self._camel_version + "-redhat-SNAPSHOT"

    def get_full_camel_release_version(self):
        """
        :return: Full version identifies, e.g. 2.15.0.DR1-redhat-1.
        """
        return self._camel_version + "." + self._release_modifier

    def get_infinispan_release_version_major(self):
        """
        :return: Major part of the version
        """
        return self._infinispan_version[0:1]

    def get_infinispan_release_version_minor(self):
        """
        :return: Minor part of the version
        """
        return self._infinispan_version[2:3]

    def get_infinispan_release_version_micro(self):
        """
        :return: Micro part of the version
        """
        return self._product_version[4:5]

    def get_release_version_modifier(self):
        """
        :return: Version modifier, e.g. DR1-redhat-1
        """
        return self._release_modifier
    
    def get_version_modifier(self):
        """
        :return: Version modifier without '-redhat-x' suffix, e.g. DR1
        """
        return self._modifier

    def get_product_release_version(self):
        """
        :return: Version identifies, e.g. 6.2.0.
        """
        return self._product_version

    def get_release_tag_name(self):
        return self._release_tag_name

    def get_release_commit_message(self):
        return self._release_commit_message
    
    def get_release_url(self):
        return "http://download.eng.bos.redhat.com/devel/jdg/stage/JDG-" + \
            self._product_version + "-" + self._modifier

    def get_dry_run(self):
        return str(self._dry_run).lower()
    
    def get_skip_clients(self):
        return self._skip_clients

    def get_release_directory(self):
        return self._release_directory

    def get_keytab_file(self):
        return self._keytab_file

    def get_releasing_username(self):
        return self._releasing_user_name

    def get_brew_repo_for_regeneration(self):
        return "jb-edg-6-rhel-6-build"

    def get_brew_repo_for_build(self):
        return "jb-edg-6-jdk7-rhel-6-candidate"

    def get_dependency_information_directory(self):
        return self.get_productization_repo_location() + "/target/dependency-report"

    def release_destination_host(self):
        return self._releasing_user_name + "@x86-014.build.bos.redhat.com"

    def get_relese_destination_directory(self):
        return "/mnt/redhat/devel/jdg/stage"

    def get_release_destination_for_artifacts(self):
        return self.release_destination_host() + ":" + self.get_relese_destination_directory()

    def get_release_destination_scripts(self):
        return self.release_destination_host() + ":" + self.get_release_scripts_destination_directory()

    def get_release_scripts_destination_directory(self):
        return self.get_relese_destination_directory() + "/JDG-" + self.get_product_release_version() + "-" + self.get_version_modifier()

    def get_quickstarts_zip_location(self):
        return self.get_quickstarts_repo_location() + "/target/jboss-datagrid-" + self._product_version + "." + \
            self._modifier + "-quickstarts.zip"

    def _validateRegexp(self, regexp, text, exception_text):
        match = re.search(regexp, text)
        if (match is None) or match.end() != len(text):
            raise AttributeError(
                exception_text + " : " + regexp)