﻿using Infinispan.HotRod.Impl;
using Infinispan.HotRod.Config;
using Infinispan.HotRod;
using NUnit.Framework;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System;

namespace Infinispan.HotRod.Tests
{
    [TestFixture]
    public abstract class SingleServerAbstractTest
    {
        private Process hrServer;

        protected Configuration conf;

        protected RemoteCacheManager remoteManager;

        [TestFixtureSetUp]
        public void BeforeClass()
        {
            try {
                StartHotrodServer();
            } catch (Exception ex) {
                ShutDownHotrodServer();
                Console.WriteLine(ex.StackTrace);
                throw ex;
            }
        }

        public void StartHotrodServer()
        {
            ConfigurationBuilder builder = new ConfigurationBuilder();
            conf = builder.AddServers("127.0.0.1:11222").Build();

            string jbossHome = System.Environment.GetEnvironmentVariable("JBOSS_HOME");
            if (jbossHome == null)
            {
                throw new Exception("JBOSS_HOME env variable not set.");
            }

            Assert.IsTrue(PortProbe.IsPortClosed(conf.Servers()[0].Host(),
                                                 conf.Servers()[0].Port(),
                                                 millisTimeout:10000),
                          "Another process already listening on the same ip/port.");

            hrServer = new Process();
            hrServer.StartInfo.FileName = buildStartCommand(jbossHome);
            hrServer.StartInfo.UseShellExecute = false;
            if (PlatformUtils.isUnix()) {
                // Drop the output generated by the server on the console (data present in log file).
                hrServer.StartInfo.RedirectStandardOutput = true;
                hrServer.StartInfo.RedirectStandardError = true;
                hrServer.OutputDataReceived += new DataReceivedEventHandler(DropOutputHandler);
                hrServer.ErrorDataReceived += new DataReceivedEventHandler(DropOutputHandler);
            }
            hrServer.Start();

            Assert.IsTrue(PortProbe.IsPortOpen(conf.Servers()[0].Host(),
                                               conf.Servers()[0].Port()),
                          "Server not listening on the expected ip/port.");

            remoteManager = new RemoteCacheManager(conf);
            remoteManager.Start();
        }

        [TestFixtureTearDown]
        public void ShutDownHotrodServer()
        {
            if (remoteManager != null && remoteManager.IsStarted()) {
                remoteManager.Stop();
                Assert.IsFalse(remoteManager.IsStarted());
            }

            if (hrServer != null) {
                PlatformUtils.killServer(hrServer);

                Assert.IsTrue(PortProbe.IsPortClosed(conf.Servers()[0].Host(),
                                                     conf.Servers()[0].Port(),
                                                     millisTimeout:10000),
                              "A process is still listening on the ip/port. Kill failed?");
            }
        }

        private string buildStartCommand(string homePath)
        {
            if (PlatformUtils.isUnix()) {
                return Path.Combine(homePath, "bin/standalone.sh");
            } else {
                return Path.Combine(homePath, "bin\\standalone.bat");
            }
        }

        private static void DropOutputHandler(object sendingProcess, DataReceivedEventArgs outLine)
        {
            // Do nothing. Drop the data.
        }
    }
}
