package cosminio

import (
	"bufio"
	"bytes"
	"crypto/tls"
	"crypto/x509"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"reflect"
	"regexp"
	"strconv"
	"strings"
	"time"

	minio "github.com/minio/minio-go"
	cisv1alpha1 "github.com/open-cluster-management/cis-controller/pkg/apis/cis/v1alpha1"
	yaml "gopkg.in/yaml.v2"
)

var ruleMap map[string]string

// ClusterName used in the multi-cluster context
var ClusterName string

//ScorePath path to score file if there is any
var ScorePath string

//Bucket is a COS bucket that holds files
type Bucket struct {
	Name             string `json:"name"`
	UseNotify        bool   `json:"useNotify"`
	NotifyKafkaTopic string `json:"notifyKafkaTopic"`
	MinioNotifyIdx   string `json:"minioNotifyIdx"`
}

//Credentials used to access the COS
type Credentials struct {
	AccessKey string `json:"accessKey"`
	SecretKey string `json:"secretKey"`
}

//CosCtl to access COS
//var CosCtl *CosController

//CosConfig includs access info
type CosConfig struct {
	EndPoint   string
	Creds      Credentials `json:"credentials"`
	Region     string      `json:"region"`
	UseSSL     bool        `json:"useSSL"`
	BucketList []Bucket    `json:"buckets"`
}

//CosController includes the connection client
type CosController struct {
	Config      CosConfig
	conn        *minio.Client
	Initialized bool
}

//ResultMap stores the results of a node
type ResultMap map[string]string

//CISRulesScore calculated based on ML finding, the values are not expected to change.
//in case we want to override these values, we can read them from a file using: SetCISMasterRulesScore("/path/to/file", readFile)
var CISRulesScore = map[string]float64{
	"1.1.1":  6.4,
	"1.1.2":  6.4,
	"1.1.3":  6.4,
	"1.1.4":  6.4,
	"1.1.5":  6.4,
	"1.1.6":  6.4,
	"1.1.7":  7.5,
	"1.1.8":  6.1,
	"1.1.9":  7.5,
	"1.1.10": 6.8,
	"1.1.11": 3.3,
	"1.1.12": 7.2,
	"1.1.13": 6.4,
	"1.1.14": 7.5,
	"1.1.15": 3.0,
	"1.1.16": 5.5,
	"1.1.17": 3.6,
	"1.1.18": 3.6,
	"1.1.19": 6.4,
	"1.1.20": 5.0,
	"1.1.21": 6.4,
	"1.1.22": 6.4,
	"1.1.23": 6.4,
	"1.1.24": 6.4,
	"1.1.25": 6.4,
	"1.1.26": 6.4,
	"1.1.27": 1.7,
	"1.1.28": 6.4,
	"1.1.29": 7.5,
	"1.1.30": 5.0,
	"1.1.31": 6.4,
	"1.1.32": 6.4,
	"1.1.33": 3.6,
	"1.1.34": 6.4,
	"1.1.35": 3.6,
	"1.1.36": 6.4,
	"1.1.37": 6.4,
	"1.1.38": 5.0,
	"1.1.39": 6.4,
	"1.2.1":  6.1,
	"1.2.2":  6.4,
	"1.3.1":  5.0,
	"1.3.2":  6.1,
	"1.3.3":  6.4,
	"1.3.4":  6.4,
	"1.3.5":  0.0,
	"1.3.6":  6.4,
	"1.3.7":  5.0,
	"1.4.1":  4.6,
	"1.4.2":  6.4,
	"1.4.3":  4.3,
	"1.4.4":  3.6,
	"1.4.5":  4.3,
	"1.4.6":  3.6,
	"1.4.7":  4.6,
	"1.4.8":  5.0,
	"1.4.9":  4.6,
	"1.4.10": 6.4,
	"1.4.11": 3.6,
	"1.4.12": 6.4,
	"1.4.13": 7.5,
	"1.4.14": 3.6,
	"1.4.15": 3.6,
	"1.4.16": 3.6,
	"1.4.17": 3.6,
	"1.4.18": 3.6,
	"1.4.19": 2.1,
	"1.4.20": 7.5,
	"1.4.21": 3.6,
	"1.5.1":  6.4,
	"1.5.2":  6.4,
	"1.5.3":  6.4,
	"1.5.4":  6.4,
	"1.5.5":  6.4,
	"1.5.6":  6.4,
	"1.6.1":  6.1,
	"1.6.2":  4.3,
	"1.7.1":  3.6,
	"1.7.2":  3.6,
	"1.7.3":  6.4,
	"1.7.4":  3.6,
	"1.7.5":  4.6,
	"2.1.1":  6.4,
	"2.1.2":  6.4,
	"2.1.3":  6.4,
	"2.1.4":  6.4,
	"2.1.5":  6.4,
	"2.1.6":  3.6,
	"2.1.7":  6.8,
	"2.1.8":  6.4,
	"2.1.9":  3.3,
	"2.1.10": 5.0,
	"2.1.11": 6.4,
	"2.1.12": 6.4,
	"2.1.13": 7.5,
	"2.1.14": 5.0,
	"2.2.1":  3.6,
	"2.2.2":  3.6,
	"2.2.3":  6.1,
	"2.2.4":  0.0,
	"2.2.5":  4.6,
	"2.2.6":  3.6,
	"2.2.7":  4.6,
	"2.2.8":  6.4,
	"2.2.9":  3.6,
	"2.2.10": 3.6,
}

func init() {
	ruleMap = make(map[string]string)
	ScorePath = os.Getenv("CIS_SCORE") // if = "" we use default
	if ScorePath == "" {
		log.Printf("env variable `CIS_SCORE` is not set, using default score")
	}
	err := SetCISRulesScore(ScorePath, readFile)
	if err != nil {
		log.Printf("Error setting CIS Rules Scores: %v\n", err.Error())
	}
}

// SetCISRulesScore reads the default values from a file, typically a mounted config map to the pod
func SetCISRulesScore(path string, readFile FileReader) error {
	//try to read them from a local file that is mounted from a configmap
	if path == "" {
		return fmt.Errorf("path to score file not found, using defaults")
	}
	bytes, err := readFile(path)
	if err != nil {
		//use defaults in CISRulesScore
		return err
	}
	var riskData interface{}

	if err := yaml.Unmarshal(bytes, &riskData); err != nil {
		log.Printf("error parsing the object `%v`, err = %v\n", path, err.Error())
		return err
	}
	// read map values:
	if reflect.ValueOf(riskData).Kind() == reflect.Map {
		riskTempMap := riskData.(map[interface{}]interface{})
		//override the default values of the CISRulesScore
		for key, val := range riskTempMap {
			ketStr := fmt.Sprintf("%v", key)
			valFloat, err := strconv.ParseFloat(fmt.Sprintf("%v", val), 64)
			if err != nil {
				log.Printf("error parsing the value `%v`, err = %v\n", val, err.Error())
				return err
			}
			if ketStr != "" {
				CISRulesScore[ketStr] = valFloat
			}
		}
	}
	return err
}

//FileReader a type that can be implemented by a func
type FileReader func(path string) (b []byte, err error)

func readFile(path string) (b []byte, err error) {
	bytes, err := ioutil.ReadFile(path) // #nosec G304
	if err != nil {
		log.Println(err)
		//no file exists => use defaults
		return nil, err
	}
	return bytes, nil
}

//==============================================
// getObject retrieves an oject from the bucket
func (cosCtl *CosController) getObject(objKey string) (f []byte, err error) {

	objParts := strings.Split(objKey, "/")
	if len(objParts) < 2 {
		return nil, fmt.Errorf("malformed objKey, `%+v`, cannot be retreived from COS ", objKey)
	}
	bucketName := string(objParts[0])
	objName := strings.Split(objKey, bucketName+"/")[1]

	tmpfn, err := ioutil.TempFile(os.TempDir(), "cis")
	defer os.Remove(tmpfn.Name())

	//download object
	fmt.Printf(cosCtl.Config.Region)
	err = cosCtl.conn.FGetObject(bucketName, objName, tmpfn.Name(), minio.GetObjectOptions{})
	if err != nil {
		log.Println(err)
		//return []byte(""), err
		return nil, err
	}

	var buf []byte
	buf, _ = ioutil.ReadFile(tmpfn.Name()) // in case we want to retrun a []byte
	return buf, nil
}

//=================================================================
//checkAndCreateBucket checks if the bucket exists, if not, it creates it
func (cosCtl *CosController) checkAndCreateBucket(cosBucket Bucket) error {
	err := cosCtl.conn.MakeBucket(cosBucket.Name, cosCtl.Config.Region)
	if err != nil {
		// Check to see if we already own this bucket (which happens if you run this twice)
		exists, err := cosCtl.conn.BucketExists(cosBucket.Name)
		if err == nil && exists {
			log.Printf("We already own %s\n", cosBucket.Name)
		} else {
			log.Fatalln("checkAndCreateBucket error, ", err)
			return err
		}
	}
	log.Printf("Successfully created %s\n", cosBucket.Name)

	if cosBucket.UseNotify {
		queueArn := minio.NewArn("minio", "sqs", cosCtl.Config.Region, cosBucket.MinioNotifyIdx, "kafka")
		queueConfig := minio.NewNotificationConfig(queueArn)
		queueConfig.AddEvents(minio.ObjectCreatedAll, minio.ObjectRemovedAll)
		bucketNotification := minio.BucketNotification{}
		bucketNotification.AddQueue(queueConfig)

		err = cosCtl.conn.SetBucketNotification(cosBucket.Name, bucketNotification)
		if err != nil {
			log.Fatalln("Unable to set the bucket notification: ", err)
			return err
		}
		log.Println("Notifications enabled on kafka bucket")
	}
	return nil
}

//InitializeCosController will initialize the cosCtl and its minio client
func InitializeCosController(cosCtl *CosController) error {
	// Initialize minio client object.
	minioClient, err := minio.New(cosCtl.Config.EndPoint, cosCtl.Config.Creds.AccessKey,
		cosCtl.Config.Creds.SecretKey, cosCtl.Config.UseSSL)
	if err != nil {
		log.Fatalln("minioClient client setting error, ", err)
		return err
	}

	// Only want to use one CA, start new pool
	rootCAs := x509.NewCertPool()

	// Path to generated CA
	genCA := "/etc/cisCerts/ca.cert"

	// Read in the CA file
	certs, _ := readFile(genCA)

	// Append our cert to the pool
	if ok := rootCAs.AppendCertsFromPEM(certs); !ok {
		log.Println("No certs in pool")
	}

	minioClient.SetCustomTransport(&http.Transport{
		MaxIdleConns:    10,
		IdleConnTimeout: 30 * time.Second,
		TLSClientConfig: &tls.Config{RootCAs: rootCAs},
	})
	cosCtl.conn = minioClient
	cosCtl.Initialized = true

	return err
}

//=================================================================
//getK8sCisMasterResult fetches a CIS policy
func (cosCtl *CosController) getK8sCisResults(instance *cisv1alpha1.CisPolicy) (resMap map[string]ResultMap, err error) {
	log.Println("Entering getK8sCisResults...")
	var buf []byte

	doneCh := make(chan struct{})

	// Indicate to our routine to exit cleanly upon return.
	defer close(doneCh)

	//resultMap := make(map[CisControl]string)
	resultMap := make(map[string]ResultMap)

	// List all objects from a bucket-name with a matching prefix.
	for object := range cosCtl.conn.ListObjects("cis-k8s", "icp-local/recent/", true, doneCh) {
		if object.Err != nil {
			fmt.Println(object.Err)
			return
		}
		resMap := make(map[string]string)
		nodeName := strings.SplitN(object.Key, "/", 5)[3]
		fmt.Println("cis-k8s/icp-local/recent/" + object.Key)
		buf, err = cosCtl.getObject("cis-k8s/" + object.Key)
		if err != nil {
			return nil, err
		}

		scanner := bufio.NewScanner(bytes.NewReader(buf))
		if err := scanner.Err(); err != nil {
			log.Fatal("getK8sCisResult NewScanner error ", err)
		}
		for scanner.Scan() { //Scanner does not deal well with lines longer than 65536 characters. this is not an issue for CIS tests. For bigger files, I should probably use Reader.Read() with new line detection

			if strings.Contains(scanner.Text(), "== Remediations ==") {
				break
			} else {
				if strings.Contains(scanner.Text(), "[FAIL]") || strings.Contains(scanner.Text(), "[PASS]") || strings.Contains(scanner.Text(), "[WARN]") {
					splitted := strings.SplitN(scanner.Text(), " ", 3)
					if len(splitted) < 3 {
						return nil, fmt.Errorf("invalide test definition, missing information -> %v", splitted)
					}
					if validateRuleNumber(splitted[1]) {
						resMap[splitted[1]] = splitted[0]
					} else {
						return nil, fmt.Errorf("invalide test definition, the test number is not valid -> %v", splitted[1])
					}
				}
			}
		}
		resultMap[nodeName] = resMap
	}

	log.Println("Exiting getK8sCisResults...")

	return resultMap, err
}

func getFakeResults(instance *cisv1alpha1.CisPolicy) (b []byte, err error) {
	path := "./config/csiResults/2018-12-12T14_21_20+0000.stdout"
	if ExtractNamespaceLabel(instance) == "cluster1" {
		path = "./config/csiResults/2019-12-12T14_21_20+0000.stdout"
	}
	file, err := os.Open(path)
	defer file.Close()
	if err != nil {
		log.Printf("error fetching the CIS results from path `%v`, %v, using local value", path, err)
		b = createTmpFile()
		return b, nil
	}
	b, err = ioutil.ReadFile(file.Name())
	return b, err
}

//createTmpFile creates a tmp file for testing
func createTmpFile() []byte {
	return []byte("[FAIL] 1.1.1 Ensure that the --anonymous-auth argument is set to false (Scored)")
}

// getCosObjectKey should retrieve the score that corresponds to master or worker node
func getCosObjectKey() string {
	//TODO add the logic here that figures out how the results are sotred in COS
	//return "kube-master-results"
	return "cis-k8s/icp-local/historical/master/9.42.25.233/2019-08-18T00:59:25+0000.stdout"
}

// ExtractNamespaceLabel to find out the cluster-namespace from the label
func ExtractNamespaceLabel(instance *cisv1alpha1.CisPolicy) string {
	if instance.ObjectMeta.Labels == nil {
		return ""
	}
	if _, ok := instance.ObjectMeta.Labels["cluster-namespace"]; ok {
		return instance.ObjectMeta.Labels["cluster-namespace"]
	}
	return ""
}

//=================================================================
//compareCisMasterResults compares the results of the CIS benchmark, against the policy specs and changes the policy status
func compareCisResults(instance *cisv1alpha1.CisPolicy, resultMap map[string]ResultMap) (err error) {
	compliant := true
	ClusterName = ExtractNamespaceLabel(instance)
	if ClusterName == "" {
		ClusterName = "cluster-" + instance.Name
	}
	if instance.Status.CisPolicyStatus[ClusterName] == nil {
		return fmt.Errorf("un-initialized map for ClusterName %v for the field  -> (instance.Status.CisPolicyStatus) %v", ClusterName, instance.Name)
	}
	var excludeRules []string
	excludeRules = append(excludeRules, instance.Spec.KubernetesCisPolicy.MasterNodeExcludeRules...)
	excludeRules = append(excludeRules, instance.Spec.KubernetesCisPolicy.WorkerNodeExcludeRules...)

	instance.Status.CisPolicyStatus[ClusterName].Risk.CumulativeRiskScore = 0
	instance.Status.CisPolicyStatus[ClusterName].NodeStatus = make(map[string]string)

	//for each rule, verify results
	for nodeName, resMap := range resultMap {
		instance.Status.CisPolicyStatus[ClusterName].NodeStatus[nodeName] = "Compliant"
		for control, result := range resMap {
			if result == "[FAIL]" {
				exclude := false
				for i := range excludeRules {
					splitted := strings.SplitN(excludeRules[i], " ", 2)

					if len(splitted) < 2 {
						return fmt.Errorf("invalide rule definition, missing information -> %v", splitted)
					}
					if validateRuleNumber(splitted[0]) {
						if splitted[0] == control {
							exclude = true
							break
						}
					} else {
						return fmt.Errorf("invalide test definition, the test number is not valid -> %v", splitted[1])
					}
				}
				if !exclude {
					if val, ok := CISRulesScore[control]; ok {
						if instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore < val {
							instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore = val
						}
						instance.Status.CisPolicyStatus[ClusterName].Risk.CumulativeRiskScore += val
					} else {
						log.Printf("missing score: %v\n", control)
					}
					instance.Status.CisPolicyStatus[ClusterName].NodeStatus[nodeName] = "NonCompliant"
					break
				}
			}
			if instance.Status.CisPolicyStatus[ClusterName].NodeStatus[nodeName] == "NonCompliant" {
				break
			}
		}
	}

	for _, status := range instance.Status.CisPolicyStatus[ClusterName].NodeStatus {
		if status == "NonCompliant" {
			compliant = false
			break
		}
	}
	if !compliant {
		instance.Status.CisPolicyStatus[ClusterName].Compliancy = cisv1alpha1.NonCompliant
	} else {
		instance.Status.CisPolicyStatus[ClusterName].Compliancy = cisv1alpha1.Compliant
		instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore = 0.0
		instance.Status.CisPolicyStatus[ClusterName].Risk.CumulativeRiskScore = 0.0
		instance.Status.CisPolicyStatus[ClusterName].Risk.RiskCategory = "None"
	}

	return nil

}

//UpdateCisCompliancy updates the compliacne status of a CIS policy
func (cosCtl *CosController) UpdateCisCompliancy(instance *cisv1alpha1.CisPolicy) (updatedInstance *cisv1alpha1.CisPolicy) {

	resultMap, err := cosCtl.getK8sCisResults(instance)
	if err != nil {
		log.Fatal("UpdateCisCompliancy error ", err)
	}

	// we have the file, we need to call a function that compares the results to the tests specified in the compliance
	err = compareCisResults(instance, resultMap)
	if err != nil {
		log.Printf("Warning, error occured processing the instance status, %v\n", err)
	}

	return instance
}

//=================================================================
//validateRuleNumber makes sure we have the #.#.## pattern
func validateRuleNumber(s string) bool {
	//re := regexp.MustCompile(`^([1-9].[1-9].[0-9]{1,2})$`)
	re := regexp.MustCompile(`^([1-9].[1-9].[0-9]{1,2})$|^([1-9].[0-9]{1,2})$`)
	return re.MatchString(s)
}

//=================================================================
//populateCISMasterRules adds all the master CIS K8s rules to map
func populateCISMasterRules(m map[string]string) {

	m["1.1.1"] = "Ensure that the --anonymous-auth argument is set to false"
	m["1.1.2"] = "Ensure that the --basic-auth-file argument is not set"
	m["1.1.3"] = "Ensure that the --insecure-allow-any-token argument is not set"
	m["1.1.4"] = "Ensure that the --kubelet-https argument is set to true"
	m["1.1.5"] = "Ensure that the --insecure-bind-address argument is not set"
	m["1.1.6"] = "Ensure that the --insecure-port argument is set to 0"
	m["1.1.7"] = "Ensure that the --secure-port argument is not set to 0"
	m["1.1.8"] = "Ensure that the --profiling argument is set to false"
	m["1.1.9"] = "Ensure that the --repair-malformed-updates argument is set to false"
	m["1.1.10"] = "Ensure that the admission control plugin AlwaysAdmit is not set"
	m["1.1.11"] = "Ensure that the admission control plugin AlwaysPullImages is set"
	m["1.1.12"] = "Ensure that the admission control plugin DenyEscalatingExec is set"
	m["1.1.13"] = "Ensure that the admission control plugin SecurityContextDeny is set"
	m["1.1.14"] = "Ensure that the admission control plugin NamespaceLifecycle is set"
	m["1.1.15"] = "Ensure that the --audit-log-path argument is set as appropriate"
	m["1.1.16"] = "Ensure that the --audit-log-maxage argument is set to 30 or as appropriate"
	m["1.1.17"] = "Ensure that the --audit-log-maxbackup argument is set to 10 or as appropriate"
	m["1.1.18"] = "Ensure that the --audit-log-maxsize argument is set to 100 or as appropriate"
	m["1.1.19"] = "Ensure that the --authorization-mode argument is not set to AlwaysAllow"
	m["1.1.20"] = "Ensure that the --token-auth-file parameter is not set"
	m["1.1.21"] = "Ensure that the --kubelet-certificate-authority argument is set as appropriate"
	m["1.1.22"] = "Ensure that the --kubelet-client-certificate and --kubelet-client-key arguments are set as appropriate"
	m["1.1.23"] = "Ensure that the --service-account-lookup argument is set to true"
	m["1.1.24"] = "Ensure that the admission control plugin PodSecurityPolicy is set"
	m["1.1.25"] = "Ensure that the --service-account-key-file argument is set as appropriate"
	m["1.1.26"] = "Ensure that the --etcd-certfile and --etcd-keyfile arguments are set as appropriate"
	m["1.1.27"] = "Ensure that the admission control plugin ServiceAccount is set"
	m["1.1.28"] = "Ensure that the --tls-cert-file and --tls-private-key-file arguments are set as appropriate"
	m["1.1.29"] = "Ensure that the --client-ca-file argument is set as appropriate"
	m["1.1.30"] = "Ensure that the API Server only makes use of Strong Cryptographic Ciphers "
	m["1.1.31"] = "Ensure that the --etcd-cafile argument is set as appropriate"
	m["1.1.32"] = "Ensure that the --authorization-mode argument is set to Node"
	m["1.1.33"] = "Ensure that the admission control plugin NodeRestriction is set"
	m["1.1.34"] = "Ensure that the --experimental-encryption-provider-config argument is set as appropriate"
	m["1.1.35"] = "Ensure that the encryption provider is set to aescbc"
	m["1.1.36"] = "Ensure that the admission control plugin EventRateLimit is set"
	m["1.1.37"] = "Ensure that the AdvancedAuditing argument is not set to false"
	m["1.1.38"] = "Ensure that the --request-timeout argument is set as appropriate"
	m["1.1.39"] = "Ensure that the API Server only makes use of Strong Cryptographic Ciphers"
	m["1.2.1"] = "Ensure that the --profiling argument is set to false"
	m["1.2.2"] = "Ensure that the --address argument is set to 127.0.0.1"
	m["1.3.1"] = "Ensure that the --terminated-pod-gc-threshold argument is set as appropriate"
	m["1.3.2"] = "Ensure that the --profiling argument is set to false"
	m["1.3.3"] = "Ensure that the --use-service-account-credentials argument is set to true"
	m["1.3.4"] = "Ensure that the --service-account-private-key-file argument is set as appropriate"
	m["1.3.5"] = "Ensure that the --root-ca-file argument is set as appropriate"
	m["1.3.6"] = "Ensure that the RotateKubeletServerCertificate argument is set to true"
	m["1.3.7"] = "Ensure that the --address argument is set to 127.0.0.1"
	m["1.4.1"] = "Ensure that the API server pod specification file permissions are set to 644 or more restrictive"
	m["1.4.2"] = "Ensure that the API server pod specification file ownership is set to root:root"
	m["1.4.3"] = "Ensure that the controller manager pod specification file permissions are set to 644 or more restrictive"
	m["1.4.4"] = "Ensure that the controller manager pod specification file ownership is set to root:root"
	m["1.4.5"] = "Ensure that the scheduler pod specification file permissions are set to 644 or more restrictive"
	m["1.4.6"] = "Ensure that the scheduler pod specification file ownership is set to root:root"
	m["1.4.7"] = "Ensure that the etcd pod specification file permissions are set to 644 or more restrictive"
	m["1.4.8"] = "Ensure that the etcd pod specification file ownership is set to root:root"
	m["1.4.9"] = "Ensure that the Container Network Interface file permissions are set to 644 or more restrictive "
	m["1.4.10"] = "Ensure that the Container Network Interface file ownership is set to root:root "
	m["1.4.11"] = "Ensure that the etcd data directory permissions are set to 700 or more restrictive"
	m["1.4.12"] = "Ensure that the etcd data directory ownership is set to etcd:etcd"
	m["1.4.13"] = "Ensure that the admin.conf file permissions are set to 644 or more restrictive"
	m["1.4.14"] = "Ensure that the admin.conf file ownership is set to root:root"
	m["1.4.15"] = "Ensure that the scheduler.conf file permissions are set to 644 or more restrictive"
	m["1.4.16"] = "Ensure that the scheduler.conf file ownership is set to root:root"
	m["1.4.17"] = "Ensure that the controller-manager.conf file permissions are set to 644 or more restrictive"
	m["1.4.18"] = "Ensure that the controller-manager.conf file ownership is set to root:root"
	m["1.5.1"] = "Ensure that the --cert-file and --key-file arguments are set as appropriate"
	m["1.5.2"] = "Ensure that the --client-cert-auth argument is set to true"
	m["1.5.3"] = "Ensure that the --auto-tls argument is not set to true"
	m["1.5.4"] = "Ensure that the --peer-cert-file and --peer-key-file arguments are set as appropriate"
	m["1.5.5"] = "Ensure that the --peer-client-cert-auth argument is set to true"
	m["1.5.6"] = "Ensure that the --peer-auto-tls argument is not set to true"
	m["1.5.7"] = "Ensure that a unique Certificate Authority is used for etcd"
	m["1.6.1"] = "Ensure that the cluster-admin role is only used where required"
	m["1.6.2"] = "Create administrative boundaries between resources using namespaces"
	m["1.6.3"] = "Create network segmentation using Network Policies"
	m["1.6.4"] = "Ensure that the seccomp profile is set to docker/default in your pod definitions"
	m["1.6.5"] = "Apply Security Context to Your Pods and Containers"
	m["1.6.6"] = "Configure Image Provenance using ImagePolicyWebhook admission controller"
	m["1.6.7"] = "Configure Network policies as appropriate"
	m["1.6.8"] = "Place compensating controls in the form of PSP and RBAC for privileged containers usage"
	m["1.7.1"] = "Do not admit privileged containers"
	m["1.7.2"] = "Do not admit containers wishing to share the host process ID namespace"
	m["1.7.3"] = "Do not admit containers wishing to share the host IPC namespace"
	m["1.7.4"] = "Do not admit containers wishing to share the host network namespace"
	m["1.7.5"] = "Do not admit containers with allowPrivilegeEscalation"
	m["1.7.6"] = "Do not admit root containers"
	m["1.7.7"] = "Do not admit containers with dangerous capabilities"
}

//=================================================================
//populateCISMasterRules adds all the master CIS K8s rules to map
func populateCISMasterRulesScore(m map[string]float32) {
	m["1.1.1"] = 6.4
	m["1.1.5"] = 6.4
	m["1.1.6"] = 6.4
	m["1.1.9"] = 7.4
	m["1.1.10"] = 6.8
	m["1.1.11"] = 3.3
	m["1.1.24"] = 6.4
	m["1.4.1"] = 4.6
	m["1.5.3"] = 6.4
	m["1.5.5"] = 6.4
}
