package set

import (
	"testing"

	"github.com/fvbock/uds-go/set"
)

func TestStringSetAdd(t *testing.T) {

	s1 := set.NewStringSet()
	t.Log(s1)
	s1.Add("A")
	t.Log(s1)

	s2 := set.NewStringSet("B", "C")
	t.Log(s2)

	s3 := set.NewStringSet([]string{"B", "D", "B"}...)
	t.Log(s3)
	s3.Remove("A")
	t.Log(s3)
	s3.Remove("B")
	t.Log(s3)

	s4 := set.NewStringSet([]string{"A", "B", "C", "D"}...)
	t.Log(s4)
	s4.Remove("B")
	t.Log(s4)
	s4.Remove("D")
	t.Log(s4)
	s4.Remove("A")
	t.Log(s4)
	s4.Remove("C")
	t.Log(s4)

	s5 := s1.Union(s2)
	t.Log("S5", s5)

	s6 := s2.Union(s1)
	t.Log("S6", s6)

	t.Log("S5 == S6", s5.IsEqual(s6))
	t.Log("S6 == S5", s6.IsEqual(s5))

	sA := set.NewStringSet([]string{"B", "宜しくお願い致します"}...)
	sB := set.NewStringSet([]string{"A", "B", "C", "宜しくお願い致します"}...)
	s7 := sA.Intersect(sB)
	t.Log("INTERSECT", sA, sB)
	t.Log("S7", s7)

	sC := set.NewStringSet([]string{}...)
	sD := set.NewStringSet([]string{}...)
	s8 := sC.Intersect(sD)
	t.Log("INTERSECT", sC, sD)
	t.Log("S8", s8)

	sE := set.NewStringSet([]string{"B", "D"}...)
	sF := set.NewStringSet([]string{"A", "C", "F", "G"}...)
	s9 := sE.Intersect(sF)
	t.Log("INTERSECT", sE, sF)
	t.Log("S9", s9)

	s10 := sF.Intersect(sE)
	t.Log("INTERSECT", sF, sE)
	t.Log("S10", s10)

	sG := set.NewStringSet([]string{"4fd78138558fbe76ff000028517f75a53c581635b5003d15", "4fed20ab558fbe15cf002aa9512e34cd6c35827b490000f4", "4fed20ab558fbe15cf002aa95142e3293c58162cd50143f9", "4fd78138558fbe76ff0000284fe97bad558fbe78c9000d4a", "4fed20ab558fbe15cf002aa9518c85116c358214b6000069", "4ffa7c41558fbe37d2000d6b51885ef03c58160aab00702d", "5057e89a558fbe2b9a003e2550f5ec203c58167572088ec4", "519cc1333c58167ae8005ccc519f05ba3c581619ea008285", "4fe362e1558fbe417d00017d502db547558fbe221700008d", "5031ed07558fbe0d94001cf050ebb3473c58167572036ebb", "4fed20ab558fbe15cf002aa9516da13f6c35820f3500054b", "4fe2aac8558fbe33060005b95182a0b73c58160d58000478", "4fe362e1558fbe417d00017d50480b18558fbe43ca00194e", "4fe362e1558fbe417d00017d516b6f173c58165772000355", "4fe2aac8558fbe33060005b95109b2663c58166bd70024e7", "4fe362e1558fbe417d00017d5037315e558fbe162d000ed2", "4fed20ab558fbe15cf002aa9515e984a6c35827e59000396", "4fd78138558fbe76ff00002851a2d2d83c58164d74003688", "4fd78138558fbe76ff00002850260f9b558fbe5af700aa5d", "4fe362e1558fbe417d00017d50ea3a573c5816757202b336", "4fed20ab558fbe15cf002aa9502e0e6b558fbe2ff6000629", "4fed20ab558fbe15cf002aa951876f1e6c3582476c0003b5", "4fd78138558fbe76ff0000284fffe4a2558fbe707d00562a", "4febe6a0558fbe0b6d00168a5125d6273c58161f330060c2", "4fe362e1558fbe417d00017d503ae357558fbe551d00316f", "4febe6a0558fbe0b6d00168a50ee10983c581675720461a5", "4fe2aac8558fbe33060005b9515cc9c93c58164737000f50", "5031d03a558fbe0bd8000169505bde38558fbe3bfe0042e1", "4fed20ab558fbe15cf002aa95111ac983c58166bd700cd60", "4fe362e1558fbe417d00017d50f378af3c5816757206712a", "4fe362e1558fbe417d00017d5174ca303c58162567005dea", "4fe362e1558fbe417d00017d5035a68d558fbe5dfd008229", "4fd78138558fbe76ff00002851234ef46c3582379f000313", "4fe362e1558fbe417d00017d502c6a5c558fbe676400258e", "4fd78138558fbe76ff000028513014db3c581625e600600c", "4fd78138558fbe76ff0000285115c355558fbe165c000001", "4fed20ab558fbe15cf002aa9519ef8816c35821ba600015a", "4fed20ab558fbe15cf002aa9504b0e20558fbe04a80015f6", "4fed20ab558fbe15cf002aa95084bc763c58164ab1007615", "4fed20ab558fbe15cf002aa9512c980f6c35823a0800010e", "4fe362e1558fbe417d00017d5162307c3c581675e4001401", "4fe362e1558fbe417d00017d51622aaa3c581675e4000bf6", "4fe362e1558fbe417d00017d502b0da5558fbe2871004158", "4fd78138558fbe76ff000028506a6a56558fbe228e00f617", "4fe2aac8558fbe33060005b9518b71b83c58160aab019a54", "4fe2c446558fbe330600240d5140a2e53c58162cd5010d14", "4fd78138558fbe76ff000028516feaa43c581619150045d7", "519cc1333c58167ae8005ccc519d6f8f3c5816394900140c", "4fe2aac8558fbe33060005b9515bdc913c581663450050a3", "4ffa7c41558fbe37d2000d6b516cb26a3c58161bf70006ac", "4fed20ab558fbe15cf002aa9510b7a533c58166bd7006d35", "4ffa7c41558fbe37d2000d6b506a488a558fbe228e00c22b", "4fe81cee558fbe6a4800019c508e0c973c5816594b0031b8", "4fd78138558fbe76ff00002850933db43c58164408003794", "4fed20ab558fbe15cf002aa95180b7606c35827ef4000123", "4fd78138558fbe76ff00002850825b6e3c58164a95003d18", "4febe6a0558fbe0b6d00168a50349f82558fbe5dfd0038e1", "4fd78138558fbe76ff00002850decf3a3c5816757200ebe3", "4fe362e1558fbe417d00017d50cee4013c5816409a00e4fa", "4fed20ab558fbe15cf002aa9518b43816c358276d00005af", "4fe81cee558fbe6a4800019c509752f93c58162a610085fe", "4fe362e1558fbe417d00017d510b2b183c58166bd70059de", "4fe362e1558fbe417d00017d5029b7f4558fbe05600023e3", "4fed20ab558fbe15cf002aa9519c97516c358210d3000091", "4fed20ab558fbe15cf002aa950d131e03c58164f7f00a6ed", "4fd7814f558fbe76ff0000385195cbd63c58161710000815", "4fd78138558fbe76ff00002850b6e0b83c5816206800509c", "519cc1333c58167ae8005ccc51a5634e3c58164a9e001855", "4fe362e1558fbe417d00017d5031aa46558fbe6b5e002763", "4fd78138558fbe76ff000028511f122c558fbe1aea00012b", "4ffa7c41558fbe37d2000d6b50693da5558fbe228e008381", "4fed20ab558fbe15cf002aa951137a403c58166bd700fb2c", "4febe6a0558fbe0b6d00168a51828c8e3c58160d580001b6", "4fed20ab558fbe15cf002aa9516b70676c358254c8000c95", "4fed20ab558fbe15cf002aa951a2bdde3c58164d740017ff", "4fd78138558fbe76ff000028518225ac3c581635b50133ac", "4fe2aac8558fbe33060005b95167c32c3c58160995004515", "4fe362e1558fbe417d00017d50495a41558fbe6f92000a5d", "4febe6a0558fbe0b6d00168a4ffa33f5558fbe1a31002db1", "4fd78138558fbe76ff00002851130a053c58166bd700ea84", "4febe6a0558fbe0b6d00168a505fca45558fbe3bfe00bd37", "4febe6a0558fbe0b6d00168a516e133a3c581662da001043", "5031ed07558fbe0d94001cf0518c93803c58166f3600337a", "4fed20ab558fbe15cf002aa9502c7e8c558fbe6764003ca7", "4fe362e1558fbe417d00017d5029b5b1558fbe0560002251", "5031d03a558fbe0bd80001695175f8663c5816256700aa54", "4fd7814f558fbe76ff00003850f629573c58167572091212", "519cc1333c58167ae8005ccc519c2a643c58167ae80001d3", "519cc1333c58167ae8005ccc51b18dab3c58167be301270c", "4fed20ab558fbe15cf002aa9519c2aba6c35826d1e0000f3", "4fed20ab558fbe15cf002aa9519445a76c358209c700033c", "4fe362e1558fbe417d00017d50d2a5483c58166252004519", "4fd78138558fbe76ff0000284ff4a0bb558fbe77210007b4", "4fd78138558fbe76ff00002850f0bc633c5816757205f599", "4fe362e1558fbe417d00017d50345634558fbe5a900005d5", "4fe362e1558fbe417d00017d514ed8ad3c58162162001c59", "4fe2aac8558fbe33060005b950b337443c58167445011010", "4fed20ab558fbe15cf002aa950ca4de33c581667af00b781", "4fed20ab558fbe15cf002aa95191d8536c35824f6600002c", "4fe362e1558fbe417d00017d5031a0ea558fbe6b5e001a96", "4fd78138558fbe76ff00002850c67e713c58164503013ff0", "4febe6a0558fbe0b6d00168a5010fc51558fbe19c70017f7", "4fd78138558fbe76ff000028516f9f4c3c58161915002cc2"}...)

	sH := set.NewStringSet([]string{"4fd78138558fbe76ff000028516cf2b53c58161bf7002963", "4fd78138558fbe76ff00002850aad9cf3c581602d50055e8", "5031d03a558fbe0bd8000169507b4a9f558fbe78b6000292", "4fd78138558fbe76ff0000284fe97bad558fbe78c9000d4a", "4fd78138558fbe76ff000028510f2c1c3c58166bd7008fd5", "4fd78138558fbe76ff00002850867f473c58164ab102a81b", "4fd78138558fbe76ff0000284ff3ce43558fbe680d000723", "4fd78138558fbe76ff000028505c1d9c558fbe3bfe00651b", "4fd78138558fbe76ff000028512ca4833c58161f3300c2f7", "4fed20ab558fbe15cf002aa950f4e5a43c5816757207c131", "4fd78138558fbe76ff00002850f8e9f83c581675720c3a6e", "4fd78138558fbe76ff00002851a2ef523c58164d74005584", "4fe2c446558fbe330600240d5019d661558fbe4f56002268", "4fd78138558fbe76ff0000285084b5dc3c58164ab1005fb8", "4fd78138558fbe76ff000028514bd6123c5816653500122d", "4fe2aac8558fbe33060005b950a39fbb3c58163aea00c611", "4fd78138558fbe76ff000028514273003c58162cd501310e", "5125d2d33c58161f33005fe65119ea233c58166bd7013f3c", "4fd78138558fbe76ff00002851942fa53c581620bb000d44", "4febe6a0558fbe0b6d00168a5148897b3c58162cd501a3da", "4fd78138558fbe76ff0000284fe91511558fbe6fc80004d4", "4fd78138558fbe76ff0000285195cded3c58162313000051", "519090753c5816386900023851a1876d3c581619ea00cac0", "4ff63b9f558fbe0bfb001ca851244a483c58161f33002cad", "4fd78138558fbe76ff000028515bc0603c581663450037a6", "4fd78138558fbe76ff000028510a02ad3c58166bd7003e5e", "4fe362e1558fbe417d00017d51622aaa3c581675e4000bf6", "4ffa7c41558fbe37d2000d6b519b06d03c58160cd00027da", "4fd78138558fbe76ff000028511f10fc558fbe1aea00008e", "5125d2d33c58161f33005fe6513182903c58162cd500144b", "5125d2d33c58161f33005fe6513062973c58162cd500091a", "4fd78138558fbe76ff00002850da63fd3c5816757200256a", "4fe2b375558fbe33060009d6513059ed3c58162cd500078f", "4fd78138558fbe76ff000028507de96f558fbe5633001555", "5031d03a558fbe0bd800016951b305153c58167104000bad", "4fe362e1558fbe417d00017d5164e1f73c581675e400be0b", "4fd78138558fbe76ff00002850ea945e3c58167572030199", "500ca86c558fbe1dc30000195093b2043c5816440800a3da", "4fe362e1558fbe417d00017d504d6ad0558fbe04a80057b9", "4fe362e1558fbe417d00017d5110879b3c58166bd700b804", "4fd78138558fbe76ff00002851ba9edf3c581661c9001915", "4fed20ab558fbe15cf002aa95127f0976c358257d7000021", "519090753c5816386900023851be7e9a3c5816147e004e97", "4fd78138558fbe76ff00002850ea46fe3c5816757202b844", "4fd78138558fbe76ff00002850ca7bad3c58162414000190", "4fd78138558fbe76ff00002851523be23c581648ab000235", "4febe6a0558fbe0b6d00168a5137ef3a6c358263c2000024", "4fd78138558fbe76ff0000285186faef3c58160aab0005b3", "4fd78138558fbe76ff00002850ee3b733c5816757204824d", "4fd78138558fbe76ff00002851383fb53c58162cd5008512", "4fe362e1558fbe417d00017d51a2d0773c58164d74003395", "4fd78138558fbe76ff00002851087e563c58166bd7000676", "4fd78138558fbe76ff000028504553f8558fbe6f9f00343d", "4fd78138558fbe76ff00002850d044803c58164f7f005486", "4fe362e1558fbe417d00017d50779d5e558fbe0f8b0024bf", "4fd78138558fbe76ff00002851666f803c581675e4012cda", "4fd78138558fbe76ff000028517a02073c581655a7001e66", "4fd78138558fbe76ff00002851ac04963c58165a37001e6e", "4fd78138558fbe76ff000028515bac0f3c581663450031f7", "4fd78138558fbe76ff000028517a3b1b3c581655a7003e08", "519090d13c581638690002d951a210ae3c581619ea00d829", "4fd78138558fbe76ff00002850438128558fbe580900013e", "509b7ea8558fbe375900077d50b6c7e7558fbe112f000184", "4fd78138558fbe76ff0000285085fc1d3c58164ab101ca7d", "4fd78138558fbe76ff000028519c33f13c58167ae8000a13", "4fe2c446558fbe330600240d50e7b5e13c58167572027432", "4febe6a0558fbe0b6d00168a5130193e3c581625e600612c", "4fd78138558fbe76ff0000285118658e3c58166bd7013193", "4fe2c446558fbe330600240d502cb43a558fbe7fd5003ea4", "4fd78138558fbe76ff000028515bc4673c58166345003ddd", "4fe362e1558fbe417d00017d50650c5d558fbe71600001a2", "4febe6a0558fbe0b6d00168a4ffe421a558fbe5911000a63", "4fd78138558fbe76ff0000285004f780558fbe1f57000160", "4fd78138558fbe76ff000028505a8aef558fbe0ef30053da", "4fe362e1558fbe417d00017d51872a3d3c58160aab002650", "4fd78138558fbe76ff00002850260f9b558fbe5af700aa5d", "4fd78138558fbe76ff000028519444733c581678ac00059a", "4fd78138558fbe76ff0000284fffe4a2558fbe707d00562a", "4fe2aac8558fbe33060005b95088a1a33c581679fe0051a0", "4fd78138558fbe76ff000028508b473e3c58165f1a000757", "4fe362e1558fbe417d00017d50a05fc73c58162aa00187f5", "519090d13c581638690002d951b84c5e3c5816364000a68f", "4fd78138558fbe76ff0000285162b21b3c581675e4004c1a", "509b7ea8558fbe375900077d510a55cc558fbe1ba7000b39", "4fed20ab558fbe15cf002aa95111ac983c58166bd700cd60", "4fd78138558fbe76ff0000285013750a558fbe3307003181", "4fe362e1558fbe417d00017d50f378af3c5816757206712a", "4fd78138558fbe76ff000028519da1e63c581639490043e7", "4fd78138558fbe76ff000028518c3dd13c58166f360010fe", "4fd78138558fbe76ff00002850052c94558fbe39c500063e", "5057e89a558fbe2b9a003e2550f4f9183c5816757207e0f2", "4fd78138558fbe76ff00002850a4f4183c5816114d000796", "4fe362e1558fbe417d00017d516374023c581675e4005ebc", "4fd78138558fbe76ff00002850d508713c5816625201241e", "500ca86c558fbe1dc300001951b6a0f73c5816364000478d", "4fd78138558fbe76ff0000285075c0cd558fbe63d7000fdf", "4fd78138558fbe76ff0000285087ae4d3c58160e3f01f211", "4fd78138558fbe76ff0000285057ea6e558fbe2b9a004614", "4fe2b375558fbe33060009d6510228bb3c58163cf50057d1", "4fe2c446558fbe330600240d501f30df558fbe6953005cfd", "4fd78138558fbe76ff0000285046f5cc558fbe31270013c0", "509b7ea8558fbe375900077d50ffa6ea558fbe47700000ae", "4fd78138558fbe76ff00002851150359558fbe07e20001c8", "4fd78138558fbe76ff000028519c778a3c58167ae800334c", "4fd78138558fbe76ff00002850a4e1883c58163aea01eb9a", "4fd78138558fbe76ff000028513161223c58162cd5001333", "4fe362e1558fbe417d00017d5008be39558fbe293000317d", "4fd78138558fbe76ff000028504d754e558fbe45ef0000c0", "4fd78138558fbe76ff00002850a65e353c5816183a001d41", "4fd78138558fbe76ff0000285137f01e6c358263c2000059", "4fd78138558fbe76ff000028501dddda558fbe69530048b6", "4fd78138558fbe76ff000028516b957f3c5816147c0004b1", "4fd78138558fbe76ff00002850e39b763c58167572016f38", "4fd78138558fbe76ff000028509b741c3c58161d1d009f9a", "4fd78138558fbe76ff000028507f51cb558fbe48790021bf", "4fe2c446558fbe330600240d500cbfce558fbe28f3000167", "4fd78138558fbe76ff000028515bc29a3c58166345003a2a", "4fd78138558fbe76ff00002850484f46558fbe5aa900020d", "4fd78138558fbe76ff0000285150f0473c58163ed50006a1", "500ca86c558fbe1dc300001950fe84483c58165f52008074", "4fe2b375558fbe33060009d651259d4c3c58161f33005503", "4fd78138558fbe76ff0000285154df1e3c581632be00490a", "4fd78138558fbe76ff000028502cc535558fbe09e40015d3", "4fd78138558fbe76ff0000284febb65f558fbe0b6d00054c", "4fd78138558fbe76ff00002850ed0d5b3c581675720418d3", "4fd78138558fbe76ff00002851b66d293c58163640003fce", "4fe2aac8558fbe33060005b951079a7b3c58166e2900244b", "4fd78138558fbe76ff0000284ff3f73d558fbe6c8c000696", "4fe2c446558fbe330600240d4fe2c478558fbe330600248a", "4fd78138558fbe76ff00002850823d763c58164a9500344c", "4fe362e1558fbe417d00017d514ed8ad3c58162162001c59", "4fd78138558fbe76ff000028519ad6693c58165b12001ef0", "4fe362e1558fbe417d00017d5019e7e5558fbe4f56002b5a", "4fd78138558fbe76ff0000285091de2c3c58161e2900529b", "4fd78138558fbe76ff000028516f9f4c3c58161915002cc2", "4fd78138558fbe76ff000028508c9a1c3c58163dbb000965", "4fe362e1558fbe417d00017d50286671558fbe5af700c1b4", "4fd78138558fbe76ff00002850599442558fbe08cf000250", "4fd78138558fbe76ff000028514bf2b13c581665350022dc", "4fd78138558fbe76ff0000285152ae053c581648ab00390e", "4fe362e1558fbe417d00017d50ea3a573c5816757202b336", "4fd78138558fbe76ff00002851bef2673c5816147e007726", "4fe362e1558fbe417d00017d51ac11823c58165a37002b5f", "4fd78138558fbe76ff000028508a7d383c58162fb00078d7", "4fd78138558fbe76ff00002850ea5cc53c5816757202ca6c", "4fd78138558fbe76ff000028513854903c58162cd50086f9", "4fd78138558fbe76ff00002851b2a56a3c58167be3017892", "4fd78138558fbe76ff000028509bc76f558fbe3e700001cf", "4fd78138558fbe76ff00002850c6c10f3c581645030201bd", "4fd78138558fbe76ff00002850dac9683c5816757200877f", "4fe362e1558fbe417d00017d50d831d93c58166252019a32", "4fd78138558fbe76ff0000285085f3393c58164ab1019bd7", "4fd78138558fbe76ff00002850976a163c58162a6100a887", "4fd78138558fbe76ff00002850a21ec53c5816723900fe76", "4fd78138558fbe76ff000028505c02da558fbe3bfe0057bd", "4fe362e1558fbe417d00017d50121703558fbe3307000300", "4fd78138558fbe76ff00002851b160593c58167be3010a3c", "4fd78138558fbe76ff00002850d02eb13c58164f7f004720", "4fd78138558fbe76ff00002850a1e1173c58167239009ae6", "4fe362e1558fbe417d00017d50232475558fbe5af7002fab", "4fd78138558fbe76ff00002850641c96558fbe58e80009db", "4fe362e1558fbe417d00017d5021ca24558fbe35ff000fcd", "4fd78138558fbe76ff0000285090804e3c581625de001214", "4fd78138558fbe76ff0000285153ed283c581632be0025dc", "4fd78138558fbe76ff000028517921103c58161c0300e02d", "4fd78138558fbe76ff0000285062b503558fbe2cda00613e", "4fd78138558fbe76ff000028500d2cf9558fbe41ae000d6a", "4fd78138558fbe76ff00002850a5995b3c5816114d003118", "4fd78138558fbe76ff000028510638e63c5816573d002718", "4fd78138558fbe76ff000028515d01b53c581647370032c1", "4fd78138558fbe76ff000028501b4b3d558fbe6953002b61", "4fe362e1558fbe417d00017d5029bacf558fbe05600025c0", "4fe362e1558fbe417d00017d505a873d558fbe0ef30050c5", "4fd78138558fbe76ff000028508e606a3c58167cda000be2", "4fd78138558fbe76ff00002850ffe6c43c581634f10040ec", "4fd78138558fbe76ff000028511efc7c3c58166bd7017173", "4fe362e1558fbe417d00017d50495a41558fbe6f92000a5d", "4febe6a0558fbe0b6d00168a513094c26c35824fcc000010", "4fd78138558fbe76ff0000285047fd43558fbe43ca0010c5", "4fe2c446558fbe330600240d502c93ae558fbe7fd50024e9", "4fd78138558fbe76ff00002851a577653c58164a9e002ffe", "4fd78138558fbe76ff000028504efd83558fbe6aaf001342", "4fd78138558fbe76ff000028502213c5558fbe5af70009e6", "4fd78138558fbe76ff00002850526e14558fbe3b29001494", "4fd78138558fbe76ff0000285116c9183c58166bd7011f3d", "4fe2b375558fbe33060009d6513938d63c58162cd5009082", "4fd78138558fbe76ff0000285006810e558fbe67ef003138", "4fe362e1558fbe417d00017d507b8c87558fbe333e0023ae", "4fd78138558fbe76ff000028514bfb403c581665350023f2", "4fd78138558fbe76ff00002850deccab3c5816757200ea02", "4fe362e1558fbe417d00017d5003780e558fbe06fa000fd2", "4fe2c446558fbe330600240d515bd3bc3c581663450045b3", "4fd78138558fbe76ff00002850c5516f3c58164503002841", "4fd78138558fbe76ff000028519f19653c581619ea0090ee", "5125d2d33c58161f33005fe6513054b03c58162cd5000661", "4fd78138558fbe76ff0000285077d432558fbe1ada0008c7", "4fd78138558fbe76ff00002850346d1e558fbe5dfd000d52", "4fd78138558fbe76ff000028509d0da73c58162aa000c0e0", "4fe362e1558fbe417d00017d50246bc2558fbe5af7006913", "4fd78138558fbe76ff0000284ff4a0bb558fbe77210007b4", "4fd78138558fbe76ff000028511f1008558fbe1aea000001", "4fd78138558fbe76ff00002851233e5f3c58161f33001898", "4fd78138558fbe76ff000028513da4b06c358231220000fd", "4fd78138558fbe76ff00002851772ccc3c58161c030009e1", "4fd78138558fbe76ff000028519dd57c3c58163949006ffa", "4fd78138558fbe76ff000028514ffc6e3c58162162003fed", "4fd78138558fbe76ff00002850e540b83c5816757201f120", "519cc1333c58167ae8005ccc519cc2b53c58167ae8005d65", "4fd78138558fbe76ff0000285180f1f23c581635b500d74d", "4fd78138558fbe76ff0000285148f71c3c58162cd501a485", "4fd78138558fbe76ff000028515a69083c58160a9d002dd0", "4fd78138558fbe76ff0000285141742f3c58162cd5011f02", "4fd78138558fbe76ff00002850b75b7e3c581620680159b2", "4fd78138558fbe76ff00002850123841558fbe330700170f", "4fd78138558fbe76ff0000285085fbaa3c58164ab101c62a", "519090753c581638690002385199c82e3c581633c1000007", "4fd78138558fbe76ff00002851666af43c581675e4012830", "4fd78138558fbe76ff0000285191d1a33c58164b4800031c", "500ca86c558fbe1dc300001951b1768f3c58167be30114cc", "4fd78138558fbe76ff000028501112dd558fbe19c7002170", "4fd78138558fbe76ff0000285040be04558fbe7668001d7e", "4fd78138558fbe76ff00002851ad66403c581603fa002d67", "4fd78138558fbe76ff000028505dfe91558fbe3bfe0093a6", "4fd78138558fbe76ff000028508f5cba3c581656f70013b2", "500ca86c558fbe1dc300001951a2abb83c58164d740007c7", "4fd78138558fbe76ff0000284fe9dc19558fbe78c9002cd5", "4fd78138558fbe76ff00002851b34c5a3c581671040023bf", "5031d03a558fbe0bd8000169505bde38558fbe3bfe0042e1", "4febe6a0558fbe0b6d00168a50483ff6558fbe43ca003908", "4fd78138558fbe76ff0000285091db2a3c58161e290041f6", "4fe362e1558fbe417d00017d5174ca303c58162567005dea", "4fd78138558fbe76ff000028509650353c58162a610025ff", "4fd78138558fbe76ff0000285089b947558fbe3509000031", "515248ff3c581648ab0003e55164dc933c581675e400bbd4", "4fe362e1558fbe417d00017d5087790f3c58160e3f015337", "4fd78138558fbe76ff00002851075e423c58166e29000fb2", "4fd78138558fbe76ff0000285111d04d3c58166bd700d53f", "4febe6a0558fbe0b6d00168a5110a461558fbe67ee00010a", "4fd78138558fbe76ff000028507386eb558fbe19d2004d48", "4fd78138558fbe76ff00002850751443558fbe575b000778", "4fd78138558fbe76ff000028512307bb3c58161f33001243", "4fd78138558fbe76ff0000285188ead03c58160aab00c048", "4fed20ab558fbe15cf002aa95150086d3c58162162004bde", "4febe6a0558fbe0b6d00168a5010d3b5558fbe19c7000341", "500ca86c558fbe1dc3000019519090dc3c5816386900032f", "4fe2aac8558fbe33060005b95007776d558fbe77a000695f", "4fe362e1558fbe417d00017d50723ae6558fbe228e028c89", "4fd78138558fbe76ff0000285077eafc558fbe1d1c002752", "4febe6a0558fbe0b6d00168a51231cbe6c358229c40001a8", "4fd78138558fbe76ff0000285163c07d3c581675e4007bd4", "4fd78138558fbe76ff000028508932eb3c581652dd001789", "4fe2b375558fbe33060009d6519995bd3c581620d7000995", "4fe362e1558fbe417d00017d51998c543c581613fb001326", "500ca86c558fbe1dc3000019506a7489558fbe228e0103d3", "519090d13c581638690002d951a8401f3c5816345e00a962", "4fe2b375558fbe33060009d6519ec5f13c581619ea001fe7", "4fd78138558fbe76ff0000285045b928558fbe122e001256", "4fd78138558fbe76ff00002850c552243c58164503002cdd", "4fd78138558fbe76ff0000285113aa28558fbe63d5000331", "4fd78138558fbe76ff000028519c0df43c58165166000872", "4fd78138558fbe76ff00002850600166558fbe5e0f0045ed", "4fd78138558fbe76ff000028515a94863c58160a9d005509", "4fd78138558fbe76ff0000285087761c3c58160e3f014f29", "4ffa7c41558fbe37d2000d6b5162a7e23c581675e4004a63", "4fd78138558fbe76ff0000285119a9ae558fbe2d1200005c", "4fd78138558fbe76ff00002850372b95558fbe162d000d04", "4fd7814f558fbe76ff0000385106218b558fbe41a2000211", "4fd78138558fbe76ff000028504825e8558fbe43ca002c12", "4fd78138558fbe76ff000028513845f56c358263c20003fa", "4fd78138558fbe76ff000028517d1fad3c581655a700b72a", "4fe2c446558fbe330600240d50569765558fbe3b290038ff", "5125d2d33c58161f33005fe65127196f3c58161f330077e2", "4fd78138558fbe76ff00002850374d5a558fbe162d002243", "4fd78138558fbe76ff000028503dcfbd558fbe3c25000d27", "4ffa7c41558fbe37d2000d6b519a0d3f3c581644fb00016d", "4fe362e1558fbe417d00017d50fd198d3c581675720e6371", "4fd78138558fbe76ff00002851b913123c5816364000ae2d", "4fd78138558fbe76ff000028514a92cd3c58167756000036", "4fd78138558fbe76ff0000285162b6993c581675e4004d1d", "4fd78138558fbe76ff0000285024b557558fbe5af7008ca9", "4fe2b375558fbe33060009d6512d62423c58161f3300c94d", "4fd78138558fbe76ff0000285183301a3c58160d5800219d", "4fd78138558fbe76ff0000285080ddf7558fbe2158001dd9", "4fd78138558fbe76ff000028507b682f558fbe78b6000cf4", "4fd78138558fbe76ff0000284ffa89d7558fbe3a0b000206", "4fd78138558fbe76ff0000285137f0973c58162cd50060c3", "4fe362e1558fbe417d00017d516b6f173c58165772000355", "4fe2b375558fbe33060009d651301b833c581625e6006202", "4fd78138558fbe76ff0000285003e39d558fbe0d6a002d3c", "4fed20ab558fbe15cf002aa9506289ec558fbe2cda00244f", "4fd78138558fbe76ff00002850a0bfa33c58162aa0026af9", "4fe2aac8558fbe33060005b94ffa3e25558fbe1a310034e4", "4fd78138558fbe76ff0000285085e80e3c58164ab10190b8", "519090753c5816386900023851c1297c3c5816147e00be03", "4fd78138558fbe76ff00002850402d2a558fbe7582000723", "4fd78138558fbe76ff0000285074fd6b558fbe50ec003816", "519cc1333c58167ae8005ccc519d6f8f3c5816394900140c", "519cc1333c58167ae8005ccc51a59afc3c58164a9e004400", "5125d2d33c58161f33005fe6515256f33c581648ab000713", "4fd78138558fbe76ff00002851496afe3c58162cd501adf0", "4fd78138558fbe76ff00002850825b6e3c58164a95003d18", "4ff63b9f558fbe0bfb001ca851538e563c58167bbf0003c2", "4febe6a0558fbe0b6d00168a500f4f0d558fbe6bcd001027", "4fd78138558fbe76ff0000285034c90f558fbe5dfd004adb", "4fd78138558fbe76ff000028513ea1793c58162cd500de0b", "4fd78138558fbe76ff00002850c12fc53c58162bd700fa03", "4fd78138558fbe76ff0000285163c6c93c581675e4007e97", "4fd78138558fbe76ff00002850d084fc3c58164f7f006e28", "4fed20ab558fbe15cf002aa95163e0c76c35821596000173", "4fd78138558fbe76ff0000285154f23d3c581632be004bae", "4fd78138558fbe76ff00002850ded1593c5816757200ec42", "4fd78138558fbe76ff000028502a1b16558fbe1d60000e27", "519cc1333c58167ae8005ccc51bb0d173c5816147e002947", "4fd78138558fbe76ff000028519ef87c3c581619ea006944", "4fd78138558fbe76ff000028508c9ea43c58163dbb000bdf", "4fd78138558fbe76ff00002851b9627a3c5816364000d82f", "4fe362e1558fbe417d00017d502b02f1558fbe2871002ba4", "4fd78138558fbe76ff000028514bcc903c58166535000d1b", "4fe2aac8558fbe33060005b951104ce23c58166bd700ad2b", "519cc1333c58167ae8005ccc519c2a643c58167ae80001d3", "4fd78138558fbe76ff00002851abecf33c58165a37000b4f", "4fd78138558fbe76ff000028519b0c533c58160cd0003270", "4fe362e1558fbe417d00017d50593448558fbe2b9a00c539", "509b7ea8558fbe375900077d50a4bedb558fbe0eb60002e8", "4fd78138558fbe76ff0000285029c0bd558fbe0560002e6f", "4fd78138558fbe76ff00002851274e503c58161f33007e09", "4fe2c446558fbe330600240d504da703558fbe45ef003b70", "4fd78138558fbe76ff000028507e2489558fbe24e0000658", "4fd78138558fbe76ff000028515d4f373c5816473700527d", "4fe362e1558fbe417d00017d507290d4558fbe12da001c27", "4fe2aac8558fbe33060005b950b748fc3c58162068015642", "4fd78138558fbe76ff000028503b2bcb558fbe70ee0000b9", "4fe362e1558fbe417d00017d51be8c723c5816147e00552c", "4fe362e1558fbe417d00017d51b48cf03c5816243e001167", "509b7ea8558fbe375900077d50a46976558fbe0b8a0002e0", "4fd78138558fbe76ff000028505b00a8558fbe3bfe001073", "4fd78138558fbe76ff00002850aa120e3c581602d500203d", "4fd78138558fbe76ff000028503200be558fbe0d940044fb", "4fd78138558fbe76ff0000285063e63b558fbe3ebb006efc", "4fd78138558fbe76ff00002851824eee3c581635b5014c09", "4fd78138558fbe76ff0000285190a6ae3c58165ef8000575", "4fd78138558fbe76ff00002850e105ca3c5816757201316b", "4fd78138558fbe76ff00002851061fb03c5816573d001b76", "4fd78138558fbe76ff0000284ffa695e558fbe37d20000fb", "50ea77d03c5816757202e7f451a2e5a73c58164d74004e8a", "4fd78138558fbe76ff000028514bdb6f3c58166535001770", "4fd78138558fbe76ff00002850decaf23c5816757200e9f2", "4fd78138558fbe76ff0000284febadeb558fbe0b6d00031c", "5093b1013c5816440800a2d7518b4ebb3c58160aab018b71", "4fd78138558fbe76ff00002850978d813c58162a6100c5f0", "4fd78138558fbe76ff000028517729643c58161c0300066a", "519090753c581638690002385190adf13c58165ef8000940", "4fe362e1558fbe417d00017d5109d70b3c58166bd7002eae", "4fd78138558fbe76ff00002850b869a03c58162068025158", "4fd78138558fbe76ff000028512d52b73c58161f3300c774", "4fd78138558fbe76ff000028503e8d60558fbe3c2500333e", "4fd78138558fbe76ff000028505a9f8f558fbe0ef30069e0", "4fd78138558fbe76ff0000285102de453c58163cf5006288", "4fd78138558fbe76ff00002850751697558fbe575b001331", "4fe2b375558fbe33060009d650690cd4558fbe228e004fe5", "4fd78138558fbe76ff000028513014db3c581625e600600c", "4fd78138558fbe76ff00002850bc68c73c58162068040793", "4fd78138558fbe76ff000028516367483c581675e4005a90", "4fd78138558fbe76ff00002850471c92558fbe399400026a", "4fd78138558fbe76ff0000285050a389558fbe0a3e002fdc", "4fd78138558fbe76ff00002850e83a693c5816757202820f", "4fe362e1558fbe417d00017d5107a5813c58166e29004c99", "4fd78138558fbe76ff00002850fcba923c581675720d6244", "4fd78138558fbe76ff0000285008ad53558fbe29300011f1", "4fd78138558fbe76ff000028513ec7c33c58162cd500e3bb", "4fd78138558fbe76ff000028519cc9ba3c58167ae80061e4", "4fd78138558fbe76ff0000284ff2abb4558fbe5c7d000c85", "4fd78138558fbe76ff000028514ad4323c5816775600103c", "4febe6a0558fbe0b6d00168a5153c6343c581632be0009cb", "4fd78138558fbe76ff0000285018d505558fbe46c4000171", "4fd78138558fbe76ff000028508601643c58164ab101dbba", "4ffa7c41558fbe37d2000d6b51a2b9f13c58164d7400126f", "519090753c5816386900023851beb5f33c5816147e006a3a", "4fe2c446558fbe330600240d5045749a558fbe6f9f00579e", "4fd78138558fbe76ff00002850cff96e3c58164f7f001ad9", "4fd78138558fbe76ff0000285180b4093c581635b500a6ce", "4fd78138558fbe76ff000028500e9c64558fbe665d000931", "4fd78138558fbe76ff00002851825f3a3c581635b5014cf3", "4fd7819b558fbe76ff00005550c978bf3c581667af00792d", "519090d13c581638690002d951b686453c5816364000434f", "4fd78138558fbe76ff0000285089705d3c58166ee000133e", "4fd78138558fbe76ff0000284ff5455b558fbe020f0001a5", "4fd78138558fbe76ff000028501daf72558fbe6953004609", "4fd78138558fbe76ff00002851130a053c58166bd700ea84", "4fd78138558fbe76ff00002850fe71593c58165f52007f16", "4fd78138558fbe76ff00002850122c83558fbe330700144d", "4fd78138558fbe76ff000028516230e83c581675e4001437", "4fd78138558fbe76ff000028519b208d3c58160cd00044a1", "4fd78138558fbe76ff000028507780a1558fbe0cb0001426", "4fd78138558fbe76ff000028512caa206c3582452d00004b", "519090d13c581638690002d951a5675b3c58164a9e001ba1", "50ea77d03c5816757202e7f4517a3ab23c581655a7003c83", "4fe2b375558fbe33060009d65119e7d73c58166bd7013e84", "4fd78138558fbe76ff00002850fde4c33c5816413e0001d4", "4fd78138558fbe76ff000028513099776c35824fcc000170", "519090753c58163869000238519586333c581671d3000ac1", "4febe6a0558fbe0b6d00168a514ab90d3c581677560005df", "4fd78138558fbe76ff000028511f01d3558fbe17dc000010", "4fd78138558fbe76ff000028519bee703c58165166000574", "4fd78138558fbe76ff0000285012338a558fbe33070015bd", "4fd78138558fbe76ff00002850c67e713c58164503013ff0", "4fd78138558fbe76ff0000284feaba67558fbe78c9005769", "4fd78138558fbe76ff00002850f773983c581675720a33d1", "4fe2aac8558fbe33060005b950a22c0f3c58167239010cf5", "50ea77d03c5816757202e7f4515bd4d73c581663450048f4", "4fd78138558fbe76ff000028500900ea558fbe442900120a", "4fd78138558fbe76ff00002850c949123c581667af002d16", "500ca86c558fbe1dc3000019518b0c8a3c58160aab01537c", "4febe6a0558fbe0b6d00168a517a42813c581655a70044e6", "4fd78138558fbe76ff0000285195e6113c5816452600017f", "4fd78138558fbe76ff0000284ffa7712558fbe37d20007ba", "4fd78138558fbe76ff00002850190a54558fbe4f56000adf", "4fd78138558fbe76ff0000285080c577558fbe1d04001bb7", "4fd78138558fbe76ff0000285164f4433c581675e400c5e1", "4fe2b375558fbe33060009d65100d8bf3c58163cf5001cbe", "4fd78138558fbe76ff000028510f1aa23c58166bd70089a0", "4fe362e1558fbe417d00017d5073968b558fbe2b65000c92", "500ca86c558fbe1dc300001951b305723c58167104000c49", "4fe2c446558fbe330600240d50c985fe3c581667af008b53", "4fd78138558fbe76ff00002850037319558fbe06fa0009d4", "4fd78138558fbe76ff0000284ffe6344558fbe5c8000705f", "4fd78138558fbe76ff0000285063b836558fbe3ebb002365", "4fe362e1558fbe417d00017d500df9b7558fbe41ae002064", "4fe2c446558fbe330600240d50c065993c58162bd700bf42", "4fd78138558fbe76ff00002851515f803c58162bcc003e15", "4fd78138558fbe76ff000028519c79be3c58167ae800363b", "4fd78138558fbe76ff00002850873c283c58160e3f0030ad", "4fd78138558fbe76ff00002850342694558fbe3ae9004a2d", "4fd78138558fbe76ff000028504d77a6558fbe45ef00038c", "500ca86c558fbe1dc3000019519090833c5816386900025b", "4fd78138558fbe76ff000028514c31c63c58166535002f63", "4fd78138558fbe76ff00002850b2ea983c5816744500518c", "4fd78138558fbe76ff0000285030630b558fbe2ff6002bc7", "4fd78138558fbe76ff00002851ad9d9f3c581603fa006980", "4fd78138558fbe76ff0000285101e8853c58163cf50036c7", "4fd78138558fbe76ff00002850933db43c58164408003794", "4fe362e1558fbe417d00017d5021c52c558fbe35ff000e7c", "5031d03a558fbe0bd8000169514fd13f3c581621620031b9", "4fd78138558fbe76ff0000285101e7fe3c58163cf50035ca", "4fd78138558fbe76ff00002850f51d893c58167572082de7", "519090753c58163869000238519998573c581620d7000bb0", "4fe362e1558fbe417d00017d50bc23ac3c58162068037af2", "4fd78138558fbe76ff000028503f3356558fbe59900034ae", "4fe2b375558fbe33060009d6516509e73c581675e400ccf3", "4fd78138558fbe76ff0000285152bf363c581648ab004608", "4fd78138558fbe76ff000028510129723c58163cf5002a22", "4fd78138558fbe76ff00002850b96ff53c5816206802cf27", "4fd78138558fbe76ff00002850ca7cce3c58162414000247", "4fd78138558fbe76ff00002850f5226c3c58167572084556", "500ca86c558fbe1dc300001950a256203c58163aea000bfd", "4fd78138558fbe76ff000028509c6af03c58162aa000442f", "4fd78138558fbe76ff000028500c9d4d558fbe4429005c58", "4fed20ab558fbe15cf002aa950d02d603c58164f7f004507", "4fd78138558fbe76ff000028505011c0558fbe6aaf00819f", "4fd78138558fbe76ff000028508de5b33c581651050000d6", "4fe2b375558fbe33060009d65164d1403c581675e400add0", "4fd78138558fbe76ff000028515132453c58162bcc002987", "4fe362e1558fbe417d00017d50458283558fbe6f9f0079f7", "4fd78138558fbe76ff000028505473be558fbe3b29001a43", "4fe362e1558fbe417d00017d5015e798558fbe33070044fc", "4febe6a0558fbe0b6d00168a51905a693c58164f2a000987", "4fd78138558fbe76ff000028515133d93c58162bcc002b16", "4fd78138558fbe76ff00002851b7f0e43c581636400084d0", "4fd78138558fbe76ff00002850a6066b3c5816114d00cc8e", "4fd78138558fbe76ff000028514fb9883c581621620029c6", "5125d2d33c58161f33005fe6516362f23c581675e40057e4", "4fd78138558fbe76ff00002850077a36558fbe0cd4000497", "5031d03a558fbe0bd8000169516d00eb3c58161bf7002e5c", "509b7ea8558fbe375900077d5100e979558fbe6ec60000ca", "4fd78138558fbe76ff0000285163a4e13c581675e4006d54", "4fd78138558fbe76ff00002851a845763c5816345e00ae56", "4fd78138558fbe76ff00002851b081743c58167be300dc78", "4fd78138558fbe76ff000028515263503c581648ab000b3f", "4fe362e1558fbe417d00017d50b2d11c3c581674450015f7", "5031d03a558fbe0bd80001695142f7343c58162cd50147ae", "4fd78138558fbe76ff000028514fa9c13c58162162002391", "4fd78138558fbe76ff000028515906083c58162621000ceb", "4fd78138558fbe76ff00002850a5c6d83c5816114d0075ec", "4fd78138558fbe76ff000028510b60d03c58166bd7006833", "4fd78138558fbe76ff00002851538ed93c58167bbf000401", "4fd78138558fbe76ff000028515cd8573c58164737001820", "4fd78138558fbe76ff0000285074e971558fbe41ef005738", "4fd78138558fbe76ff000028510b0a423c58166bd7005418", "500ca86c558fbe1dc30000195062d18f558fbe38650008fd", "4fd78138558fbe76ff0000285039ac39558fbe321d000a4d", "4fe362e1558fbe417d00017d5046c298558fbe122e00762a", "4febe6a0558fbe0b6d00168a5142c0d53c58162cd5013c51", "519090d13c581638690002d951a210e13c581619ea00d84d", "4fd78138558fbe76ff00002850e660623c58167572024498", "4fd78138558fbe76ff0000285089f74c3c58166ee00084d2", "4febe6a0558fbe0b6d00168a513045033c58162cd500020f", "4fd78138558fbe76ff0000285121aa283c58164351000863", "4fd78138558fbe76ff0000285179fc3f3c581655a7001b99", "4fd78138558fbe76ff00002850e80bd33c58167572027b6e", "4fd78138558fbe76ff0000285179f31b3c581655a70016c9", "4fd78138558fbe76ff0000284fe81ab7558fbe693d000350", "4fd78138558fbe76ff000028517dd5333c581655a700c09b", "5125d2d33c58161f33005fe65132b5473c58162cd5001704", "4fe362e1558fbe417d00017d501b3314558fbe69530019c2", "4fd78138558fbe76ff0000285163d9543c581675e4008317", "4fe362e1558fbe417d00017d5162307c3c581675e4001401", "519090753c5816386900023851933c6c3c58161fee00061c", "4fd78138558fbe76ff0000284fe2c2cb558fbe330600235d", "4fd78138558fbe76ff00002850c55f6b3c58164503006c8d", "4fd78138558fbe76ff000028512d59da3c58161f3300c82d", "500ca86c558fbe1dc3000019510394a53c58163cf5006498", "4fd78138558fbe76ff000028506926b8558fbe228e006be4", "4fe362e1558fbe417d00017d50dbacc23c58167572009a47", "4fd78138558fbe76ff000028510df63d3c58166bd7007e86", "4fd78138558fbe76ff00002850065753558fbe67ef000fa7", "4fd78138558fbe76ff000028514fc5873c58162162002f61", "5125d2d33c58161f33005fe651257e793c58161f33004e0f", "4fd78138558fbe76ff000028500e7c2a558fbe5f31001c04", "4fe2c446558fbe330600240d5121f882558fbe1aea000ffc", "4fd78138558fbe76ff0000284ffe744e558fbe5c80007b3b", "4fd78138558fbe76ff00002851bfc0ce3c5816147e008b9d", "4fd78138558fbe76ff000028509d09243c58162aa000bd0e", "4fd78138558fbe76ff000028514fbb9c3c58162162002aa3", "4ff63b9f558fbe0bfb001ca850c69bf43c5816450301921d", "4fd78138558fbe76ff00002850c165663c58162bd701541c", "4fd78138558fbe76ff00002850fcc6963c581675720d83d9", "4fd78138558fbe76ff000028517d24003c581655a700b78e", "4fd78138558fbe76ff00002850e506c03c5816757201cf07", "4fd78138558fbe76ff00002850320016558fbe0d940041b7", "4fe362e1558fbe417d00017d502c5e59558fbe67640012d3", "4fd78138558fbe76ff00002851998e0b3c581620d700014a", "4fd78138558fbe76ff00002850d13b093c58164f7f00b1b9", "4fd78138558fbe76ff0000285019e0a5558fbe4f560025a6", "4fd78138558fbe76ff000028512ed5f33c581625e6000398", "4fd78138558fbe76ff0000285073d198558fbe33970003d7", "4fd78138558fbe76ff000028505145b9558fbe21570004b6", "4fd78138558fbe76ff000028518225ac3c581635b50133ac", "4fd78138558fbe76ff0000285046bded558fbe122e006840", "50ffa67a558fbe477000005850ffb29d558fbe48ff000070", "4fd78138558fbe76ff0000285183190c3c58160d5800169b", "4fd78138558fbe76ff0000284fe2b8a5558fbe3306001705", "4fd78138558fbe76ff00002850ab4aa23c5816275b008070", "4fd78138558fbe76ff000028519aa08e3c58165b12000772", "500ca86c558fbe1dc300001950867835558fbe7261002289", "4fd78138558fbe76ff0000285077ddfe558fbe1ada002144", "4fd78138558fbe76ff0000285170fa3d3c58162567002110", "4fd78138558fbe76ff000028518912bf3c58160aab00c99f", "4fd78138558fbe76ff0000285174d81e3c581625670068c3", "4fd78138558fbe76ff000028512d75de3c58161f3300cd12", "4fd78138558fbe76ff0000284fe9b2fd558fbe78c9002177", "4fd78138558fbe76ff000028509f09763c58162aa00112c6", "4fd78138558fbe76ff000028512d77433c58161f3300cd55", "4fd78138558fbe76ff00002850f0bc633c5816757205f599", "4fd78138558fbe76ff00002850597a20558fbe749000077a", "4fd78138558fbe76ff000028519c301f3c58167ae80004d5", "4fd78138558fbe76ff00002850376ed2558fbe162d002b3d", "519090753c58163869000238519195c43c58166fdd00020f", "4fd78138558fbe76ff000028516b8fde3c5816147c00022f", "4fd78138558fbe76ff000028502c4344558fbe6764000358", "4fd78138558fbe76ff00002850efbf3d3c58167572052adb", "4fd78138558fbe76ff0000284ff675c1558fbe0fea0014ff"}...)

	s11 := sG.Intersect(sH)
	t.Log("INTERSECT", sG.Len(), sH.Len())
	t.Log("S11", s11, s11.Len())

	sI := set.NewStringSet([]string{"B", "D"}...)
	sJ := set.NewStringSet([]string{"A", "B", "D", "G"}...)
	s12 := sI.Difference(sJ)
	t.Log("DIFFERENCE", sI, sJ)
	t.Log("S12", s12)

	sK := set.NewStringSet([]string{}...)
	sL := set.NewStringSet([]string{"A", "B", "D", "G"}...)
	s13 := sK.Difference(sL)
	t.Log("DIFFERENCE", sK, sL)
	t.Log("S13", s13)

	sM := set.NewStringSet([]string{"A", "B"}...)
	sN := set.NewStringSet([]string{"A", "B", "C"}...)
	sO := set.NewStringSet([]string{"A", "B", "D", "G"}...)

	t.Log("SM", sM)
	t.Log("SN", sN)
	t.Log("SO", sO)
	t.Log("sM isSubset of sN", sM.IsSubset(sN))
	t.Log("sM isSubset of sO", sM.IsSubset(sO))
	t.Log("sN isSubset of sO", sN.IsSubset(sO))
	t.Log("sN isSuperset of sM", sN.IsSuperset(sM))
	t.Log("sO isSuperset of sM", sM.IsSuperset(sM))

	t.Log("sO hasMember B", sO.HasMember("B"))
	t.Log("sO hasMember F", sO.HasMember("F"))
}
