package face

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"github.com/Azure/go-autorest/tracing"
	"github.com/satori/go.uuid"
	"io"
	"net/http"
)

// LargeFaceListClient is the an API for face detection, verification, and identification.
type LargeFaceListClient struct {
	BaseClient
}

// NewLargeFaceListClient creates an instance of the LargeFaceListClient client.
func NewLargeFaceListClient(endpoint string) LargeFaceListClient {
	return LargeFaceListClient{New(endpoint)}
}

// AddFaceFromStream add a face to a large face list. The input face is specified as an image with a targetFace
// rectangle. It returns a persistedFaceId representing the added face, and persistedFaceId will not expire.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// imageParameter - an image stream.
// userData - user-specified data about the face for any purpose. The maximum length is 1KB.
// targetFace - a face rectangle to specify the target face to be added to a person in the format of
// "targetFace=left,top,width,height". E.g. "targetFace=10,10,100,100". If there is more than one face in the
// image, targetFace is required to specify which face to add. No targetFace means there is only one face
// detected in the entire image.
func (client LargeFaceListClient) AddFaceFromStream(ctx context.Context, largeFaceListID string, imageParameter io.ReadCloser, userData string, targetFace []int32) (result PersistedFace, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.AddFaceFromStream")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}},
		{TargetValue: userData,
			Constraints: []validation.Constraint{{Target: "userData", Name: validation.MaxLength, Rule: 1024, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "AddFaceFromStream", err.Error())
	}

	req, err := client.AddFaceFromStreamPreparer(ctx, largeFaceListID, imageParameter, userData, targetFace)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "AddFaceFromStream", nil, "Failure preparing request")
		return
	}

	resp, err := client.AddFaceFromStreamSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "AddFaceFromStream", resp, "Failure sending request")
		return
	}

	result, err = client.AddFaceFromStreamResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "AddFaceFromStream", resp, "Failure responding to request")
	}

	return
}

// AddFaceFromStreamPreparer prepares the AddFaceFromStream request.
func (client LargeFaceListClient) AddFaceFromStreamPreparer(ctx context.Context, largeFaceListID string, imageParameter io.ReadCloser, userData string, targetFace []int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	queryParameters := map[string]interface{}{}
	if len(userData) > 0 {
		queryParameters["userData"] = autorest.Encode("query", userData)
	}
	if targetFace != nil && len(targetFace) > 0 {
		queryParameters["targetFace"] = autorest.Encode("query", targetFace, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/persistedfaces", pathParameters),
		autorest.WithFile(imageParameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AddFaceFromStreamSender sends the AddFaceFromStream request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) AddFaceFromStreamSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AddFaceFromStreamResponder handles the response to the AddFaceFromStream request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) AddFaceFromStreamResponder(resp *http.Response) (result PersistedFace, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// AddFaceFromURL add a face to a large face list. The input face is specified as an image with a targetFace rectangle.
// It returns a persistedFaceId representing the added face, and persistedFaceId will not expire.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// imageURL - a JSON document with a URL pointing to the image that is to be analyzed.
// userData - user-specified data about the face for any purpose. The maximum length is 1KB.
// targetFace - a face rectangle to specify the target face to be added to a person in the format of
// "targetFace=left,top,width,height". E.g. "targetFace=10,10,100,100". If there is more than one face in the
// image, targetFace is required to specify which face to add. No targetFace means there is only one face
// detected in the entire image.
func (client LargeFaceListClient) AddFaceFromURL(ctx context.Context, largeFaceListID string, imageURL ImageURL, userData string, targetFace []int32) (result PersistedFace, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.AddFaceFromURL")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}},
		{TargetValue: userData,
			Constraints: []validation.Constraint{{Target: "userData", Name: validation.MaxLength, Rule: 1024, Chain: nil}}},
		{TargetValue: imageURL,
			Constraints: []validation.Constraint{{Target: "imageURL.URL", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "AddFaceFromURL", err.Error())
	}

	req, err := client.AddFaceFromURLPreparer(ctx, largeFaceListID, imageURL, userData, targetFace)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "AddFaceFromURL", nil, "Failure preparing request")
		return
	}

	resp, err := client.AddFaceFromURLSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "AddFaceFromURL", resp, "Failure sending request")
		return
	}

	result, err = client.AddFaceFromURLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "AddFaceFromURL", resp, "Failure responding to request")
	}

	return
}

// AddFaceFromURLPreparer prepares the AddFaceFromURL request.
func (client LargeFaceListClient) AddFaceFromURLPreparer(ctx context.Context, largeFaceListID string, imageURL ImageURL, userData string, targetFace []int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	queryParameters := map[string]interface{}{}
	if len(userData) > 0 {
		queryParameters["userData"] = autorest.Encode("query", userData)
	}
	if targetFace != nil && len(targetFace) > 0 {
		queryParameters["targetFace"] = autorest.Encode("query", targetFace, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/persistedfaces", pathParameters),
		autorest.WithJSON(imageURL),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AddFaceFromURLSender sends the AddFaceFromURL request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) AddFaceFromURLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AddFaceFromURLResponder handles the response to the AddFaceFromURL request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) AddFaceFromURLResponder(resp *http.Response) (result PersistedFace, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Create create an empty large face list with user-specified largeFaceListId, name, an optional userData and
// recognitionModel.
// <br /> Large face list is a list of faces, up to 1,000,000 faces, and used by [Face - Find
// Similar](/docs/services/563879b61984550e40cbbe8d/operations/563879b61984550f30395237).
// <br /> After creation, user should use [LargeFaceList Face -
// Add](/docs/services/563879b61984550e40cbbe8d/operations/5a158c10d2de3616c086f2d3) to import the faces and
// [LargeFaceList - Train](/docs/services/563879b61984550e40cbbe8d/operations/5a158422d2de3616c086f2d1) to make it
// ready for [Face - FindSimilar](/docs/services/563879b61984550e40cbbe8d/operations/563879b61984550f30395237). Faces
// are stored on server until [LargeFaceList -
// Delete](/docs/services/563879b61984550e40cbbe8d/operations/5a1580d5d2de3616c086f2cd) is called.
// <br /> Find Similar is used for scenario like finding celebrity-like faces, similar face filtering, or as a light
// way face identification. But if the actual use is to identify person, please use
// [PersonGroup](/docs/services/563879b61984550e40cbbe8d/operations/563879b61984550f30395244) /
// [LargePersonGroup](/docs/services/563879b61984550e40cbbe8d/operations/599acdee6ac60f11b48b5a9d) and [Face -
// Identify](/docs/services/563879b61984550e40cbbe8d/operations/563879b61984550f30395239).
// <br />
// * Free-tier subscription quota: 64 large face lists.
// * S0-tier subscription quota: 1,000,000 large face lists.
// <br />
// 'recognitionModel' should be specified to associate with this large face list. The default value for
// 'recognitionModel' is 'recognition_01', if the latest model needed, please explicitly specify the model you need in
// this parameter. New faces that are added to an existing large face list will use the recognition model that's
// already associated with the collection. Existing face features in a large face list can't be updated to features
// extracted by another version of recognition model.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// body - request body for creating a large face list.
func (client LargeFaceListClient) Create(ctx context.Context, largeFaceListID string, body MetaDataContract) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.Create")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "Create", err.Error())
	}

	req, err := client.CreatePreparer(ctx, largeFaceListID, body)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Create", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Create", resp, "Failure sending request")
		return
	}

	result, err = client.CreateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Create", resp, "Failure responding to request")
	}

	return
}

// CreatePreparer prepares the Create request.
func (client LargeFaceListClient) CreatePreparer(ctx context.Context, largeFaceListID string, body MetaDataContract) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}", pathParameters),
		autorest.WithJSON(body))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateSender sends the Create request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) CreateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateResponder handles the response to the Create request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) CreateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Delete delete an existing large face list according to faceListId. Persisted face images in the large face list will
// also be deleted.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
func (client LargeFaceListClient) Delete(ctx context.Context, largeFaceListID string) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.Delete")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "Delete", err.Error())
	}

	req, err := client.DeletePreparer(ctx, largeFaceListID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Delete", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Delete", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Delete", resp, "Failure responding to request")
	}

	return
}

// DeletePreparer prepares the Delete request.
func (client LargeFaceListClient) DeletePreparer(ctx context.Context, largeFaceListID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}", pathParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteSender sends the Delete request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) DeleteSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteResponder handles the response to the Delete request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) DeleteResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeleteFace delete an existing face from a large face list (given by a persistedFaceId and a largeFaceListId).
// Persisted image related to the face will also be deleted.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// persistedFaceID - id referencing a particular persistedFaceId of an existing face.
func (client LargeFaceListClient) DeleteFace(ctx context.Context, largeFaceListID string, persistedFaceID uuid.UUID) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.DeleteFace")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "DeleteFace", err.Error())
	}

	req, err := client.DeleteFacePreparer(ctx, largeFaceListID, persistedFaceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "DeleteFace", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteFaceSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "DeleteFace", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteFaceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "DeleteFace", resp, "Failure responding to request")
	}

	return
}

// DeleteFacePreparer prepares the DeleteFace request.
func (client LargeFaceListClient) DeleteFacePreparer(ctx context.Context, largeFaceListID string, persistedFaceID uuid.UUID) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
		"persistedFaceId": autorest.Encode("path", persistedFaceID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/persistedfaces/{persistedFaceId}", pathParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteFaceSender sends the DeleteFace request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) DeleteFaceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteFaceResponder handles the response to the DeleteFace request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) DeleteFaceResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Get retrieve a large face list’s largeFaceListId, name, userData and recognitionModel.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// returnRecognitionModel - a value indicating whether the operation should return 'recognitionModel' in
// response.
func (client LargeFaceListClient) Get(ctx context.Context, largeFaceListID string, returnRecognitionModel *bool) (result LargeFaceList, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.Get")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "Get", err.Error())
	}

	req, err := client.GetPreparer(ctx, largeFaceListID, returnRecognitionModel)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Get", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Get", resp, "Failure sending request")
		return
	}

	result, err = client.GetResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Get", resp, "Failure responding to request")
	}

	return
}

// GetPreparer prepares the Get request.
func (client LargeFaceListClient) GetPreparer(ctx context.Context, largeFaceListID string, returnRecognitionModel *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	queryParameters := map[string]interface{}{}
	if returnRecognitionModel != nil {
		queryParameters["returnRecognitionModel"] = autorest.Encode("query", *returnRecognitionModel)
	} else {
		queryParameters["returnRecognitionModel"] = autorest.Encode("query", false)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSender sends the Get request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) GetSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetResponder handles the response to the Get request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) GetResponder(resp *http.Response) (result LargeFaceList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetFace retrieve information about a persisted face (specified by persistedFaceId and its belonging
// largeFaceListId).
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// persistedFaceID - id referencing a particular persistedFaceId of an existing face.
func (client LargeFaceListClient) GetFace(ctx context.Context, largeFaceListID string, persistedFaceID uuid.UUID) (result PersistedFace, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.GetFace")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "GetFace", err.Error())
	}

	req, err := client.GetFacePreparer(ctx, largeFaceListID, persistedFaceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "GetFace", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetFaceSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "GetFace", resp, "Failure sending request")
		return
	}

	result, err = client.GetFaceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "GetFace", resp, "Failure responding to request")
	}

	return
}

// GetFacePreparer prepares the GetFace request.
func (client LargeFaceListClient) GetFacePreparer(ctx context.Context, largeFaceListID string, persistedFaceID uuid.UUID) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
		"persistedFaceId": autorest.Encode("path", persistedFaceID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/persistedfaces/{persistedFaceId}", pathParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetFaceSender sends the GetFace request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) GetFaceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetFaceResponder handles the response to the GetFace request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) GetFaceResponder(resp *http.Response) (result PersistedFace, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetTrainingStatus retrieve the training status of a large face list (completed or ongoing).
// Parameters:
// largeFaceListID - id referencing a particular large face list.
func (client LargeFaceListClient) GetTrainingStatus(ctx context.Context, largeFaceListID string) (result TrainingStatus, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.GetTrainingStatus")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "GetTrainingStatus", err.Error())
	}

	req, err := client.GetTrainingStatusPreparer(ctx, largeFaceListID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "GetTrainingStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetTrainingStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "GetTrainingStatus", resp, "Failure sending request")
		return
	}

	result, err = client.GetTrainingStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "GetTrainingStatus", resp, "Failure responding to request")
	}

	return
}

// GetTrainingStatusPreparer prepares the GetTrainingStatus request.
func (client LargeFaceListClient) GetTrainingStatusPreparer(ctx context.Context, largeFaceListID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/training", pathParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetTrainingStatusSender sends the GetTrainingStatus request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) GetTrainingStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetTrainingStatusResponder handles the response to the GetTrainingStatus request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) GetTrainingStatusResponder(resp *http.Response) (result TrainingStatus, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// List list large face lists’ information of largeFaceListId, name, userData and recognitionModel. <br />
// To get face information inside largeFaceList use [LargeFaceList Face -
// Get](/docs/services/563879b61984550e40cbbe8d/operations/5a158cf2d2de3616c086f2d5)<br />
// * Large face lists are stored in alphabetical order of largeFaceListId.
// * "start" parameter (string, optional) is a user-provided largeFaceListId value that returned entries have larger
// ids by string comparison. "start" set to empty to indicate return from the first item.
// * "top" parameter (int, optional) specifies the number of entries to return. A maximal of 1000 entries can be
// returned in one call. To fetch more, you can specify "start" with the last returned entry’s Id of the current call.
// <br />
// For example, total 5 large person lists: "list1", ..., "list5".
// <br /> "start=&top=" will return all 5 lists.
// <br /> "start=&top=2" will return "list1", "list2".
// <br /> "start=list2&top=3" will return "list3", "list4", "list5".
// Parameters:
// returnRecognitionModel - a value indicating whether the operation should return 'recognitionModel' in
// response.
func (client LargeFaceListClient) List(ctx context.Context, returnRecognitionModel *bool) (result ListLargeFaceList, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.List")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.ListPreparer(ctx, returnRecognitionModel)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "List", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "List", resp, "Failure sending request")
		return
	}

	result, err = client.ListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "List", resp, "Failure responding to request")
	}

	return
}

// ListPreparer prepares the List request.
func (client LargeFaceListClient) ListPreparer(ctx context.Context, returnRecognitionModel *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	queryParameters := map[string]interface{}{}
	if returnRecognitionModel != nil {
		queryParameters["returnRecognitionModel"] = autorest.Encode("query", *returnRecognitionModel)
	} else {
		queryParameters["returnRecognitionModel"] = autorest.Encode("query", false)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPath("/largefacelists"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListSender sends the List request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) ListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListResponder handles the response to the List request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) ListResponder(resp *http.Response) (result ListLargeFaceList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ListFaces list all faces in a large face list, and retrieve face information (including userData and
// persistedFaceIds of registered faces of the face).
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// start - starting face id to return (used to list a range of faces).
// top - number of faces to return starting with the face id indicated by the 'start' parameter.
func (client LargeFaceListClient) ListFaces(ctx context.Context, largeFaceListID string, start string, top *int32) (result ListPersistedFace, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.ListFaces")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}},
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMaximum, Rule: int64(1000), Chain: nil},
					{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "ListFaces", err.Error())
	}

	req, err := client.ListFacesPreparer(ctx, largeFaceListID, start, top)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "ListFaces", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListFacesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "ListFaces", resp, "Failure sending request")
		return
	}

	result, err = client.ListFacesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "ListFaces", resp, "Failure responding to request")
	}

	return
}

// ListFacesPreparer prepares the ListFaces request.
func (client LargeFaceListClient) ListFacesPreparer(ctx context.Context, largeFaceListID string, start string, top *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	queryParameters := map[string]interface{}{}
	if len(start) > 0 {
		queryParameters["start"] = autorest.Encode("query", start)
	}
	if top != nil {
		queryParameters["top"] = autorest.Encode("query", *top)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/persistedfaces", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListFacesSender sends the ListFaces request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) ListFacesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListFacesResponder handles the response to the ListFaces request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) ListFacesResponder(resp *http.Response) (result ListPersistedFace, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Train queue a large face list training task, the training task may not be started immediately.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
func (client LargeFaceListClient) Train(ctx context.Context, largeFaceListID string) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.Train")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "Train", err.Error())
	}

	req, err := client.TrainPreparer(ctx, largeFaceListID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Train", nil, "Failure preparing request")
		return
	}

	resp, err := client.TrainSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Train", resp, "Failure sending request")
		return
	}

	result, err = client.TrainResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Train", resp, "Failure responding to request")
	}

	return
}

// TrainPreparer prepares the Train request.
func (client LargeFaceListClient) TrainPreparer(ctx context.Context, largeFaceListID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/train", pathParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// TrainSender sends the Train request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) TrainSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// TrainResponder handles the response to the Train request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) TrainResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Update update information of a large face list.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// body - request body for updating a large face list.
func (client LargeFaceListClient) Update(ctx context.Context, largeFaceListID string, body NameAndUserDataContract) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.Update")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "Update", err.Error())
	}

	req, err := client.UpdatePreparer(ctx, largeFaceListID, body)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Update", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Update", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "Update", resp, "Failure responding to request")
	}

	return
}

// UpdatePreparer prepares the Update request.
func (client LargeFaceListClient) UpdatePreparer(ctx context.Context, largeFaceListID string, body NameAndUserDataContract) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}", pathParameters),
		autorest.WithJSON(body))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateSender sends the Update request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) UpdateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateResponder handles the response to the Update request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) UpdateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UpdateFace update a persisted face's userData field.
// Parameters:
// largeFaceListID - id referencing a particular large face list.
// persistedFaceID - id referencing a particular persistedFaceId of an existing face.
// body - request body for updating persisted face.
func (client LargeFaceListClient) UpdateFace(ctx context.Context, largeFaceListID string, persistedFaceID uuid.UUID, body UpdateFaceRequest) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/LargeFaceListClient.UpdateFace")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: largeFaceListID,
			Constraints: []validation.Constraint{{Target: "largeFaceListID", Name: validation.MaxLength, Rule: 64, Chain: nil},
				{Target: "largeFaceListID", Name: validation.Pattern, Rule: `^[a-z0-9-_]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("face.LargeFaceListClient", "UpdateFace", err.Error())
	}

	req, err := client.UpdateFacePreparer(ctx, largeFaceListID, persistedFaceID, body)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "UpdateFace", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateFaceSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "UpdateFace", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateFaceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "face.LargeFaceListClient", "UpdateFace", resp, "Failure responding to request")
	}

	return
}

// UpdateFacePreparer prepares the UpdateFace request.
func (client LargeFaceListClient) UpdateFacePreparer(ctx context.Context, largeFaceListID string, persistedFaceID uuid.UUID, body UpdateFaceRequest) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	pathParameters := map[string]interface{}{
		"largeFaceListId": autorest.Encode("path", largeFaceListID),
		"persistedFaceId": autorest.Encode("path", persistedFaceID),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{Endpoint}/face/v1.0", urlParameters),
		autorest.WithPathParameters("/largefacelists/{largeFaceListId}/persistedfaces/{persistedFaceId}", pathParameters),
		autorest.WithJSON(body))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateFaceSender sends the UpdateFace request. The method will close the
// http.Response Body if it receives an error.
func (client LargeFaceListClient) UpdateFaceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateFaceResponder handles the response to the UpdateFace request. The method always
// closes the http.Response Body.
func (client LargeFaceListClient) UpdateFaceResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}
