package pop

import (
	"fmt"

	"github.com/gobuffalo/packd"
	"github.com/pkg/errors"
)

// MigrationBox is a wrapper around packr.Box and Migrator.
// This will allow you to run migrations from a packed box
// inside of a compiled binary.
type MigrationBox struct {
	Migrator
	Box packd.Walkable
}

// NewMigrationBox from a packr.Box and a Connection.
func NewMigrationBox(box packd.Walkable, c *Connection) (MigrationBox, error) {
	fm := MigrationBox{
		Migrator: NewMigrator(c),
		Box:      box,
	}

	err := fm.findMigrations()
	if err != nil {
		return fm, err
	}

	return fm, nil
}

func (fm *MigrationBox) findMigrations() error {
	return fm.Box.Walk(func(p string, f packd.File) error {
		info, err := f.FileInfo()
		if err != nil {
			return err
		}
		matches := mrx.FindAllStringSubmatch(info.Name(), -1)
		if len(matches) == 0 {
			return nil
		}
		m := matches[0]
		var dbType string
		if m[3] == "" {
			dbType = "all"
		} else {
			dbType = normalizeSynonyms(m[3][1:])
			if !DialectSupported(dbType) {
				return fmt.Errorf("unsupported dialect %s", dbType)
			}
		}
		mf := Migration{
			Path:      p,
			Version:   m[1],
			Name:      m[2],
			DBType:    dbType,
			Direction: m[4],
			Type:      m[5],
			Runner: func(mf Migration, tx *Connection) error {
				content, err := migrationContent(mf, tx, f)
				if err != nil {
					return errors.Wrapf(err, "error processing %s", mf.Path)
				}

				if content == "" {
					return nil
				}

				err = tx.RawQuery(content).Exec()
				if err != nil {
					return errors.Wrapf(err, "error executing %s, sql: %s", mf.Path, content)
				}
				return nil
			},
		}
		fm.Migrations[mf.Direction] = append(fm.Migrations[mf.Direction], mf)
		return nil
	})
}
