// +build ignore

package main

import (
	"bytes"
	"go/format"
	"log"
	"os"
	"sort"
	"strings"
	"text/template"

	"github.com/terraform-providers/terraform-provider-aws/aws/internal/keyvaluetags"
)

const filename = `create_tags_gen.go`

var serviceNames = []string{
	"ec2",
}

type TemplateData struct {
	ServiceNames []string
}

func main() {
	// Always sort to reduce any potential generation churn
	sort.Strings(serviceNames)

	templateData := TemplateData{
		ServiceNames: serviceNames,
	}
	templateFuncMap := template.FuncMap{
		"ClientType":                        keyvaluetags.ServiceClientType,
		"ResourceNotFoundErrorCode":         keyvaluetags.ServiceResourceNotFoundErrorCode,
		"ResourceNotFoundErrorCodeContains": keyvaluetags.ServiceResourceNotFoundErrorCodeContains,
		"RetryCreationOnResourceNotFound":   keyvaluetags.ServiceRetryCreationOnResourceNotFound,
		"TagFunction":                       keyvaluetags.ServiceTagFunction,
		"TagFunctionBatchSize":              keyvaluetags.ServiceTagFunctionBatchSize,
		"TagInputCustomValue":               keyvaluetags.ServiceTagInputCustomValue,
		"TagInputIdentifierField":           keyvaluetags.ServiceTagInputIdentifierField,
		"TagInputIdentifierRequiresSlice":   keyvaluetags.ServiceTagInputIdentifierRequiresSlice,
		"TagInputResourceTypeField":         keyvaluetags.ServiceTagInputResourceTypeField,
		"TagInputTagsField":                 keyvaluetags.ServiceTagInputTagsField,
		"TagPackage":                        keyvaluetags.ServiceTagPackage,
		"Title":                             strings.Title,
	}

	tmpl, err := template.New("createtags").Funcs(templateFuncMap).Parse(templateBody)

	if err != nil {
		log.Fatalf("error parsing template: %s", err)
	}

	var buffer bytes.Buffer
	err = tmpl.Execute(&buffer, templateData)

	if err != nil {
		log.Fatalf("error executing template: %s", err)
	}

	generatedFileContents, err := format.Source(buffer.Bytes())

	if err != nil {
		log.Fatalf("error formatting generated file: %s", err)
	}

	f, err := os.Create(filename)

	if err != nil {
		log.Fatalf("error creating file (%s): %s", filename, err)
	}

	defer f.Close()

	_, err = f.Write(generatedFileContents)

	if err != nil {
		log.Fatalf("error writing to file (%s): %s", filename, err)
	}
}

var templateBody = `
// Code generated by generators/createtags/main.go; DO NOT EDIT.

package keyvaluetags

import (
	"errors"
	"fmt"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
{{- range .ServiceNames }}
	"github.com/aws/aws-sdk-go/service/{{ . }}"
{{- end }}
	"github.com/hashicorp/terraform-plugin-sdk/helper/resource"
)

const EventualConsistencyTimeout = 5 * time.Minute

// Similar to isAWSErr from aws/awserr.go
// TODO: Add and export in shared package
func isAWSErrCode(err error, code string) bool {
	var awsErr awserr.Error
	if errors.As(err, &awsErr) {
		return awsErr.Code() == code
	}
	return false
}

// TODO: Add and export in shared package
func isAWSErrCodeContains(err error, code string) bool {
	var awsErr awserr.Error
	if errors.As(err, &awsErr) {
		return strings.Contains(awsErr.Code(), code)
	}
	return false
}

// Copied from aws/utils.go
// TODO: Export in shared package or add to Terraform Plugin SDK
func isResourceTimeoutError(err error) bool {
	timeoutErr, ok := err.(*resource.TimeoutError)
	return ok && timeoutErr.LastError == nil
}

{{- range .ServiceNames }}

// {{ . | Title }}CreateTags creates {{ . }} service tags for new resources.
// The identifier is typically the Amazon Resource Name (ARN), although
// it may also be a different identifier depending on the service.
func {{ . | Title }}CreateTags(conn {{ . | ClientType }}, identifier string{{ if . | TagInputResourceTypeField }}, resourceType string{{ end }}, tagsMap interface{}) error {
	tags := New(tagsMap)

	{{- if . | TagFunctionBatchSize }}
	for _, tags := range tags.Chunks({{ . | TagFunctionBatchSize }}) {
	{{- end }}
	input := &{{ . | TagPackage }}.{{ . | TagFunction }}Input{
		{{- if . | TagInputIdentifierRequiresSlice }}
		{{ . | TagInputIdentifierField }}:   aws.StringSlice([]string{identifier}),
		{{- else }}
		{{ . | TagInputIdentifierField }}:   aws.String(identifier),
		{{- end }}
		{{- if . | TagInputResourceTypeField }}
		{{ . | TagInputResourceTypeField }}: aws.String(resourceType),
		{{- end }}
		{{- if . | TagInputCustomValue }}
		{{ . | TagInputTagsField }}:         {{ . | TagInputCustomValue }},
		{{- else }}
		{{ . | TagInputTagsField }}:         tags.IgnoreAws().{{ . | Title }}Tags(),
		{{- end }}
	}

	{{- if . | RetryCreationOnResourceNotFound }}

	err := resource.Retry(EventualConsistencyTimeout, func() *resource.RetryError {
		_, err := conn.{{ . | TagFunction }}(input)

		{{- if . | ResourceNotFoundErrorCodeContains }}

		if isAWSErrCodeContains(err, "{{ . | ResourceNotFoundErrorCodeContains }}") {

		{{- else }}

		if isAWSErrCode(err, {{ . | ResourceNotFoundErrorCode }}) {

		{{- end }}
			return resource.RetryableError(err)
		}

		if err != nil {
			return resource.NonRetryableError(err)
		}

		return nil
	})

	if isResourceTimeoutError(err) {
		_, err = conn.{{ . | TagFunction }}(input)
	}
	{{- else }}
	_, err := conn.{{ . | TagFunction }}(input)
	{{- end }}

	if err != nil {
		return fmt.Errorf("error tagging resource (%s): %w", identifier, err)
	}
	{{- if . | TagFunctionBatchSize }}
	}
	{{- end }}

	return nil
}
{{- end }}
`
