package tests

import (
	"fmt"
	"testing"

	"github.com/hashicorp/terraform-plugin-sdk/helper/resource"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/acceptance"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/features"
)

func TestAccAzureRMLogicAppTriggerRecurrence_month(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Month", 1),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Month"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "1"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_requiresImport(t *testing.T) {
	if !features.ShouldResourcesBeImported() {
		t.Skip("Skipping since resources aren't required to be imported")
		return
	}

	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Month", 1),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Month"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "1"),
				),
			},
			{
				Config:      testAccAzureRMLogicAppTriggerRecurrence_requiresImport(data, "Month", 1),
				ExpectError: acceptance.RequiresImportError("azurerm_logic_app_trigger_recurrence"),
			},
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_week(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Week", 2),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Week"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "2"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_day(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Day", 3),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Day"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "3"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_minute(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Minute", 4),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Minute"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "4"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_second(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Second", 30),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Second"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "30"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_hour(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Hour", 4),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Hour"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "4"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_update(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Month", 1),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Month"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "1"),
				),
			},
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_basic(data, "Month", 3),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "frequency", "Month"),
					resource.TestCheckResourceAttr(data.ResourceName, "interval", "3"),
				),
			},
		},
	})
}

func TestAccAzureRMLogicAppTriggerRecurrence_startTime(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_logic_app_trigger_recurrence", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMLogicAppWorkflowDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMLogicAppTriggerRecurrence_startTime(data, "2020-01-01T01:02:03Z"),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMLogicAppTriggerExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "start_time", "2020-01-01T01:02:03Z"),
				),
			},
			data.ImportStep(),
		},
	})
}

func testAccAzureRMLogicAppTriggerRecurrence_basic(data acceptance.TestData, frequency string, interval int) string {
	return fmt.Sprintf(`
resource "azurerm_resource_group" "test" {
  name     = "acctestRG-%d"
  location = "%s"
}

resource "azurerm_logic_app_workflow" "test" {
  name                = "acctestlaw-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"
}

resource "azurerm_logic_app_trigger_recurrence" "test" {
  name         = "frequency-trigger"
  logic_app_id = "${azurerm_logic_app_workflow.test.id}"
  frequency    = "%s"
  interval     = %d
}
`, data.RandomInteger, data.Locations.Primary, data.RandomInteger, frequency, interval)
}

func testAccAzureRMLogicAppTriggerRecurrence_startTime(data acceptance.TestData, startTime string) string {
	return fmt.Sprintf(`
resource "azurerm_resource_group" "test" {
  name     = "acctestRG-%d"
  location = "%s"
}

resource "azurerm_logic_app_workflow" "test" {
  name                = "acctestlaw-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"
}

resource "azurerm_logic_app_trigger_recurrence" "test" {
  name         = "frequency-trigger"
  logic_app_id = "${azurerm_logic_app_workflow.test.id}"
  frequency    = "Month"
  interval     = 1
  start_time   = "%s"
}
`, data.RandomInteger, data.Locations.Primary, data.RandomInteger, startTime)
}

func testAccAzureRMLogicAppTriggerRecurrence_requiresImport(data acceptance.TestData, frequency string, interval int) string {
	template := testAccAzureRMLogicAppTriggerRecurrence_basic(data, frequency, interval)
	return fmt.Sprintf(`
%s

resource "azurerm_logic_app_trigger_recurrence" "import" {
  name         = "${azurerm_logic_app_trigger_recurrence.test.name}"
  logic_app_id = "${azurerm_logic_app_trigger_recurrence.test.logic_app_id}"
  frequency    = "${azurerm_logic_app_trigger_recurrence.test.frequency}"
  interval     = "${azurerm_logic_app_trigger_recurrence.test.interval}"
}
`, template)
}
