package tests

import (
	"fmt"
	"testing"

	"github.com/hashicorp/terraform-plugin-sdk/helper/resource"
	"github.com/hashicorp/terraform-plugin-sdk/terraform"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/acceptance"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/clients"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/features"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/utils"
)

func TestAccAzureRMNetworkProfile_basic(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_network_profile", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMNetworkProfileDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMNetworkProfile_basic(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMNetworkProfileExists(data.ResourceName),
					resource.TestCheckResourceAttrSet(data.ResourceName, "container_network_interface_ids.#"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMNetworkProfile_requiresImport(t *testing.T) {
	if !features.ShouldResourcesBeImported() {
		t.Skip("Skipping since resources aren't required to be imported")
		return
	}

	data := acceptance.BuildTestData(t, "azurerm_network_profile", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMNetworkProfileDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMNetworkProfile_basic(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMNetworkProfileExists(data.ResourceName),
				),
			},
			{
				Config:      testAccAzureRMNetworkProfile_requiresImport(data),
				ExpectError: acceptance.RequiresImportError("azurerm_network_profile"),
			},
		},
	})
}

func TestAccAzureRMNetworkProfile_withTags(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_network_profile", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMNetworkProfileDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMNetworkProfile_withTags(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMNetworkProfileExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "tags.%", "2"),
					resource.TestCheckResourceAttr(data.ResourceName, "tags.environment", "Production"),
					resource.TestCheckResourceAttr(data.ResourceName, "tags.cost_center", "MSFT"),
				),
			},
			{
				Config: testAccAzureRMNetworkProfile_withUpdatedTags(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMNetworkProfileExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "tags.%", "1"),
					resource.TestCheckResourceAttr(data.ResourceName, "tags.environment", "Staging"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMNetworkProfile_disappears(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_network_profile", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMNetworkProfileDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMNetworkProfile_basic(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMNetworkProfileExists(data.ResourceName),
					testCheckAzureRMNetworkProfileDisappears(data.ResourceName),
				),
				ExpectNonEmptyPlan: true,
			},
		},
	})
}

func testCheckAzureRMNetworkProfileExists(resourceName string) resource.TestCheckFunc {
	return func(s *terraform.State) error {
		client := acceptance.AzureProvider.Meta().(*clients.Client).Network.ProfileClient
		ctx := acceptance.AzureProvider.Meta().(*clients.Client).StopContext

		// Ensure we have enough information in state to look up in API
		rs, ok := s.RootModule().Resources[resourceName]
		if !ok {
			return fmt.Errorf("Not found: %s", resourceName)
		}

		name := rs.Primary.Attributes["name"]
		resourceGroup, hasResourceGroup := rs.Primary.Attributes["resource_group_name"]
		if !hasResourceGroup {
			return fmt.Errorf("Bad: no resource group found in state for Network Profile: %q", name)
		}

		resp, err := client.Get(ctx, resourceGroup, name, "")
		if err != nil {
			if utils.ResponseWasNotFound(resp.Response) {
				return fmt.Errorf("Bad: Network Profile %q (Resource Group: %q) does not exist", name, resourceGroup)
			}

			return fmt.Errorf("Bad: Get on netProfileClient: %+v", err)
		}

		return nil
	}
}

func testCheckAzureRMNetworkProfileDisappears(resourceName string) resource.TestCheckFunc {
	return func(s *terraform.State) error {
		client := acceptance.AzureProvider.Meta().(*clients.Client).Network.ProfileClient
		ctx := acceptance.AzureProvider.Meta().(*clients.Client).StopContext

		// Ensure we have enough information in state to look up in API
		rs, ok := s.RootModule().Resources[resourceName]
		if !ok {
			return fmt.Errorf("Not found: %s", resourceName)
		}

		name := rs.Primary.Attributes["name"]
		resourceGroup, hasResourceGroup := rs.Primary.Attributes["resource_group_name"]
		if !hasResourceGroup {
			return fmt.Errorf("Bad: no resource group found in state for Network Profile: %q", name)
		}

		if _, err := client.Delete(ctx, resourceGroup, name); err != nil {
			return fmt.Errorf("Bad: Delete on netProfileClient: %+v", err)
		}

		return nil
	}
}

func testCheckAzureRMNetworkProfileDestroy(s *terraform.State) error {
	client := acceptance.AzureProvider.Meta().(*clients.Client).Network.ProfileClient
	ctx := acceptance.AzureProvider.Meta().(*clients.Client).StopContext

	for _, rs := range s.RootModule().Resources {
		if rs.Type != "azurerm_network_profile" {
			continue
		}

		name := rs.Primary.Attributes["name"]
		resourceGroup := rs.Primary.Attributes["resource_group_name"]

		resp, err := client.Get(ctx, resourceGroup, name, "")
		if err != nil {
			if utils.ResponseWasNotFound(resp.Response) {
				return nil
			}

			return err
		}

		return fmt.Errorf("Network Profile still exists:\n%#v", resp.ProfilePropertiesFormat)
	}

	return nil
}

func testAccAzureRMNetworkProfile_basic(data acceptance.TestData) string {
	return fmt.Sprintf(`
resource "azurerm_resource_group" "test" {
  name     = "acctestRG-%d"
  location = "%s"
}

resource "azurerm_virtual_network" "test" {
  name                = "acctestvirtnet-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"
  address_space       = ["10.1.0.0/16"]
}

resource "azurerm_subnet" "test" {
  name                 = "acctestsubnet-%d"
  resource_group_name  = "${azurerm_resource_group.test.name}"
  virtual_network_name = "${azurerm_virtual_network.test.name}"
  address_prefix       = "10.1.0.0/24"

  delegation {
    name = "acctestdelegation-%d"

    service_delegation {
      name    = "Microsoft.ContainerInstance/containerGroups"
      actions = ["Microsoft.Network/virtualNetworks/subnets/action"]
    }
  }
}

resource "azurerm_network_profile" "test" {
  name                = "acctestnetprofile-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"

  container_network_interface {
    name = "acctesteth-%d"

    ip_configuration {
      name      = "acctestipconfig-%d"
      subnet_id = "${azurerm_subnet.test.id}"
    }
  }
}
`, data.RandomInteger, data.Locations.Primary, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger)
}

func testAccAzureRMNetworkProfile_requiresImport(data acceptance.TestData) string {
	basicConfig := testAccAzureRMNetworkProfile_basic(data)
	return fmt.Sprintf(`
%s

resource "azurerm_network_profile" "import" {
  name                = "${azurerm_network_profile.test.name}"
  location            = "${azurerm_network_profile.test.location}"
  resource_group_name = "${azurerm_network_profile.test.resource_group_name}"

  container_network_interface {
    name = "${azurerm_network_profile.test.container_network_interface.0.name}"

    ip_configuration {
      name      = "${azurerm_network_profile.test.container_network_interface.0.ip_configuration.0.name}"
      subnet_id = "${azurerm_subnet.test.id}"
    }
  }
}
`, basicConfig)
}

func testAccAzureRMNetworkProfile_withTags(data acceptance.TestData) string {
	return fmt.Sprintf(`
resource "azurerm_resource_group" "test" {
  name     = "acctestRG-%d"
  location = "%s"
}

resource "azurerm_virtual_network" "test" {
  name                = "acctestvirtnet-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"
  address_space       = ["10.1.0.0/16"]
}

resource "azurerm_subnet" "test" {
  name                 = "acctestsubnet-%d"
  resource_group_name  = "${azurerm_resource_group.test.name}"
  virtual_network_name = "${azurerm_virtual_network.test.name}"
  address_prefix       = "10.1.0.0/24"

  delegation {
    name = "acctestdelegation-%d"

    service_delegation {
      name    = "Microsoft.ContainerInstance/containerGroups"
      actions = ["Microsoft.Network/virtualNetworks/subnets/action"]
    }
  }
}

resource "azurerm_network_profile" "test" {
  name                = "acctestnetprofile-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"

  container_network_interface {
    name = "acctesteth-%d"

    ip_configuration {
      name      = "acctestipconfig-%d"
      subnet_id = "${azurerm_subnet.test.id}"
    }
  }

  tags = {
    environment = "Production"
    cost_center = "MSFT"
  }
}
`, data.RandomInteger, data.Locations.Primary, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger)
}

func testAccAzureRMNetworkProfile_withUpdatedTags(data acceptance.TestData) string {
	return fmt.Sprintf(`
resource "azurerm_resource_group" "test" {
  name     = "acctestRG-%d"
  location = "%s"
}

resource "azurerm_virtual_network" "test" {
  name                = "acctestvirtnet-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"
  address_space       = ["10.1.0.0/16"]
}

resource "azurerm_subnet" "test" {
  name                 = "acctestsubnet-%d"
  resource_group_name  = "${azurerm_resource_group.test.name}"
  virtual_network_name = "${azurerm_virtual_network.test.name}"
  address_prefix       = "10.1.0.0/24"

  delegation {
    name = "acctestdelegation-%d"

    service_delegation {
      name    = "Microsoft.ContainerInstance/containerGroups"
      actions = ["Microsoft.Network/virtualNetworks/subnets/action"]
    }
  }
}

resource "azurerm_network_profile" "test" {
  name                = "acctestnetprofile-%d"
  location            = "${azurerm_resource_group.test.location}"
  resource_group_name = "${azurerm_resource_group.test.name}"

  container_network_interface {
    name = "acctesteth-%d"

    ip_configuration {
      name      = "acctestipconfig-%d"
      subnet_id = "${azurerm_subnet.test.id}"
    }
  }

  tags = {
    environment = "Staging"
  }
}
`, data.RandomInteger, data.Locations.Primary, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger, data.RandomInteger)
}
