package managementgroup

import (
	"fmt"
	"time"

	"github.com/Azure/azure-sdk-for-go/services/preview/resources/mgmt/2018-03-01-preview/managementgroups"
	"github.com/hashicorp/terraform-plugin-sdk/helper/schema"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/clients"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/services/managementgroup/validate"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/timeouts"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/utils"
)

func dataSourceArmManagementGroup() *schema.Resource {
	return &schema.Resource{
		Read: dataSourceArmManagementGroupRead,

		Timeouts: &schema.ResourceTimeout{
			Read: schema.DefaultTimeout(5 * time.Minute),
		},

		Schema: map[string]*schema.Schema{
			"group_id": {
				Type:         schema.TypeString,
				Optional:     true,
				Computed:     true,
				ExactlyOneOf: []string{"name", "group_id"},
				Deprecated:   "Deprecated in favour of `name`",
				ValidateFunc: validate.ManagementGroupName,
			},

			"name": {
				Type:         schema.TypeString,
				Optional:     true, // TODO -- change back to required after the deprecation
				Computed:     true, // TODO -- remove computed after the deprecation
				ExactlyOneOf: []string{"name", "group_id"},
				ValidateFunc: validate.ManagementGroupName,
			},

			"display_name": {
				Type:     schema.TypeString,
				Computed: true,
			},

			"parent_management_group_id": {
				Type:     schema.TypeString,
				Computed: true,
			},

			"subscription_ids": {
				Type:     schema.TypeSet,
				Computed: true,
				Elem:     &schema.Schema{Type: schema.TypeString},
				Set:      schema.HashString,
			},
		},
	}
}

func dataSourceArmManagementGroupRead(d *schema.ResourceData, meta interface{}) error {
	client := meta.(*clients.Client).ManagementGroups.GroupsClient
	ctx, cancel := timeouts.ForRead(meta.(*clients.Client).StopContext, d)
	defer cancel()

	groupName := ""
	if v, ok := d.GetOk("name"); ok {
		groupName = v.(string)
	}
	if v, ok := d.GetOk("group_id"); ok {
		groupName = v.(string)
	}

	recurse := true
	resp, err := client.Get(ctx, groupName, "children", &recurse, "", managementGroupCacheControl)
	if err != nil {
		if utils.ResponseWasForbidden(resp.Response) {
			return fmt.Errorf("Management Group %q was not found", groupName)
		}

		return fmt.Errorf("Error reading Management Group %q: %+v", d.Id(), err)
	}

	if resp.ID == nil {
		return fmt.Errorf("Client returned an nil ID for Management Group %q", groupName)
	}

	d.SetId(*resp.ID)
	d.Set("name", groupName)
	d.Set("group_id", groupName)

	if props := resp.Properties; props != nil {
		d.Set("display_name", props.DisplayName)

		subscriptionIds, err := flattenArmManagementGroupDataSourceSubscriptionIds(props.Children)
		if err != nil {
			return fmt.Errorf("Error flattening `subscription_ids`: %+v", err)
		}
		d.Set("subscription_ids", subscriptionIds)

		parentId := ""
		if details := props.Details; details != nil {
			if parent := details.Parent; parent != nil {
				if pid := parent.ID; pid != nil {
					parentId = *pid
				}
			}
		}
		d.Set("parent_management_group_id", parentId)
	}

	return nil
}

func flattenArmManagementGroupDataSourceSubscriptionIds(input *[]managementgroups.ChildInfo) (*schema.Set, error) {
	subscriptionIds := &schema.Set{F: schema.HashString}
	if input == nil {
		return subscriptionIds, nil
	}

	for _, child := range *input {
		if child.ID == nil {
			continue
		}

		id, err := parseManagementGroupSubscriptionID(*child.ID)
		if err != nil {
			return nil, fmt.Errorf("Unable to parse child Subscription ID %+v", err)
		}

		if id != nil {
			subscriptionIds.Add(id.subscriptionId)
		}
	}

	return subscriptionIds, nil
}
