"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateDockerfileBaseImageName = void 0;
const dockerfile_ast_1 = require("dockerfile-ast");
const os_1 = require("os");
const instruction_parser_1 = require("./instruction-parser");
const types_1 = require("./types");
/**
 * Updates the image name of the last from stage, after resolving all aliases
 * @param contents Contents of the Dockerfile to update
 * @param newBaseImageName New base image name Dockerfile contents should be updated to
 */
function updateDockerfileBaseImageName(contents, newBaseImageName) {
    const dockerfile = dockerfile_ast_1.DockerfileParser.parse(contents);
    const currentBaseImageName = instruction_parser_1.getDockerfileBaseImageName(dockerfile);
    if (currentBaseImageName === undefined) {
        return {
            contents,
            error: {
                code: types_1.UpdateDockerfileBaseImageNameErrorCode.BASE_IMAGE_NAME_NOT_FOUND,
            },
        };
    }
    const fromRanges = dockerfile
        .getFROMs()
        .filter((from) => from.getImage() === currentBaseImageName)
        .map((from) => from.getImageRange());
    const argRanges = dockerfile
        .getARGs()
        .filter((arg) => { var _a; return ((_a = arg.getProperty()) === null || _a === void 0 ? void 0 : _a.getValue()) === currentBaseImageName; })
        .map((arg) => { var _a; return (_a = arg.getProperty()) === null || _a === void 0 ? void 0 : _a.getValueRange(); });
    const ranges = fromRanges.concat(argRanges);
    if (ranges.length === 0) {
        /**
         * This happens when the image is split over multiple FROM and ARG statements
         * making it difficult to update Dockerfiles that fall into these edge cases.
         * e.g.:
         *    ARG REPO=repo
         *    ARG TAG=tag
         *    FROM ${REPO}:${TAG}
         */
        return {
            contents,
            error: {
                code: types_1.UpdateDockerfileBaseImageNameErrorCode.BASE_IMAGE_NAME_FRAGMENTED,
            },
        };
    }
    const lines = contents.split(os_1.EOL);
    for (const range of ranges) {
        const lineNumber = range.start.line;
        const start = range.start.character;
        const end = range.end.character;
        const content = lines[lineNumber];
        const updated = content.substring(0, start) + newBaseImageName + content.substring(end);
        lines[lineNumber] = updated;
    }
    const updatedContents = lines.join(os_1.EOL);
    const updatedDockerfile = dockerfile_ast_1.DockerfileParser.parse(updatedContents);
    if (dockerfile.getInstructions().length !==
        updatedDockerfile.getInstructions().length) {
        return {
            contents,
            error: {
                code: types_1.UpdateDockerfileBaseImageNameErrorCode.DOCKERFILE_GENERATION_FAILED,
            },
        };
    }
    return {
        contents: updatedContents,
    };
}
exports.updateDockerfileBaseImageName = updateDockerfileBaseImageName;
//# sourceMappingURL=instruction-updater.js.map