"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isMatch = exports.composeFilePayloads = exports.resolveBundleFilePath = exports.resolveBundleFiles = exports.getFileInfo = exports.prepareExtendingBundle = exports.collectBundleFiles = exports.determineBaseDir = exports.collectIgnoreRules = exports.getGlobPatterns = exports.parseFileIgnores = exports.notEmpty = void 0;
const nodePath = __importStar(require("path"));
const fs = __importStar(require("fs"));
const fast_glob_1 = __importDefault(require("@snyk/fast-glob"));
const multimatch_1 = __importDefault(require("multimatch"));
const crypto_1 = __importDefault(require("crypto"));
const lodash_union_1 = __importDefault(require("lodash.union"));
const util_1 = __importDefault(require("util"));
const cache_1 = require("./cache");
const constants_1 = require("./constants");
const isWindows = nodePath.sep === '\\';
const asyncLStat = util_1.default.promisify(fs.lstat);
const lStat = async (path) => {
    let fileStats = null;
    try {
        // eslint-disable-next-line no-await-in-loop
        fileStats = await asyncLStat(path);
    }
    catch (err) {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        if (err.code === 'EACCES' || err.code === 'EPERM') {
            console.log(`${path} is not accessible. Please check permissions and adjust .dcignore file to not even test this file`);
        }
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        if (err.code === 'ENOENT') {
            console.log(`no such file or directory: ${path}`);
        }
    }
    return fileStats;
};
function notEmpty(value) {
    return value !== null && value !== undefined;
}
exports.notEmpty = notEmpty;
const multiMatchOptions = { matchBase: true, dot: true };
const fgOptions = {
    dot: true,
    absolute: true,
    baseNameMatch: true,
    onlyFiles: true,
    suppressErrors: true,
};
function filterSupportedFiles(files, supportedFiles) {
    const patters = getGlobPatterns(supportedFiles);
    return multimatch_1.default(files, patters, multiMatchOptions);
}
function parseIgnoreRulesToGlobs(rules, baseDir) {
    // Mappings from .gitignore format to glob format:
    // `/foo/` => `/foo/**` (meaning: Ignore root (not sub) foo dir and its paths underneath.)
    // `/foo`	=> `/foo/**`, `/foo` (meaning: Ignore root (not sub) file and dir and its paths underneath.)
    // `foo/` => `**/foo/**` (meaning: Ignore (root/sub) foo dirs and their paths underneath.)
    // `foo` => `**/foo/**`, `foo` (meaning: Ignore (root/sub) foo files and dirs and their paths underneath.)
    return rules.reduce((results, rule) => {
        let prefix = '';
        if (rule.startsWith('!')) {
            // eslint-disable-next-line no-param-reassign
            rule = rule.substring(1);
            prefix = '!';
        }
        const startingSlash = rule.startsWith('/');
        const startingGlobstar = rule.startsWith('**');
        const endingSlash = rule.endsWith('/');
        const endingGlobstar = rule.endsWith('**');
        if (startingSlash || startingGlobstar) {
            // case `/foo/`, `/foo` => `{baseDir}/foo/**`
            // case `**/foo/`, `**/foo` => `{baseDir}/**/foo/**`
            if (!endingGlobstar)
                results.push(prefix + nodePath.posix.join(baseDir, rule, '**'));
            // case `/foo` => `{baseDir}/foo`
            // case `**/foo` => `{baseDir}/**/foo`
            // case `/foo/**` => `{baseDir}/foo/**`
            // case `**/foo/**` => `{baseDir}/**/foo/**`
            if (!endingSlash)
                results.push(prefix + nodePath.posix.join(baseDir, rule));
        }
        else {
            // case `foo/`, `foo` => `{baseDir}/**/foo/**`
            if (!endingGlobstar)
                results.push(prefix + nodePath.posix.join(baseDir, '**', rule, '**'));
            // case `foo` => `{baseDir}/**/foo`
            // case `foo/**` => `{baseDir}/**/foo/**`
            if (!endingSlash)
                results.push(prefix + nodePath.posix.join(baseDir, '**', rule));
        }
        return results;
    }, []);
}
function parseFileIgnores(path) {
    let rules = [];
    const dirname = nodePath.dirname(path);
    try {
        const f = fs.readFileSync(path, { encoding: 'utf8' });
        rules = f
            .split('\n')
            .map(l => l.trim())
            .filter(l => !!l && !l.startsWith('#'));
    }
    catch (err) {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        if (err.code === 'EACCES' || err.code === 'EPERM') {
            console.log(`${path} is not accessible. Please check permissions and adjust .dcignore file to not even test this file`);
        }
    }
    return parseIgnoreRulesToGlobs(rules, dirname);
}
exports.parseFileIgnores = parseFileIgnores;
function getGlobPatterns(supportedFiles) {
    return [
        ...supportedFiles.extensions.map(e => `*${e}`),
        ...supportedFiles.configFiles.filter(e => !constants_1.IGNORE_FILES_NAMES.includes(e)),
    ];
}
exports.getGlobPatterns = getGlobPatterns;
async function collectIgnoreRules(dirs, symlinksEnabled = false, fileIgnores = constants_1.IGNORES_DEFAULT) {
    const tasks = dirs.map(async (folder) => {
        const fileStats = await lStat(folder);
        // Check if symlink and exclude if requested
        if (!fileStats || (fileStats.isSymbolicLink() && !symlinksEnabled) || fileStats.isFile())
            return [];
        // Find ignore files inside this directory
        const localIgnoreFiles = await fast_glob_1.default(constants_1.IGNORE_FILES_NAMES.map(i => `*${i}`), {
            ...fgOptions,
            cwd: folder,
            followSymbolicLinks: symlinksEnabled,
        });
        // Read ignore files and merge new patterns
        return lodash_union_1.default(...localIgnoreFiles.map(parseFileIgnores));
    });
    const customRules = await Promise.all(tasks);
    return lodash_union_1.default(fileIgnores, ...customRules);
}
exports.collectIgnoreRules = collectIgnoreRules;
function determineBaseDir(paths) {
    if (paths.length === 1) {
        const path = paths[0];
        const stats = fs.lstatSync(path);
        if (stats.isFile()) {
            return nodePath.dirname(path);
        }
        return path;
    }
    return '';
}
exports.determineBaseDir = determineBaseDir;
async function* searchFiles(patterns, cwd, symlinksEnabled, ignores) {
    const positiveIgnores = ignores.filter(rule => !rule.startsWith('!'));
    const negativeIgnores = ignores.filter(rule => rule.startsWith('!')).map(rule => rule.substring(1));
    // We need to use the ignore rules directly in the stream. Otherwise we would expand all the branches of the file system
    // that should be ignored, leading to performance issues (the parser would look stuck while analyzing each ignored file).
    // However, fast-glob doesn't address the negative rules in the ignore option correctly.
    // As a compromise between correctness and performance, we split the search in two streams, the first one using the
    // extension patterns as a search term and the positive ignore rules in the options, while the second that manually
    // expands those branches that should be excluded from the ignore rules throught the negative ignores as search term
    // and then matches the extensions as a second step to exclude any file that should not be analyzed.
    const positiveSearcher = fast_glob_1.default.stream(patterns, {
        ...fgOptions,
        cwd,
        followSymbolicLinks: symlinksEnabled,
        ignore: positiveIgnores,
    });
    for await (const filePath of positiveSearcher) {
        yield filePath;
    }
    // TODO: This is incorrect because the .gitignore format allows to specify exceptions to previous rules, therefore
    // the separation between positive and negative ignores is incorrect in a scenario with 2+ exeptions like the one below:
    // `node_module/` <= ignores everything in a `node_module` folder and it's relative subfolders
    // `!node_module/my_module/` <= excludes the `my_module` subfolder from the ignore
    // `node_module/my_module/build/` <= re-includes the `build` subfolder in the ignore
    if (negativeIgnores.length) {
        const negativeSearcher = fast_glob_1.default.stream(negativeIgnores, {
            ...fgOptions,
            cwd,
            followSymbolicLinks: symlinksEnabled,
            baseNameMatch: false,
        });
        for await (const filePath of negativeSearcher) {
            if (isMatch(filePath.toString(), patterns.map(p => `**/${p}`)))
                yield filePath;
        }
    }
}
/**
 * Returns bundle files from requested paths
 * */
async function* collectBundleFiles(baseDir, paths, supportedFiles, fileIgnores = constants_1.IGNORES_DEFAULT, maxFileSize = constants_1.MAX_PAYLOAD, symlinksEnabled = false) {
    const cache = new cache_1.Cache(constants_1.CACHE_KEY, baseDir);
    const files = [];
    const dirs = [];
    // Split into directories and files and exclude symlinks if needed
    for (const path of paths) {
        // eslint-disable-next-line no-await-in-loop
        const fileStats = await lStat(path);
        // Check if symlink and exclude if requested
        if (!fileStats || (fileStats.isSymbolicLink() && !symlinksEnabled))
            continue;
        if (fileStats.isFile() && fileStats.size <= maxFileSize) {
            files.push(path);
        }
        else if (fileStats.isDirectory()) {
            dirs.push(path);
        }
    }
    // Scan folders
    const globPatterns = getGlobPatterns(supportedFiles);
    for (const folder of dirs) {
        const searcher = searchFiles(globPatterns, folder, symlinksEnabled, fileIgnores);
        // eslint-disable-next-line no-await-in-loop
        for await (const filePath of searcher) {
            const fileInfo = await getFileInfo(filePath.toString(), baseDir, false, cache);
            if (fileInfo && fileInfo.size <= maxFileSize) {
                yield fileInfo;
            }
        }
    }
    // Sanitize files
    if (files.length) {
        const searcher = searchFiles(filterSupportedFiles(files, supportedFiles), baseDir, symlinksEnabled, fileIgnores);
        for await (const filePath of searcher) {
            const fileInfo = await getFileInfo(filePath.toString(), baseDir, false, cache);
            if (fileInfo && fileInfo.size <= maxFileSize) {
                yield fileInfo;
            }
        }
    }
    cache.save();
}
exports.collectBundleFiles = collectBundleFiles;
async function prepareExtendingBundle(baseDir, files, supportedFiles, fileIgnores = constants_1.IGNORES_DEFAULT, maxFileSize = constants_1.MAX_PAYLOAD, symlinksEnabled = false) {
    let removedFiles = [];
    let bundleFiles = [];
    const cache = new cache_1.Cache(constants_1.CACHE_KEY, baseDir);
    // Filter for supported extensions/files only
    let processingFiles = filterSupportedFiles(files, supportedFiles);
    // Exclude files to be ignored based on ignore rules. We assume here, that ignore rules have not been changed.
    processingFiles = processingFiles.filter(f => !isMatch(f, fileIgnores));
    if (processingFiles.length) {
        // Determine existing files (minus removed)
        const entries = await fast_glob_1.default(processingFiles, {
            ...fgOptions,
            cwd: baseDir,
            followSymbolicLinks: symlinksEnabled,
            objectMode: true,
            stats: true,
        });
        let foundFiles = new Set(); // This initialization is needed to help Typescript checker
        foundFiles = entries.reduce((s, e) => {
            if (e.stats && e.stats.size <= maxFileSize) {
                s.add(e.path);
            }
            return s;
        }, foundFiles);
        removedFiles = processingFiles.reduce((s, p) => {
            if (!foundFiles.has(p)) {
                s.push(getBundleFilePath(p, baseDir));
            }
            return s;
        }, []);
        if (foundFiles.size) {
            bundleFiles = (await Promise.all([...foundFiles].map((p) => getFileInfo(p, baseDir, false, cache)))).filter(notEmpty);
        }
    }
    return {
        files: bundleFiles,
        removedFiles,
    };
}
exports.prepareExtendingBundle = prepareExtendingBundle;
function getBundleFilePath(filePath, baseDir) {
    const relPath = nodePath.relative(baseDir, filePath);
    const posixPath = !isWindows ? relPath : relPath.replace(/\\/g, '/');
    return encodeURI(`/${posixPath}`);
}
async function getFileInfo(filePath, baseDir, withContent = false, cache = null) {
    const fileStats = await lStat(filePath);
    if (fileStats === null) {
        return fileStats;
    }
    const bundlePath = getBundleFilePath(filePath, baseDir);
    const calcHash = (content) => {
        return crypto_1.default.createHash(constants_1.HASH_ALGORITHM).update(content).digest(constants_1.ENCODE_TYPE);
    };
    let fileContent = '';
    let fileHash = '';
    if (!withContent && !!cache) {
        // Try to get hash from cache
        const cachedData = cache.getKey(filePath);
        if (cachedData) {
            if (cachedData[0] === fileStats.size && cachedData[1] === fileStats.mtimeMs) {
                fileHash = cachedData[2];
            }
            else {
                // console.log(`did not match cache for: ${filePath} | ${cachedData} !== ${[fileStats.size, fileStats.mtime]}`);
            }
        }
    }
    if (!fileHash) {
        try {
            fileContent = fs.readFileSync(filePath, { encoding: 'utf8' });
            fileHash = calcHash(fileContent);
            cache === null || cache === void 0 ? void 0 : cache.setKey(filePath, [fileStats.size, fileStats.mtimeMs, fileHash]);
        }
        catch (err) {
            // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
            if (err.code === 'EACCES' || err.code === 'EPERM') {
                console.log(`${filePath} is not accessible. Please check permissions and adjust .dcignore file to not even test this file`);
            }
        }
    }
    return {
        filePath,
        bundlePath,
        size: fileStats.size,
        hash: fileHash,
        content: withContent ? fileContent : undefined,
    };
}
exports.getFileInfo = getFileInfo;
async function resolveBundleFiles(baseDir, bundleMissingFiles) {
    const cache = new cache_1.Cache('.dccache', baseDir);
    const tasks = bundleMissingFiles.map(mf => {
        const filePath = resolveBundleFilePath(baseDir, mf);
        return getFileInfo(filePath, baseDir, true, cache);
    });
    const res = (await Promise.all(tasks)).filter(notEmpty);
    cache.save(true);
    return res;
}
exports.resolveBundleFiles = resolveBundleFiles;
function resolveBundleFilePath(baseDir, bundleFilePath) {
    let relPath = bundleFilePath.slice(1);
    if (isWindows) {
        relPath = relPath.replace(/\//g, '\\');
    }
    return nodePath.resolve(baseDir, decodeURI(relPath));
}
exports.resolveBundleFilePath = resolveBundleFilePath;
function* composeFilePayloads(files, bucketSize = constants_1.MAX_PAYLOAD) {
    const buckets = [{ size: bucketSize, files: [] }];
    let bucketIndex = -1;
    const isLowerSize = (bucket, fileData) => bucket.size >= fileData.size;
    for (let fileData of files) {
        if (fileData.size > bucketSize) {
            // This file is too large. but it should not be here as previosly checked
            fileData = { ...fileData, size: 1, content: '' };
        }
        // Find suitable bucket
        bucketIndex = buckets.findIndex(b => isLowerSize(b, fileData));
        if (bucketIndex === -1) {
            // Create a new bucket
            buckets.push({ size: bucketSize, files: [] });
            bucketIndex = buckets.length - 1;
        }
        buckets[bucketIndex].files.push(fileData);
        buckets[bucketIndex].size -= fileData.size;
        if (buckets[bucketIndex].size < bucketSize * 0.01) {
            yield buckets[bucketIndex].files; // Give bucket to requester
            buckets.splice(bucketIndex); // Remove it as fullfilled
        }
    }
    // Send all left-over buckets
    for (const bucket of buckets.filter(b => b.files.length)) {
        yield bucket.files;
    }
}
exports.composeFilePayloads = composeFilePayloads;
function isMatch(filePath, rules) {
    return !!multimatch_1.default([filePath], rules, { ...multiMatchOptions, matchBase: false }).length;
}
exports.isMatch = isMatch;
//# sourceMappingURL=files.js.map