"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPathDetailsForYamlFile = exports.buildYamlTreeMap = void 0;
var yamlJs = require('yaml-js');
var utils_1 = require("../utils");
var NULL_TAG = 'tag:yaml.org,2002:null';
var STR_TAG = 'tag:yaml.org,2002:str';
var INT_TAG = 'tag:yaml.org,2002:int';
var FLOAT_TAG = 'tag:yaml.org,2002:float';
var BOOL_TAG = 'tag:yaml.org,2002:bool';
var MAP_TAG = 'tag:yaml.org,2002:map';
var SEQ_TAG = 'tag:yaml.org,2002:seq';
var KEY_NODE_INDEX = 0;
var VAL_NODE_INDEX = 1;
var COMMENT_CHAR = '#';
var MULTI_DOC_SEPARATOR = '---';
function buildYamlTreeMap(yamlContent) {
    var yamlTrees = {};
    var docsArray = [];
    try {
        docsArray = yamlJs.compose_all(yamlContent);
    }
    catch (error) {
        throw new Error('failed to compose_all for given yaml');
    }
    // Edge case that yamlJs does not handle -
    // The first lines, before the first doc separator (---) are comments
    // The yamlJs will ignore this lines and will have 1 less document than expected.
    // This will only happen for the first document which document object will not be added for
    if (yamlContent.startsWith(COMMENT_CHAR) &&
        yamlContent.split(MULTI_DOC_SEPARATOR).length === docsArray.length + 1) {
        /* eslint-disable @typescript-eslint/camelcase */
        // Disable calmecale  - object structure from yamlJs
        var commentObject = {
            start_mark: { line: 0, column: 0, pointer: 0, buffer: yamlContent },
            end_mark: { line: 0, column: 0, pointer: 0, buffer: yamlContent },
            style: undefined,
            tag: NULL_TAG,
            unique_id: 'node_0',
            value: '',
        };
        /* eslint-enable @typescript-eslint/camelcase */
        docsArray.unshift(commentObject);
    }
    for (var i = 0; i < docsArray.length; i++) {
        var yamlDoc = convertComposeElementToType(docsArray[i]);
        // Handle case of empty document - the tag will be null
        // No need to build tree for this document
        if (yamlDoc.tag !== NULL_TAG) {
            yamlTrees[i] = buildTree(yamlDoc);
        }
    }
    return yamlTrees;
}
exports.buildYamlTreeMap = buildYamlTreeMap;
function buildTree(yamlDoc) {
    var _a;
    var tree = { nodes: [] };
    if (yamlDoc.tag !== MAP_TAG) {
        throw new Error('Invalid tree - cannot build from map');
    }
    for (var _i = 0, _b = yamlDoc.value; _i < _b.length; _i++) {
        var yamlElementValues = _b[_i];
        var values = [
            convertComposeElementToType(yamlElementValues[KEY_NODE_INDEX]),
            convertComposeElementToType(yamlElementValues[VAL_NODE_INDEX]),
        ];
        (_a = tree.nodes).push.apply(_a, buildTreeBranch(values, []));
    }
    return tree;
}
function buildTreeBranch(yamlElements, path) {
    //We are checking on the valNode - which is the second element in the yamlElements array ([1])
    switch (yamlElements[VAL_NODE_INDEX].tag) {
        case NULL_TAG:
        case STR_TAG:
        case INT_TAG:
        case FLOAT_TAG:
        case BOOL_TAG:
            return buildPrimitiveLeaf(yamlElements);
        case MAP_TAG:
            return buildMapBranch(yamlElements, path);
        case SEQ_TAG:
            return buildSeqBranch(yamlElements, path);
        default:
            return [];
    }
}
function buildPrimitiveLeaf(yamlElements) {
    var keyNode = yamlElements[0], valNode = yamlElements[1];
    var key = keyNode.value;
    var values = valNode.value;
    var lineLocation = utils_1.getLineLocationForYamlElement(keyNode);
    return [
        {
            key: key,
            lineLocation: lineLocation,
            values: values,
        },
    ];
}
function buildMapBranch(yamlElements, path) {
    var keyNode = yamlElements[KEY_NODE_INDEX];
    var fullPath = path.concat([keyNode.value]);
    return [
        getFileStructureNodesForYamlElement(yamlElements, keyNode.value, fullPath),
    ];
}
function buildSeqBranch(yamlElements, path) {
    var keyNode = yamlElements[0], valNode = yamlElements[1];
    var seqNodes = [];
    for (var i = 0; i < valNode.value.length; i++) {
        var nodeElement = convertComposeElementToType(valNode.value[i]);
        var key = keyNode.value + "[" + i + "]";
        var fullPath = path.concat(key);
        if (typeof nodeElement.value === 'string') {
            var lineLocation = utils_1.getLineLocationForYamlElement(nodeElement);
            var node = {
                key: key,
                lineLocation: lineLocation,
                values: nodeElement.value,
            };
            seqNodes.push(node);
        }
        else {
            seqNodes.push(getFileStructureNodesForYamlElement([nodeElement, nodeElement], key, fullPath));
        }
    }
    return seqNodes;
}
function convertComposeElementToType(yamlElement) {
    return {
        id: yamlElement.id,
        tag: yamlElement.tag,
        startMark: {
            line: yamlElement.start_mark.line,
            column: yamlElement.start_mark.column,
            Buffer: yamlElement.start_mark.buffer,
            pointer: yamlElement.start_mark.pointer,
        },
        endMark: {
            line: yamlElement.end_mark.line,
            column: yamlElement.end_mark.column,
            Buffer: yamlElement.end_mark.buffer,
            pointer: yamlElement.end_mark.pointer,
        },
        value: yamlElement.value,
    };
}
function getFileStructureNodesForYamlElement(yamlElements, key, fullPath) {
    var keyNode = yamlElements[0], valNode = yamlElements[1];
    var nodes = [];
    var lineLocation = utils_1.getLineLocationForYamlElement(keyNode);
    for (var _i = 0, _a = valNode.value; _i < _a.length; _i++) {
        var yamlElementValues = _a[_i];
        var values = [
            convertComposeElementToType(yamlElementValues[KEY_NODE_INDEX]),
            convertComposeElementToType(yamlElementValues[VAL_NODE_INDEX]),
        ];
        nodes.push.apply(nodes, buildTreeBranch(values, fullPath));
    }
    return {
        key: key,
        lineLocation: lineLocation,
        values: nodes,
    };
}
function getPathDetailsForYamlFile(path) {
    var firstPath = path[0];
    if (firstPath.includes('[DocId:')) {
        var docId = firstPath.replace('[DocId: ', '').replace(']', '');
        var pathWithoutDocId = path.splice(1);
        return {
            docId: parseInt(docId),
            path: utils_1.removeInputPathPrefix(pathWithoutDocId),
        };
    }
    return {
        docId: 0,
        path: utils_1.removeInputPathPrefix(path),
    };
}
exports.getPathDetailsForYamlFile = getPathDetailsForYamlFile;
//# sourceMappingURL=parser.js.map