"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.execute = void 0;
const child_process_1 = require("child_process");
// Executes a subprocess.
// Resolves successfully on exit code 0 with all the info
// available
async function execute(command, args, options) {
    var _a;
    const spawnOptions = {
        detached: true,
        cwd: (_a = options === null || options === void 0 ? void 0 : options.cwd) !== null && _a !== void 0 ? _a : undefined,
    };
    const fullCommand = `${command} ${args.join(' ')}`;
    const startTime = Date.now();
    let processId;
    try {
        const worker = child_process_1.spawn(command, args, spawnOptions);
        processId = worker.pid;
        return await new Promise((resolve, reject) => {
            var _a, _b;
            let stderr = '';
            let stdout = '';
            (_a = worker.stdout) === null || _a === void 0 ? void 0 : _a.on('data', (data) => {
                stdout += data;
            });
            (_b = worker.stderr) === null || _b === void 0 ? void 0 : _b.on('data', (data) => {
                stderr += data;
            });
            worker.on('error', (e) => {
                reject({
                    stderr,
                    stdout,
                    error: e,
                    duration: Date.now() - startTime,
                    command: fullCommand,
                });
            });
            worker.on('exit', (code) => {
                if (code && code > 0) {
                    resolve({
                        stderr,
                        stdout,
                        duration: Date.now() - startTime,
                        command: fullCommand,
                        exitCode: code,
                    });
                }
                else {
                    resolve({
                        stderr,
                        stdout,
                        duration: Date.now() - startTime,
                        command: fullCommand,
                        exitCode: code,
                    });
                }
            });
        });
    }
    finally {
        if (processId) {
            // Additional anti-zombie protection.
            // Process here should be already stopped.
            try {
                process.kill(processId, 'SIGKILL');
            }
            catch (e) {
                // Process already stopped.
            }
        }
    }
}
exports.execute = execute;
//# sourceMappingURL=sub-process.js.map