"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FailedToDetectJsonConfigError = exports.InvalidYamlFileError = exports.InvalidJsonFileError = exports.UnsupportedFileTypeError = exports.tryParseIacFile = exports.parseFiles = void 0;
const YAML = require("yaml");
const k8s_or_cloudformation_parser_1 = require("./parsers/k8s-or-cloudformation-parser");
const terraform_file_parser_1 = require("./parsers/terraform-file-parser");
const terraform_plan_parser_1 = require("./parsers/terraform-plan-parser");
const types_1 = require("./types");
const analytics = require("../../../../lib/analytics");
const errors_1 = require("../../../../lib/errors");
const error_utils_1 = require("./error-utils");
async function parseFiles(filesData, options = {}) {
    const parsedFiles = [];
    const failedFiles = [];
    for (const fileData of filesData) {
        try {
            parsedFiles.push(...tryParseIacFile(fileData, options));
        }
        catch (err) {
            if (filesData.length === 1) {
                throw err;
            }
            failedFiles.push(generateFailedParsedFile(fileData, err));
        }
    }
    return {
        parsedFiles,
        failedFiles,
    };
}
exports.parseFiles = parseFiles;
function generateFailedParsedFile({ fileType, filePath, fileContent }, err) {
    return {
        err,
        failureReason: err.message,
        fileType,
        filePath,
        fileContent,
        engineType: null,
        jsonContent: null,
    };
}
function parseYAMLOrJSONFileData(fileData) {
    let yamlDocuments;
    try {
        // the YAML library can parse both YAML and JSON content, as well as content with singe/multiple YAMLs
        // by using this library we don't have to disambiguate between these different contents ourselves
        yamlDocuments = YAML.parseAllDocuments(fileData.fileContent).map((doc) => {
            if (doc.errors.length !== 0) {
                throw doc.errors[0];
            }
            return doc.toJSON();
        });
    }
    catch (e) {
        if (fileData.fileType === 'json') {
            throw new InvalidJsonFileError(fileData.filePath);
        }
        else {
            throw new InvalidYamlFileError(fileData.filePath);
        }
    }
    return yamlDocuments;
}
function tryParseIacFile(fileData, options = {}) {
    analytics.add('iac-terraform-plan', false);
    switch (fileData.fileType) {
        case 'yaml':
        case 'yml': {
            const parsedIacFile = parseYAMLOrJSONFileData(fileData);
            try {
                return k8s_or_cloudformation_parser_1.detectConfigType(fileData, parsedIacFile);
            }
            catch (e) {
                if (e instanceof k8s_or_cloudformation_parser_1.HelmFileNotSupportedError) {
                    throw new k8s_or_cloudformation_parser_1.HelmFileNotSupportedError(fileData.filePath);
                }
                else {
                    throw new k8s_or_cloudformation_parser_1.FailedToDetectYamlConfigError(fileData.filePath);
                }
            }
        }
        case 'json': {
            const parsedIacFile = parseYAMLOrJSONFileData(fileData);
            // the Kubernetes file can have more than one JSON object in it
            // but the Terraform plan can only have one
            if (parsedIacFile.length === 1 && terraform_plan_parser_1.isTerraformPlan(parsedIacFile[0])) {
                analytics.add('iac-terraform-plan', true);
                return terraform_plan_parser_1.tryParsingTerraformPlan(fileData, parsedIacFile[0], {
                    isFullScan: options.scan === types_1.TerraformPlanScanMode.FullScan,
                });
            }
            else {
                try {
                    return k8s_or_cloudformation_parser_1.detectConfigType(fileData, parsedIacFile);
                }
                catch (e) {
                    if (e instanceof k8s_or_cloudformation_parser_1.FailedToDetectYamlConfigError) {
                        throw new FailedToDetectJsonConfigError(fileData.filePath);
                    }
                    else {
                        throw e;
                    }
                }
            }
        }
        case 'tf':
            return terraform_file_parser_1.tryParsingTerraformFile(fileData);
        default:
            throw new UnsupportedFileTypeError(fileData.fileType);
    }
}
exports.tryParseIacFile = tryParseIacFile;
class UnsupportedFileTypeError extends errors_1.CustomError {
    constructor(fileType) {
        super('Unsupported file extension');
        this.code = types_1.IaCErrorCodes.UnsupportedFileTypeError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `Unable to process the file with extension ${fileType}. Supported file extensions are tf, yml, yaml & json.\nMore information can be found by running \`snyk iac test --help\` or through our documentation:\nhttps://support.snyk.io/hc/en-us/articles/360012429477-Test-your-Kubernetes-files-with-our-CLI-tool\nhttps://support.snyk.io/hc/en-us/articles/360013723877-Test-your-Terraform-files-with-our-CLI-tool`;
    }
}
exports.UnsupportedFileTypeError = UnsupportedFileTypeError;
class InvalidJsonFileError extends errors_1.CustomError {
    constructor(filename) {
        super('Failed to parse JSON file');
        this.code = types_1.IaCErrorCodes.InvalidJsonFileError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to parse the JSON file "${filename}". Please ensure that it contains properly structured JSON`;
    }
}
exports.InvalidJsonFileError = InvalidJsonFileError;
class InvalidYamlFileError extends errors_1.CustomError {
    constructor(filename) {
        super('Failed to parse YAML file');
        this.code = types_1.IaCErrorCodes.InvalidYamlFileError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to parse the YAML file "${filename}". Please ensure that it contains properly structured YAML`;
    }
}
exports.InvalidYamlFileError = InvalidYamlFileError;
class FailedToDetectJsonConfigError extends errors_1.CustomError {
    constructor(filename) {
        super('Failed to detect either a Kubernetes file, a CloudFormation file or a Terraform Plan, missing required fields');
        this.code = types_1.IaCErrorCodes.FailedToDetectJsonConfigError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to detect whether the JSON file "${filename}" is either a valid Kubernetes file, CloudFormation file or a Terraform Plan. For Kubernetes it is missing the following fields: "${k8s_or_cloudformation_parser_1.REQUIRED_K8S_FIELDS.join('", "')}".  For CloudFormation required fields are: "Resources". For Terraform Plan it was expected to contain fields "planned_values.root_module" and "resource_changes". Please contact support@snyk.io, if possible with a redacted version of the file`;
    }
}
exports.FailedToDetectJsonConfigError = FailedToDetectJsonConfigError;
//# sourceMappingURL=file-parser.js.map