"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapIacTestResponseToSarifResults = exports.extractReportingDescriptor = exports.createSarifOutputForIac = exports.capitalizePackageManager = exports.getIacDisplayErrorFileOutput = exports.getIacDisplayedOutput = void 0;
const chalk_1 = require("chalk");
const Debug = require("debug");
const pathLib = require("path");
const formatters_1 = require("./formatters");
const remediation_based_format_issues_1 = require("./formatters/remediation-based-format-issues");
const legacy_format_issue_1 = require("./formatters/legacy-format-issue");
const legacy_1 = require("../../../lib/snyk-test/legacy");
const common_1 = require("../../../lib/snyk-test/common");
const upperFirst = require("lodash.upperfirst");
const debug = Debug('iac-output');
function formatIacIssue(issue, isNew, path) {
    const newBadge = isNew ? ' (new)' : '';
    const name = issue.subType ? ` in ${chalk_1.default.bold(issue.subType)}` : '';
    let introducedBy = '';
    if (path) {
        // In this mode, we show only one path by default, for compactness
        const pathStr = remediation_based_format_issues_1.printPath(path);
        introducedBy = `\n    introduced by ${pathStr}`;
    }
    const severityColor = common_1.getSeveritiesColour(issue.severity);
    return (severityColor.colorFunc(`  ✗ ${chalk_1.default.bold(issue.title)}${newBadge} [${legacy_format_issue_1.titleCaseText(issue.severity)} Severity]`) +
        ` [${issue.id}]` +
        name +
        introducedBy +
        '\n');
}
function getIacDisplayedOutput(iacTest, testedInfoText, meta, prefix) {
    const issuesTextArray = [
        chalk_1.default.bold.white('\nInfrastructure as code issues:'),
    ];
    const NotNew = false;
    const issues = iacTest.result.cloudConfigResults;
    debug(`iac display output - ${issues.length} issues`);
    issues
        .sort((a, b) => formatters_1.getSeverityValue(b.severity) - formatters_1.getSeverityValue(a.severity))
        .forEach((issue) => {
        issuesTextArray.push(formatIacIssue(issue, NotNew, issue.cloudConfigPath));
    });
    const issuesInfoOutput = [];
    debug(`Iac display output - ${issuesTextArray.length} issues text`);
    if (issuesTextArray.length > 0) {
        issuesInfoOutput.push(issuesTextArray.join('\n'));
    }
    let body = issuesInfoOutput.join('\n\n') + '\n\n' + meta;
    const vulnCountText = `found ${issues.length} issues`;
    const summary = testedInfoText + ', ' + chalk_1.default.red.bold(vulnCountText);
    body = body + '\n\n' + summary;
    return prefix + body;
}
exports.getIacDisplayedOutput = getIacDisplayedOutput;
function getIacDisplayErrorFileOutput(iacFileResult) {
    const fileName = pathLib.basename(iacFileResult.filePath);
    return `

-------------------------------------------------------

Testing ${fileName}...

${iacFileResult.failureReason}`;
}
exports.getIacDisplayErrorFileOutput = getIacDisplayErrorFileOutput;
function capitalizePackageManager(type) {
    switch (type) {
        case 'k8sconfig': {
            return 'Kubernetes';
        }
        case 'helmconfig': {
            return 'Helm';
        }
        case 'terraformconfig': {
            return 'Terraform';
        }
        case 'cloudformationconfig': {
            return 'CloudFormation';
        }
        default: {
            return 'Infrastracture as Code';
        }
    }
}
exports.capitalizePackageManager = capitalizePackageManager;
// Used to reference the base path in results.
const PROJECT_ROOT_KEY = 'PROJECTROOT';
function createSarifOutputForIac(iacTestResponses) {
    const issues = iacTestResponses.reduce((collect, res) => {
        if (res.result) {
            // FIXME: For single file tests the targetFile includes the full file
            // path, for directory tests only the filename is returned and we need
            // too build the path manually.
            const targetPath = res.targetFile.startsWith(res.path)
                ? pathLib.join(res.targetFile)
                : pathLib.join(res.path, res.targetFile);
            const mapped = res.result.cloudConfigResults.map((issue) => ({
                issue,
                targetPath,
            }));
            collect.push(...mapped);
        }
        return collect;
    }, []);
    const tool = {
        driver: {
            name: 'Snyk Infrastructure as Code',
            rules: extractReportingDescriptor(issues),
        },
    };
    return {
        version: '2.1.0',
        runs: [
            {
                // https://docs.oasis-open.org/sarif/sarif/v2.1.0/os/sarif-v2.1.0-os.html#_Toc34317498
                originalUriBaseIds: {
                    [PROJECT_ROOT_KEY]: {
                        // The base path is the current working directory.
                        // See: https://github.com/snyk/snyk/blob/6408c730c88902f0f6a00e732ee83c812903f240/src/lib/iac/detect-iac.ts#L94
                        uri: 'file://' + pathLib.join(pathLib.resolve('.'), '/'),
                        description: {
                            text: 'The root directory for all project files.',
                        },
                    },
                },
                tool,
                results: mapIacTestResponseToSarifResults(issues),
            },
        ],
    };
}
exports.createSarifOutputForIac = createSarifOutputForIac;
function getIssueLevel(severity) {
    return severity === legacy_1.SEVERITY.HIGH ? 'error' : 'warning';
}
const iacTypeToText = {
    k8s: 'Kubernetes',
    terraform: 'Terraform',
};
function extractReportingDescriptor(results) {
    const tool = {};
    results.forEach(({ issue }) => {
        if (tool[issue.id]) {
            return;
        }
        tool[issue.id] = {
            id: issue.id,
            shortDescription: {
                text: `${upperFirst(issue.severity)} severity - ${issue.title}`,
            },
            fullDescription: {
                text: `${iacTypeToText[issue.type]} ${issue.subType}`,
            },
            help: {
                text: `The issue is... \n${issue.iacDescription.issue}\n\n The impact of this is... \n ${issue.iacDescription.impact}\n\n You can resolve this by... \n${issue.iacDescription.resolve}`.replace(/^\s+/g, ''),
                markdown: `**The issue is...** \n${issue.iacDescription.issue}\n\n **The impact of this is...** \n ${issue.iacDescription.impact}\n\n **You can resolve this by...** \n${issue.iacDescription.resolve}`.replace(/^\s+/g, ''),
            },
            defaultConfiguration: {
                level: getIssueLevel(issue.severity),
            },
            properties: {
                tags: ['security', `${issue.type}/${issue.subType}`],
                documentation: issue.documentation,
            },
        };
    });
    return Object.values(tool);
}
exports.extractReportingDescriptor = extractReportingDescriptor;
function mapIacTestResponseToSarifResults(issues) {
    return issues.map(({ targetPath, issue }) => ({
        ruleId: issue.id,
        message: {
            text: `This line contains a potential ${issue.severity} severity misconfiguration affecting the ${iacTypeToText[issue.type]} ${issue.subType}`,
        },
        locations: [
            {
                physicalLocation: {
                    artifactLocation: {
                        uri: targetPath,
                        uriBaseId: PROJECT_ROOT_KEY,
                    },
                    region: {
                        startLine: issue.lineNumber,
                    },
                },
            },
        ],
    }));
}
exports.mapIacTestResponseToSarifResults = mapIacTestResponseToSarifResults;
//# sourceMappingURL=iac-output.js.map