"use strict";
const Debug = require("debug");
const pathLib = require("path");
const cloneDeep = require('lodash.clonedeep');
const assign = require('lodash.assign');
const chalk_1 = require("chalk");
const snyk = require("../../../lib");
const is_ci_1 = require("../../../lib/is-ci");
const detect_1 = require("../../../lib/detect");
const types_1 = require("../../commands/types");
const iac_test_result_1 = require("../../../lib/snyk-test/iac-test-result");
const formatters_1 = require("./formatters");
const utils = require("./utils");
const iac_output_1 = require("./iac-output");
const ecosystems_1 = require("../../../lib/ecosystems");
const is_multi_project_scan_1 = require("../../../lib/is-multi-project-scan");
const constants_1 = require("../../../lib/iac/constants");
const vuln_helpers_1 = require("./vuln-helpers");
const format_test_results_1 = require("./formatters/format-test-results");
const iac_test_shim_1 = require("./iac-test-shim");
const validate_credentials_1 = require("./validate-credentials");
const validate_test_options_1 = require("./validate-test-options");
const set_default_test_options_1 = require("./set-default-test-options");
const process_command_args_1 = require("../process-command-args");
const format_test_error_1 = require("./format-test-error");
const debug = Debug('snyk-test');
const SEPARATOR = '\n-------------------------------------------------------\n';
// TODO: avoid using `as any` whenever it's possible
async function test(...args) {
    const { options: originalOptions, paths } = process_command_args_1.processCommandArgs(...args);
    const options = set_default_test_options_1.setDefaultTestOptions(originalOptions);
    validate_test_options_1.validateTestOptions(options);
    validate_credentials_1.validateCredentials(options);
    const ecosystem = ecosystems_1.getEcosystemForTest(options);
    if (ecosystem) {
        try {
            const commandResult = await ecosystems_1.testEcosystem(ecosystem, paths, options);
            return commandResult;
        }
        catch (error) {
            if (error instanceof Error) {
                throw error;
            }
            else {
                throw new Error(error);
            }
        }
    }
    const resultOptions = [];
    const results = [];
    // Holds an array of scanned file metadata for output.
    let iacScanFailures;
    // Promise waterfall to test all other paths sequentially
    for (const path of paths) {
        // Create a copy of the options so a specific test can
        // modify them i.e. add `options.file` etc. We'll need
        // these options later.
        const testOpts = cloneDeep(options);
        testOpts.path = path;
        testOpts.projectName = testOpts['project-name'];
        let res;
        try {
            if (options.iac) {
                // this path is an experimental feature feature for IaC which does issue scanning locally without sending files to our Backend servers.
                // once ready for GA, it is aimed to deprecate our remote-processing model, so IaC file scanning in the CLI is done locally.
                const { results, failures } = await iac_test_shim_1.test(path, testOpts);
                res = results;
                iacScanFailures = failures;
            }
            else {
                res = await snyk.test(path, testOpts);
            }
        }
        catch (error) {
            // not throwing here but instead returning error response
            // for legacy flow reasons.
            res = format_test_error_1.formatTestError(error);
        }
        // Not all test results are arrays in order to be backwards compatible
        // with scripts that use a callback with test. Coerce results/errors to be arrays
        // and add the result options to each to be displayed
        const resArray = Array.isArray(res) ? res : [res];
        for (let i = 0; i < resArray.length; i++) {
            const pathWithOptionalProjectName = utils.getPathWithOptionalProjectName(path, resArray[i]);
            results.push(assign(resArray[i], { path: pathWithOptionalProjectName }));
            // currently testOpts are identical for each test result returned even if it's for multiple projects.
            // we want to return the project names, so will need to be crafty in a way that makes sense.
            if (!testOpts.projectNames) {
                resultOptions.push(testOpts);
            }
            else {
                resultOptions.push(assign(cloneDeep(testOpts), {
                    projectName: testOpts.projectNames[i],
                }));
            }
        }
    }
    const vulnerableResults = results.filter((res) => (res.vulnerabilities && res.vulnerabilities.length) ||
        (res.result &&
            res.result.cloudConfigResults &&
            res.result.cloudConfigResults.length));
    const errorResults = results.filter((res) => res instanceof Error);
    const notSuccess = errorResults.length > 0;
    const foundVulnerabilities = vulnerableResults.length > 0;
    // resultOptions is now an array of 1 or more options used for
    // the tests results is now an array of 1 or more test results
    // values depend on `options.json` value - string or object
    const errorMappedResults = !options.iac
        ? format_test_results_1.createErrorMappedResultsForJsonOutput(results)
        : results.map(iac_test_result_1.mapIacTestResult);
    // backwards compat - strip array IFF only one result
    const jsonData = errorMappedResults.length === 1
        ? errorMappedResults[0]
        : errorMappedResults;
    const { stdout: dataToSend, stringifiedData, stringifiedJsonData, stringifiedSarifData, } = format_test_results_1.extractDataToSendFromResults(results, jsonData, options);
    if (options.json || options.sarif) {
        // if all results are ok (.ok == true)
        if (errorMappedResults.every((res) => res.ok)) {
            return types_1.TestCommandResult.createJsonTestCommandResult(stringifiedData, stringifiedJsonData, stringifiedSarifData);
        }
        const err = new Error(stringifiedData);
        if (foundVulnerabilities) {
            if (options.failOn) {
                const fail = shouldFail(vulnerableResults, options.failOn);
                if (!fail) {
                    // return here to prevent failure
                    return types_1.TestCommandResult.createJsonTestCommandResult(stringifiedData, stringifiedJsonData, stringifiedSarifData);
                }
            }
            err.code = 'VULNS';
            const dataToSendNoVulns = dataToSend;
            delete dataToSendNoVulns.vulnerabilities;
            err.jsonNoVulns = dataToSendNoVulns;
        }
        if (notSuccess) {
            // Take the code of the first problem to go through error
            // translation.
            // Note: this is done based on the logic done below
            // for non-json/sarif outputs, where we take the code of
            // the first error.
            err.code = errorResults[0].code;
        }
        err.json = stringifiedData;
        err.jsonStringifiedResults = stringifiedJsonData;
        err.sarifStringifiedResults = stringifiedSarifData;
        throw err;
    }
    let response = results
        .map((result, i) => {
        return displayResult(results[i], resultOptions[i], result.foundProjectCount);
    })
        .join(`\n${SEPARATOR}`);
    if (notSuccess) {
        debug(`Failed to test ${errorResults.length} projects, errors:`);
        errorResults.forEach((err) => {
            const errString = err.stack ? err.stack.toString() : err.toString();
            debug('error: %s', errString);
        });
    }
    let summaryMessage = '';
    let errorResultsLength = errorResults.length;
    if (options.iac && iacScanFailures) {
        errorResultsLength = iacScanFailures.length || errorResults.length;
        for (const reason of iacScanFailures) {
            response += chalk_1.default.bold.red(iac_output_1.getIacDisplayErrorFileOutput(reason));
        }
    }
    if (results.length > 1) {
        const projects = results.length === 1 ? 'project' : 'projects';
        summaryMessage =
            `\n\n\nTested ${results.length} ${projects}` +
                formatters_1.summariseVulnerableResults(vulnerableResults, options) +
                formatters_1.summariseErrorResults(errorResultsLength) +
                '\n';
    }
    if (notSuccess) {
        response += chalk_1.default.bold.red(summaryMessage);
        const error = new Error(response);
        // take the code of the first problem to go through error
        // translation
        // HACK as there can be different errors, and we pass only the
        // first one
        error.code = errorResults[0].code;
        error.userMessage = errorResults[0].userMessage;
        error.strCode = errorResults[0].strCode;
        throw error;
    }
    if (foundVulnerabilities) {
        if (options.failOn) {
            const fail = shouldFail(vulnerableResults, options.failOn);
            if (!fail) {
                // return here to prevent throwing failure
                response += chalk_1.default.bold.green(summaryMessage);
                return types_1.TestCommandResult.createHumanReadableTestCommandResult(response, stringifiedJsonData, stringifiedSarifData);
            }
        }
        response += chalk_1.default.bold.red(summaryMessage);
        const error = new Error(response);
        // take the code of the first problem to go through error
        // translation
        // HACK as there can be different errors, and we pass only the
        // first one
        error.code = vulnerableResults[0].code || 'VULNS';
        error.userMessage = vulnerableResults[0].userMessage;
        error.jsonStringifiedResults = stringifiedJsonData;
        error.sarifStringifiedResults = stringifiedSarifData;
        throw error;
    }
    response += chalk_1.default.bold.green(summaryMessage);
    return types_1.TestCommandResult.createHumanReadableTestCommandResult(response, stringifiedJsonData, stringifiedSarifData);
}
function shouldFail(vulnerableResults, failOn) {
    // find reasons not to fail
    if (failOn === 'all') {
        return vuln_helpers_1.hasFixes(vulnerableResults);
    }
    if (failOn === 'upgradable') {
        return vuln_helpers_1.hasUpgrades(vulnerableResults);
    }
    if (failOn === 'patchable') {
        return vuln_helpers_1.hasPatches(vulnerableResults);
    }
    // should fail by default when there are vulnerable results
    return vulnerableResults.length > 0;
}
function displayResult(res, options, foundProjectCount) {
    const meta = formatters_1.formatTestMeta(res, options);
    const dockerAdvice = formatters_1.dockerRemediationForDisplay(res);
    const projectType = res.packageManager || options.packageManager;
    const localPackageTest = detect_1.isLocalFolder(options.path);
    let testingPath = options.path;
    if (options.iac && res.targetFile) {
        testingPath = pathLib.basename(res.targetFile);
    }
    const prefix = chalk_1.default.bold.white('\nTesting ' + testingPath + '...\n\n');
    // handle errors by extracting their message
    if (res instanceof Error) {
        return prefix + res.message;
    }
    const issuesText = res.licensesPolicy ||
        constants_1.TEST_SUPPORTED_IAC_PROJECTS.includes(projectType)
        ? 'issues'
        : 'vulnerabilities';
    let pathOrDepsText = '';
    if (res.dependencyCount) {
        pathOrDepsText += res.dependencyCount + ' dependencies';
    }
    else if (options.iac && res.targetFile) {
        pathOrDepsText += pathLib.basename(res.targetFile);
    }
    else {
        pathOrDepsText += options.path;
    }
    const testedInfoText = `Tested ${pathOrDepsText} for known ${issuesText}`;
    let multiProjAdvice = '';
    const advertiseGradleSubProjectsCount = projectType === 'gradle' &&
        !options['gradle-sub-project'] &&
        !options.allProjects &&
        foundProjectCount;
    if (advertiseGradleSubProjectsCount) {
        multiProjAdvice = chalk_1.default.bold.white(`\n\nTip: This project has multiple sub-projects (${foundProjectCount}), ` +
            'use --all-sub-projects flag to scan all sub-projects.');
    }
    const advertiseAllProjectsCount = projectType !== 'gradle' &&
        !is_multi_project_scan_1.isMultiProjectScan(options) &&
        foundProjectCount;
    if (advertiseAllProjectsCount) {
        multiProjAdvice = chalk_1.default.bold.white(`\n\nTip: Detected multiple supported manifests (${foundProjectCount}), ` +
            'use --all-projects to scan all of them at once.');
    }
    // OK  => no vulns found, return
    if (res.ok && res.vulnerabilities.length === 0) {
        const vulnPathsText = options.showVulnPaths
            ? 'no vulnerable paths found.'
            : 'none were found.';
        const summaryOKText = chalk_1.default.green(`✓ ${testedInfoText}, ${vulnPathsText}`);
        const nextStepsText = localPackageTest
            ? '\n\nNext steps:' +
                '\n- Run `snyk monitor` to be notified ' +
                'about new related vulnerabilities.' +
                '\n- Run `snyk test` as part of ' +
                'your CI/test.'
            : '';
        // user tested a package@version and got 0 vulns back, but there were dev deps
        // to consider
        // to consider
        const snykPackageTestTip = !(options.docker ||
            localPackageTest ||
            options.dev)
            ? '\n\nTip: Snyk only tests production dependencies by default. You can try re-running with the `--dev` flag.'
            : '';
        const dockerCTA = format_test_results_1.dockerUserCTA(options);
        return (prefix +
            meta +
            '\n\n' +
            summaryOKText +
            multiProjAdvice +
            (is_ci_1.isCI()
                ? ''
                : dockerAdvice + nextStepsText + snykPackageTestTip + dockerCTA));
    }
    if (constants_1.TEST_SUPPORTED_IAC_PROJECTS.includes(res.packageManager)) {
        return iac_output_1.getIacDisplayedOutput(res, testedInfoText, meta, prefix);
    }
    // NOT OK => We found some vulns, let's format the vulns info
    return format_test_results_1.getDisplayedOutput(res, options, testedInfoText, localPackageTest, projectType, meta, prefix, multiProjAdvice, dockerAdvice);
}
module.exports = test;
//# sourceMappingURL=index.js.map