"use strict";
/*
  We are collecting Snyk CLI usage in our official integrations

  We distinguish them by either:
  - Setting SNYK_INTEGRATION_NAME or SNYK_INTEGRATION_VERSION in environment when CLI is run
  - passing an --integration-name or --integration-version flags on CLI invocation

  Integration name is validated with a list
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCommandVersion = exports.isInstalled = exports.validateHomebrew = exports.isHomebrew = exports.validateScoopManifestFile = exports.isScoop = exports.getIntegrationEnvironmentVersion = exports.getIntegrationEnvironment = exports.getIntegrationVersion = exports.getIntegrationName = exports.INTEGRATION_ENVIRONMENT_VERSION_ENVVAR = exports.INTEGRATION_ENVIRONMENT_ENVVAR = exports.INTEGRATION_VERSION_ENVVAR = exports.INTEGRATION_NAME_ENVVAR = void 0;
const debug = require('debug')('snyk');
const fs = require("fs");
const semver = require("semver");
const path_1 = require("path");
const { exec } = require('child_process');
exports.INTEGRATION_NAME_ENVVAR = 'SNYK_INTEGRATION_NAME';
exports.INTEGRATION_VERSION_ENVVAR = 'SNYK_INTEGRATION_VERSION';
exports.INTEGRATION_ENVIRONMENT_ENVVAR = 'SNYK_INTEGRATION_ENVIRONMENT';
exports.INTEGRATION_ENVIRONMENT_VERSION_ENVVAR = 'SNYK_INTEGRATION_ENVIRONMENT_VERSION';
var TrackedIntegration;
(function (TrackedIntegration) {
    // Distribution builds/packages
    TrackedIntegration["NPM"] = "NPM";
    TrackedIntegration["STANDALONE"] = "STANDALONE";
    // tracked by passing envvar on CLI invocation
    TrackedIntegration["HOMEBREW"] = "HOMEBREW";
    TrackedIntegration["SCOOP"] = "SCOOP";
    // Our Docker images - tracked by passing envvar on CLI invocation
    TrackedIntegration["DOCKER_SNYK_CLI"] = "DOCKER_SNYK_CLI";
    TrackedIntegration["DOCKER_SNYK"] = "DOCKER_SNYK";
    // IDE plugins - tracked by passing flag or envvar on CLI invocation
    TrackedIntegration["JETBRAINS_IDE"] = "JETBRAINS_IDE";
    TrackedIntegration["ECLIPSE"] = "ECLIPSE";
    TrackedIntegration["VISUAL_STUDIO"] = "VISUAL_STUDIO";
    TrackedIntegration["VS_CODE_VULN_COST"] = "VS_CODE_VULN_COST";
    // CI - tracked by passing flag or envvar on CLI invocation
    TrackedIntegration["JENKINS"] = "JENKINS";
    TrackedIntegration["TEAMCITY"] = "TEAMCITY";
    TrackedIntegration["BITBUCKET_PIPELINES"] = "BITBUCKET_PIPELINES";
    TrackedIntegration["AZURE_PIPELINES"] = "AZURE_PIPELINES";
    TrackedIntegration["CIRCLECI_ORB"] = "CIRCLECI_ORB";
    TrackedIntegration["GITHUB_ACTIONS"] = "GITHUB_ACTIONS";
    TrackedIntegration["MAVEN_PLUGIN"] = "MAVEN_PLUGIN";
    // Partner integrations - tracked by passing envvar on CLI invocation
    TrackedIntegration["DOCKER_DESKTOP"] = "DOCKER_DESKTOP";
    // DevRel integrations and plugins
    // Netlify plugin: https://github.com/snyk-labs/netlify-plugin-snyk
    TrackedIntegration["NETLIFY_PLUGIN"] = "NETLIFY_PLUGIN";
})(TrackedIntegration || (TrackedIntegration = {}));
exports.getIntegrationName = (args) => {
    var _a;
    const maybeHomebrew = isHomebrew() ? 'HOMEBREW' : '';
    const maybeScoop = isScoop() ? 'SCOOP' : '';
    const integrationName = (((_a = args[0]) === null || _a === void 0 ? void 0 : _a.integrationName) || // Integration details passed through CLI flag
        process.env[exports.INTEGRATION_NAME_ENVVAR] ||
        maybeHomebrew ||
        maybeScoop ||
        '').toUpperCase();
    if (integrationName in TrackedIntegration) {
        return integrationName;
    }
    return '';
};
exports.getIntegrationVersion = (args) => {
    var _a;
    return ((_a = args[0]) === null || _a === void 0 ? void 0 : _a.integrationVersion) ||
        process.env[exports.INTEGRATION_VERSION_ENVVAR] ||
        '';
};
exports.getIntegrationEnvironment = (args) => {
    var _a;
    return ((_a = args[0]) === null || _a === void 0 ? void 0 : _a.integrationEnvironment) ||
        process.env[exports.INTEGRATION_ENVIRONMENT_ENVVAR] ||
        '';
};
exports.getIntegrationEnvironmentVersion = (args) => {
    var _a;
    return ((_a = args[0]) === null || _a === void 0 ? void 0 : _a.integrationEnvironmentVersion) ||
        process.env[exports.INTEGRATION_ENVIRONMENT_VERSION_ENVVAR] ||
        '';
};
function isScoop() {
    const currentProcessPath = process.execPath;
    const looksLikeScoop = currentProcessPath.includes('snyk-win.exe') &&
        currentProcessPath.includes('scoop');
    if (looksLikeScoop) {
        return validateScoopManifestFile(currentProcessPath);
    }
    else {
        return false;
    }
}
exports.isScoop = isScoop;
function validateScoopManifestFile(snykExecutablePath) {
    // If this really is installed with scoop, there should be a `manifest.json` file adjacent to the running CLI executable (`snyk-win.exe`) which
    // we can look at for further validation that this really is from scoop.
    try {
        const snykScoopManifiestPath = snykExecutablePath.replace('snyk-win.exe', 'manifest.json');
        if (fs.existsSync(snykScoopManifiestPath)) {
            const manifestJson = JSON.parse(fs.readFileSync(snykScoopManifiestPath, 'utf8'));
            const url = manifestJson.url;
            if (url.startsWith('https://github.com/snyk/snyk') &&
                url.endsWith('snyk-win.exe')) {
                return true;
            }
        }
    }
    catch (error) {
        debug('Error validating scoop manifest file', error);
    }
    return false;
}
exports.validateScoopManifestFile = validateScoopManifestFile;
function isHomebrew() {
    const currentProcessPath = process.execPath;
    const isHomebrewPath = currentProcessPath.includes('/Cellar/snyk/');
    if (isHomebrewPath) {
        return validateHomebrew(currentProcessPath);
    }
    else {
        return false;
    }
}
exports.isHomebrew = isHomebrew;
function validateHomebrew(snykExecutablePath) {
    try {
        const expectedFormulaFilePath = path_1.join(snykExecutablePath, '../../.brew/snyk.rb');
        const formulaFileExists = fs.existsSync(expectedFormulaFilePath);
        return formulaFileExists;
    }
    catch (error) {
        debug('Error checking for Homebrew Formula file', error);
    }
    return false;
}
exports.validateHomebrew = validateHomebrew;
function runCommand(cmd) {
    return new Promise((resolve) => {
        exec(cmd, (error, stdout, stderr) => {
            if (error) {
                debug("Error trying to get program's version", error);
            }
            return resolve(stdout ? stdout : stderr);
        });
    });
}
async function isInstalled(commandToCheck) {
    let whichCommand = 'which';
    const os = process.platform;
    if (os === 'win32') {
        whichCommand = 'where';
    }
    else if (os === 'android') {
        whichCommand = 'adb shell which';
    }
    try {
        await runCommand(`${whichCommand} ${commandToCheck}`);
    }
    catch (error) {
        return false;
    }
    return true;
}
exports.isInstalled = isInstalled;
// This only works for programs that output a valid version when called with --version flag!
async function getCommandVersion(commandToCheck) {
    const isCommandInstalled = await isInstalled(commandToCheck);
    if (isCommandInstalled) {
        try {
            let version = await runCommand(`${commandToCheck} --version`);
            // Remove newline
            version = version.trim();
            version = semver.valid(version);
            if (version !== null) {
                return version;
            }
        }
        catch (error) {
            return undefined;
        }
    }
    return undefined;
}
exports.getCommandVersion = getCommandVersion;
//# sourceMappingURL=analytics-sources.js.map