"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPrefix = exports.getMeta = exports.getCodeDisplayedOutput = void 0;
const Debug = require("debug");
const chalk_1 = require("chalk");
const common_1 = require("../../../snyk-test/common");
const right_pad_1 = require("../../../right-pad");
const debug = Debug('code-output');
function getCodeDisplayedOutput(codeTest, meta, prefix) {
    let issues = {};
    if (codeTest.runs[0].results) {
        const results = codeTest.runs[0].results;
        const rulesMap = getRulesMap(codeTest.runs[0].tool.driver.rules || []);
        issues = getIssues(results, rulesMap);
    }
    const issuesText = issues.low.join('') + issues.medium.join('') + issues.high.join('');
    const summaryOKText = chalk_1.default.green('✔ Test completed');
    const codeIssueSummary = getCodeIssuesSummary(issues);
    return (prefix +
        issuesText +
        '\n' +
        summaryOKText +
        '\n\n' +
        meta +
        '\n\n' +
        codeIssueSummary);
}
exports.getCodeDisplayedOutput = getCodeDisplayedOutput;
function getCodeIssuesSummary(issues) {
    const lowSeverityText = issues.low.length
        ? common_1.getLegacySeveritiesColour(common_1.SEVERITY.LOW).colorFunc(` ${issues.low.length} [Low] `)
        : '';
    const mediumSeverityText = issues.medium.length
        ? common_1.getLegacySeveritiesColour(common_1.SEVERITY.MEDIUM).colorFunc(` ${issues.medium.length} [Medium] `)
        : '';
    const highSeverityText = issues.high.length
        ? common_1.getLegacySeveritiesColour(common_1.SEVERITY.HIGH).colorFunc(`${issues.high.length} [High] `)
        : '';
    const codeIssueCount = issues.low.length + issues.medium.length + issues.high.length;
    const codeIssueFound = `${codeIssueCount} Code issue${codeIssueCount > 0 ? 's' : ''} found`;
    const issuesBySeverityText = highSeverityText + mediumSeverityText + lowSeverityText;
    const vulnPathsText = chalk_1.default.green('✔ Awesome! No issues were found.');
    return codeIssueCount > 0
        ? codeIssueFound + '\n' + issuesBySeverityText
        : vulnPathsText;
}
function getIssues(results, rulesMap) {
    const issuesInit = {
        low: [],
        medium: [],
        high: [],
    };
    const issues = results.reduce((acc, res) => {
        var _a, _b;
        if ((_a = res.locations) === null || _a === void 0 ? void 0 : _a.length) {
            const location = res.locations[0].physicalLocation;
            if (res.level && (location === null || location === void 0 ? void 0 : location.artifactLocation) && (location === null || location === void 0 ? void 0 : location.region)) {
                const severity = sarifToSeverityLevel(res.level);
                const ruleId = res.ruleId;
                if (!(ruleId in rulesMap)) {
                    debug('Rule ID does not exist in the rules list');
                }
                const ruleName = ((_b = rulesMap[ruleId].shortDescription) === null || _b === void 0 ? void 0 : _b.text) || rulesMap[ruleId].name;
                const ruleIdSeverityText = common_1.getLegacySeveritiesColour(severity.toLowerCase()).colorFunc(` ✗ [${severity}] ${ruleName}`);
                const artifactLocationUri = location.artifactLocation.uri;
                const startLine = location.region.startLine;
                const markdown = res.message.markdown;
                const title = ruleIdSeverityText;
                const path = `    Path: ${artifactLocationUri}, line ${startLine}`;
                const info = `    Info: ${markdown}`;
                acc[severity.toLowerCase()].push(`${title} \n ${path} \n ${info}\n\n`);
            }
        }
        return acc;
    }, issuesInit);
    return issues;
}
function getRulesMap(rules) {
    const rulesMapByID = rules.reduce((acc, rule) => {
        acc[rule.id] = rule;
        return acc;
    }, {});
    return rulesMapByID;
}
function sarifToSeverityLevel(sarifConfigurationLevel) {
    const severityLevel = {
        note: 'Low',
        warning: 'Medium',
        error: 'High',
    };
    return severityLevel[sarifConfigurationLevel];
}
function getMeta(options, path) {
    const padToLength = 19; // chars to align
    const orgName = options.org;
    const projectPath = options.path || path;
    const meta = [
        chalk_1.default.bold(right_pad_1.rightPadWithSpaces('Organization: ', padToLength)) + orgName,
    ];
    meta.push(chalk_1.default.bold(right_pad_1.rightPadWithSpaces('Test type: ', padToLength)) +
        'Static code analysis');
    meta.push(chalk_1.default.bold(right_pad_1.rightPadWithSpaces('Project path: ', padToLength)) + projectPath);
    return meta.join('\n');
}
exports.getMeta = getMeta;
function getPrefix(path) {
    return chalk_1.default.bold.white('\nTesting ' + path + ' ...\n\n');
}
exports.getPrefix = getPrefix;
//# sourceMappingURL=output-format.js.map