import _isPlainObject from "lodash/isPlainObject";
import _isFunction from "lodash/isFunction";
import _isNil from "lodash/isNil";
import _defaults from "lodash/defaults";
import _assign from "lodash/assign";
import { Helpers, Scale, Domain, Data, LabelHelpers } from "victory-core";
var TYPES = ["close", "open", "high", "low"];

var getData = function (props) {
  var accessorTypes = ["x", "high", "low", "close", "open"];
  return Data.formatData(props.data, props, accessorTypes);
};

var reduceData = function (dataset, axis, type) {
  var yDataTypes = {
    min: "_low",
    max: "_high"
  };
  var dataType = axis === "x" ? "_x" : yDataTypes[type];
  var baseCondition = type === "min" ? Infinity : -Infinity;
  return dataset.reduce(function (memo, datum) {
    var current = datum[dataType];
    return memo < current && type === "min" || memo > current && type === "max" ? memo : current;
  }, baseCondition);
};

var getDomainFromData = function (props, axis) {
  var minDomain = Domain.getMinFromProps(props, axis);
  var maxDomain = Domain.getMaxFromProps(props, axis);
  var dataset = getData(props);

  if (dataset.length < 1) {
    return minDomain !== undefined && maxDomain !== undefined ? Domain.getDomainFromMinMax(minDomain, maxDomain) : undefined;
  }

  var min = minDomain !== undefined ? minDomain : reduceData(dataset, axis, "min");
  var max = maxDomain !== undefined ? maxDomain : reduceData(dataset, axis, "max");
  return Domain.getDomainFromMinMax(min, max);
};

var getDomain = function (props, axis) {
  return Domain.createDomainFunction(getDomainFromData)(props, axis);
};

var getStyles = function (style) {
  var defaultStyles = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var width = "100%";
  var height = "100%";

  if (!style) {
    return _defaults({
      parent: {
        height: height,
        width: width
      }
    }, defaultStyles);
  }

  var defaultParent = defaultStyles.parent || {};
  var defaultLabels = defaultStyles.labels || {};
  var defaultData = defaultStyles.data || {};

  var labelStyle = _defaults({}, style.labels, defaultLabels);

  return {
    parent: _defaults({}, style.parent, defaultParent, {
      width: width,
      height: height
    }),
    labels: labelStyle,
    data: _defaults({}, style.data, defaultData),
    openLabels: _defaults({}, style.openLabels, defaultStyles.openLabels, labelStyle),
    closeLabels: _defaults({}, style.closeLabels, defaultStyles.closeLabels, labelStyle),
    lowLabels: _defaults({}, style.lowLabels, defaultStyles.lowLabels, labelStyle),
    highLabels: _defaults({}, style.highLabels, defaultStyles.highLabels, labelStyle)
  };
};

var getCalculatedValues = function (props) {
  var theme = props.theme,
      polar = props.polar;
  var defaultStyle = theme && theme.candlestick && theme.candlestick.style ? theme.candlestick.style : {};
  var style = getStyles(props.style, defaultStyle);
  var data = getData(props);
  var range = {
    x: Helpers.getRange(props, "x"),
    y: Helpers.getRange(props, "y")
  };
  var domain = {
    x: getDomain(props, "x"),
    y: getDomain(props, "y")
  };
  var scale = {
    x: Scale.getBaseScale(props, "x").domain(domain.x).range(props.horizontal ? range.y : range.x),
    y: Scale.getBaseScale(props, "y").domain(domain.y).range(props.horizontal ? range.x : range.y)
  };
  var origin = polar ? props.origin || Helpers.getPolarOrigin(props) : undefined;
  var defaultOrientation = props.horizontal ? "top" : "right";
  var labelOrientation = props.labelOrientation || defaultOrientation;
  return {
    domain: domain,
    data: data,
    scale: scale,
    style: style,
    origin: origin,
    labelOrientation: labelOrientation
  };
};

var isTransparent = function (attr) {
  return attr === "none" || attr === "transparent";
};

var getDataStyles = function (datum, style, props) {
  style = style || {};
  var candleColor = datum._open > datum._close ? props.candleColors.negative : props.candleColors.positive;
  var fill = style.fill || candleColor;
  var strokeColor = style.stroke;
  var stroke = isTransparent(strokeColor) ? fill : strokeColor || "black";
  return _assign({}, style, {
    stroke: stroke,
    fill: fill
  });
};

var getText = function (props, type) {
  var datum = props.datum,
      index = props.index,
      labels = props.labels;
  var propName = "".concat(type, "Labels");
  var labelProp = props[propName];

  if (!labelProp && !labels) {
    return null;
  } else if (labelProp === true || labels === true) {
    var dataName = "_".concat(type);
    return "".concat(datum[dataName]);
  }

  return Array.isArray(labelProp) ? labelProp[index] : labelProp;
};

var getCandleWidth = function (props, style) {
  var data = props.data,
      candleWidth = props.candleWidth,
      scale = props.scale,
      defaultCandleWidth = props.defaultCandleWidth;

  if (candleWidth) {
    return _isFunction(candleWidth) ? Helpers.evaluateProp(candleWidth, props) : candleWidth;
  } else if (style && style.width) {
    return style.width;
  }

  var range = scale.x.range();
  var extent = Math.abs(range[1] - range[0]);
  var candles = data.length + 2;
  var candleRatio = props.candleRatio || 0.5;
  var defaultWidth = candleRatio * (data.length < 2 ? defaultCandleWidth : extent / candles);
  return Math.max(1, defaultWidth);
};

var getOrientation = function (labelOrientation) {
  var type = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : "labels";
  return _isPlainObject(labelOrientation) ? labelOrientation[type] : labelOrientation;
};
/* eslint-disable complexity*/


var calculatePlotValues = function (props) {
  var positions = props.positions,
      labelStyle = props.labelStyle,
      x = props.x,
      horizontal = props.horizontal,
      computedType = props.computedType,
      candleWidth = props.candleWidth,
      orientation = props.orientation;
  positions.labels = (positions.open + positions.close) / 2;
  var signX = orientation === "left" ? -1 : 1;
  var signY = orientation === "top" ? -1 : 1;

  if (horizontal) {
    var yValue = x;
    var xValue = positions[computedType];
    var dy = orientation === "top" || orientation === "bottom" ? signY * (candleWidth / 2) + signY * (labelStyle.padding || 0) : 0;
    var dx = orientation === "top" || orientation === "bottom" ? 0 : signX * (labelStyle.padding || 1);
    return {
      yValue: yValue,
      xValue: xValue,
      dx: dx,
      dy: dy
    };
  } else {
    var _xValue = x;
    var _yValue = positions[computedType];

    var _dy = orientation === "top" || orientation === "bottom" ? signY * (labelStyle.padding || 1) : 0;

    var _dx = orientation === "top" || orientation === "bottom" ? 0 : signX * (candleWidth / 2) + signX * (labelStyle.padding || 0);

    return {
      yValue: _yValue,
      xValue: _xValue,
      dx: _dx,
      dy: _dy
    };
  }
};
/* eslint-enable complexity*/

/* eslint-disable max-params*/


var getLabelProps = function (props, text, style, type) {
  var x = props.x,
      high = props.high,
      low = props.low,
      open = props.open,
      close = props.close,
      index = props.index,
      scale = props.scale,
      datum = props.datum,
      data = props.data,
      horizontal = props.horizontal,
      candleWidth = props.candleWidth,
      labelOrientation = props.labelOrientation;
  var component = props["".concat(type, "LabelComponent")] || props.labelComponent;
  var defaultOrientation = horizontal ? "top" : "right";
  var orientation = component.props && component.props.orientation || getOrientation(labelOrientation, type) || defaultOrientation;
  var positions = {
    high: high,
    low: low,
    open: open,
    close: close
  };
  var namespace = type ? "".concat(type, "Labels") : "labels";
  var labelStyle = style[namespace] || style.labels;
  var defaultVerticalAnchors = {
    top: "end",
    bottom: "start",
    left: "middle",
    right: "middle"
  };
  var defaultTextAnchors = {
    left: "end",
    right: "start",
    top: "middle",
    bottom: "middle"
  };
  var computedType = type ? type : "labels";
  var plotProps = {
    positions: positions,
    labelStyle: labelStyle,
    x: x,
    horizontal: horizontal,
    computedType: computedType,
    candleWidth: candleWidth,
    orientation: orientation
  };

  var _calculatePlotValues = calculatePlotValues(plotProps),
      yValue = _calculatePlotValues.yValue,
      xValue = _calculatePlotValues.xValue,
      dx = _calculatePlotValues.dx,
      dy = _calculatePlotValues.dy;

  return {
    style: labelStyle,
    y: yValue,
    x: xValue,
    dx: dx,
    dy: dy,
    text: text,
    index: index,
    scale: scale,
    datum: datum,
    data: data,
    orientation: orientation,
    textAnchor: labelStyle.textAnchor || defaultTextAnchors[orientation],
    verticalAnchor: labelStyle.verticalAnchor || defaultVerticalAnchors[orientation],
    angle: labelStyle.angle,
    horizontal: horizontal
  };
};
/* eslint-enable max-params*/


var getBaseProps = function (props, fallbackProps) {
  // eslint-disable-line max-statements
  props = Helpers.modifyProps(props, fallbackProps, "candlestick");
  var calculatedValues = getCalculatedValues(props);
  var data = calculatedValues.data,
      style = calculatedValues.style,
      scale = calculatedValues.scale,
      domain = calculatedValues.domain,
      origin = calculatedValues.origin,
      labelOrientation = calculatedValues.labelOrientation;
  var _props = props,
      groupComponent = _props.groupComponent,
      width = _props.width,
      height = _props.height,
      padding = _props.padding,
      standalone = _props.standalone,
      name = _props.name,
      candleWidth = _props.candleWidth,
      candleRatio = _props.candleRatio,
      theme = _props.theme,
      polar = _props.polar,
      wickStrokeWidth = _props.wickStrokeWidth,
      labels = _props.labels,
      events = _props.events,
      sharedEvents = _props.sharedEvents,
      horizontal = _props.horizontal;
  var initialChildProps = {
    parent: {
      domain: domain,
      scale: scale,
      width: width,
      height: height,
      data: data,
      standalone: standalone,
      theme: theme,
      polar: polar,
      origin: origin,
      name: name,
      style: style.parent,
      padding: padding,
      horizontal: horizontal
    }
  };
  return data.reduce(function (childProps, datum, index) {
    var eventKey = !_isNil(datum.eventKey) ? datum.eventKey : index;
    var x = scale.x(datum._x1 !== undefined ? datum._x1 : datum._x);
    var high = scale.y(datum._high);
    var close = scale.y(datum._close);
    var open = scale.y(datum._open);
    var low = scale.y(datum._low);
    var dataStyle = getDataStyles(datum, style.data, props);
    var dataProps = {
      x: x,
      high: high,
      low: low,
      candleWidth: candleWidth,
      candleRatio: candleRatio,
      scale: scale,
      data: data,
      datum: datum,
      groupComponent: groupComponent,
      index: index,
      style: dataStyle,
      width: width,
      polar: polar,
      origin: origin,
      wickStrokeWidth: wickStrokeWidth,
      open: open,
      close: close,
      horizontal: horizontal,
      labelOrientation: labelOrientation
    };
    dataProps.candleWidth = getCandleWidth(dataProps);

    var extendedProps = _defaults(Object.assign({}, dataProps), props);

    childProps[eventKey] = {
      data: dataProps
    };

    if (labels) {
      var text = LabelHelpers.getText(props, datum, index);

      if (text !== undefined && text !== null || labels && (events || sharedEvents)) {
        childProps[eventKey].labels = getLabelProps(extendedProps, text, style);
      }
    }

    TYPES.forEach(function (type) {
      var labelText = getText(extendedProps, type);
      var labelProp = props.labels || props["".concat(type, "Labels")];

      if (labelText !== null && labelText !== undefined || labelProp && (events || sharedEvents)) {
        var target = "".concat(type, "Labels");
        childProps[eventKey][target] = getLabelProps(extendedProps, labelText, style, type);
      }
    });
    return childProps;
  }, initialChildProps);
};

export { getBaseProps, getDomain, getData };