"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseRequirementsFile = void 0;
const debugLib = require("debug");
const standardize_package_name_1 = require("./standardize-package-name");
const debug = debugLib('snyk-fix:python:requirements-file-parser');
function parseRequirementsFile(requirementsFile) {
    const endsWithNewLine = requirementsFile.endsWith('\n');
    const lines = requirementsFile.replace(/\n$/, '').split('\n');
    const requirements = [];
    lines.map((requirementText, line) => {
        const requirement = extractDependencyDataFromLine(requirementText, line);
        if (requirement) {
            requirements.push(requirement);
        }
    });
    return { requirements, endsWithNewLine };
}
exports.parseRequirementsFile = parseRequirementsFile;
function extractDependencyDataFromLine(requirementText, line) {
    try {
        const requirement = { originalText: requirementText, line };
        const trimmedText = requirementText.trim();
        // Quick returns for cases we cannot remediate
        // - Empty line i.e. ''
        // - 'editable' packages i.e. '-e git://git.myproject.org/MyProject.git#egg=MyProject'
        // - Comments i.e. # This is a comment
        // - Local files i.e. file:../../lib/project#egg=MyProject
        if (requirementText === '' ||
            trimmedText.startsWith('-e') ||
            trimmedText.startsWith('#') ||
            trimmedText.startsWith('file:')) {
            return requirement;
        }
        // Regex to match against a Python package specifier. Any invalid lines (or
        // lines we can't handle) should have been returned this point.
        const regex = /([A-Z0-9-._]*)(!=|===|==|>=|<=|>|<|~=)(\d*\.?\d*\.?\d*[A-Z0-9]*)(.*)/i;
        const result = regex.exec(requirementText);
        if (result !== null) {
            requirement.name = standardize_package_name_1.standardizePackageName(result[1]);
            requirement.originalName = result[1];
            requirement.versionComparator = result[2];
            requirement.version = result[3];
            requirement.extras = result[4];
        }
        if (!(requirement.version && requirement.name)) {
            throw new Error('Failed to extract dependency data');
        }
        return requirement;
    }
    catch (err) {
        debug({ error: err.message, requirementText, line }, 'failed to parse requirement');
        return { originalText: requirementText, line };
    }
}
//# sourceMappingURL=requirements-file-parser.js.map