"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FailedToLoadFileError = exports.NoFilesToScanError = exports.tryLoadFileData = exports.loadFiles = void 0;
const makeDirectoryIterator_1 = require("../../../../lib/iac/makeDirectoryIterator");
const fs_1 = require("fs");
const types_1 = require("./types");
const iac_parser_1 = require("../../../../lib/iac/iac-parser");
const detect_1 = require("../../../../lib/detect");
const errors_1 = require("../../../../lib/errors");
const error_utils_1 = require("./error-utils");
const DEFAULT_ENCODING = 'utf-8';
async function loadFiles(pathToScan, options = {}) {
    let filePaths = [pathToScan];
    if (detect_1.isLocalFolder(pathToScan)) {
        filePaths = getFilePathsFromDirectory(pathToScan, {
            maxDepth: options.detectionDepth,
        });
    }
    const filesToScan = [];
    for (const filePath of filePaths) {
        try {
            const fileData = await tryLoadFileData(filePath);
            if (fileData) {
                filesToScan.push(fileData);
            }
        }
        catch (e) {
            throw new FailedToLoadFileError(filePath);
        }
    }
    if (filesToScan.length === 0) {
        throw new NoFilesToScanError();
    }
    return filesToScan.filter((file) => file.fileContent !== '');
}
exports.loadFiles = loadFiles;
function getFilePathsFromDirectory(pathToScan, options = {}) {
    const directoryPaths = makeDirectoryIterator_1.makeDirectoryIterator(pathToScan, {
        maxDepth: options.maxDepth,
    });
    const directoryFilePaths = [];
    for (const filePath of directoryPaths) {
        directoryFilePaths.push(filePath);
    }
    return directoryFilePaths;
}
async function tryLoadFileData(pathToScan) {
    const fileType = iac_parser_1.getFileType(pathToScan);
    if (!types_1.VALID_FILE_TYPES.includes(fileType)) {
        return null;
    }
    const fileContent = (await fs_1.promises.readFile(pathToScan, DEFAULT_ENCODING)).toString();
    return {
        filePath: pathToScan,
        fileType: fileType,
        fileContent,
    };
}
exports.tryLoadFileData = tryLoadFileData;
class NoFilesToScanError extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Could not find any valid IaC files');
        this.code = types_1.IaCErrorCodes.NoFilesToScanError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage =
            'Could not find any valid infrastructure as code files. Supported file extensions are tf, yml, yaml & json.\nMore information can be found by running `snyk iac test --help` or through our documentation:\nhttps://support.snyk.io/hc/en-us/articles/360012429477-Test-your-Kubernetes-files-with-our-CLI-tool\nhttps://support.snyk.io/hc/en-us/articles/360013723877-Test-your-Terraform-files-with-our-CLI-tool';
    }
}
exports.NoFilesToScanError = NoFilesToScanError;
class FailedToLoadFileError extends errors_1.CustomError {
    constructor(filename) {
        super('Failed to load file content');
        this.code = types_1.IaCErrorCodes.FailedToLoadFileError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to read file "${filename}" for scanning. Please ensure that it is readable.`;
    }
}
exports.FailedToLoadFileError = FailedToLoadFileError;
//# sourceMappingURL=file-loader.js.map