"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getProjectType = void 0;
const fs = require("fs");
const pathLib = require("path");
const debugLib = require("debug");
const detect_1 = require("../detect");
const errors_1 = require("../errors");
const iac_parser_1 = require("./iac-parser");
const constants_1 = require("./constants");
const makeDirectoryIterator_1 = require("./makeDirectoryIterator");
const unsupported_options_iac_error_1 = require("../errors/unsupported-options-iac-error");
const invalid_iac_file_1 = require("../errors/invalid-iac-file");
const Queue = require("promise-queue");
const debug = debugLib('snyk-detect-iac');
async function getProjectType(root, options) {
    if (options.file) {
        debug('Iac - --file specified ' + options.file);
        throw unsupported_options_iac_error_1.UnsupportedOptionFileIacError(options.file);
    }
    if (detect_1.isLocalFolder(root)) {
        // Due to the fact we are first getting the project type and only then
        // scanning the projects - we need save the files we need to scan on the options
        // so we could create assembly payloads for the relevant files.
        // We are sending it as a `Multi IaC` project - and later assign the relevant type for each project
        const directoryFiles = await getDirectoryFiles(root, {
            maxDepth: options.detectionDepth,
        });
        options.iacDirFiles = directoryFiles;
        return constants_1.IacProjectType.MULTI_IAC;
    }
    if (detect_1.localFileSuppliedButNotFound(root, '.') || !fs.existsSync(root)) {
        throw unsupported_options_iac_error_1.SupportLocalFileOnlyIacError();
    }
    const filePath = pathLib.resolve(root, '.');
    return getProjectTypeForIacFile(filePath);
}
exports.getProjectType = getProjectType;
async function getProjectTypeForIacFile(filePath) {
    const fileContent = fs.readFileSync(filePath, 'utf-8');
    const fileType = pathLib.extname(filePath).substr(1);
    const fileName = pathLib.basename(filePath);
    const projectType = constants_1.projectTypeByFileType[fileType];
    switch (projectType) {
        case constants_1.IacProjectType.K8S: {
            const { isValidFile, reason } = iac_parser_1.validateK8sFile(fileContent, filePath, fileName);
            if (!isValidFile) {
                throw invalid_iac_file_1.InvalidK8SFileError(reason);
            }
            break;
        }
        case constants_1.IacProjectType.TERRAFORM: {
            const { isValidFile, reason } = await iac_parser_1.makeValidateTerraformRequest(fileContent);
            if (!isValidFile) {
                throw invalid_iac_file_1.IllegalTerraformFileError([fileName], reason);
            }
            break;
        }
        default:
            throw invalid_iac_file_1.IllegalIacCustomError(fileName);
    }
    return projectType;
}
async function getDirectoryFiles(root, options = {}) {
    const dirPath = pathLib.resolve(root, '.');
    const supportedExtensions = new Set(Object.keys(constants_1.projectTypeByFileType));
    const directoryPaths = makeDirectoryIterator_1.makeDirectoryIterator(dirPath, {
        maxDepth: options.maxDepth,
    });
    const iacFiles = [];
    const maxConcurrent = 25;
    const queue = new Queue(maxConcurrent);
    for (const filePath of directoryPaths) {
        const fileType = pathLib
            .extname(filePath)
            .substr(1)
            .toLowerCase();
        if (!fileType || !supportedExtensions.has(fileType)) {
            continue;
        }
        iacFiles.push(queue.add(async () => {
            try {
                const projectType = await getProjectTypeForIacFile(filePath);
                return {
                    filePath,
                    projectType,
                    fileType,
                };
            }
            catch (err) {
                return {
                    filePath,
                    fileType,
                    failureReason: err instanceof errors_1.CustomError ? err.userMessage : 'Unhandled Error',
                };
            }
        }));
    }
    if (iacFiles.length === 0) {
        throw unsupported_options_iac_error_1.IacDirectoryWithoutAnyIacFileError();
    }
    return Promise.all(iacFiles);
}
//# sourceMappingURL=detect-iac.js.map