"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeValidateTerraformRequest = exports.validateK8sFile = exports.getFileType = void 0;
//TODO(orka): take out into a new lib
const YAML = require("yaml");
const debugLib = require("debug");
const errors_1 = require("../errors");
const request = require("../request");
const api_token_1 = require("../api-token");
const config = require("./../config");
const debug = debugLib('snyk-detect');
const requiredK8SObjectFields = ['apiVersion', 'kind', 'metadata'];
function getFileType(filePath) {
    const filePathSplit = filePath.split('.');
    return filePathSplit[filePathSplit.length - 1].toLowerCase();
}
exports.getFileType = getFileType;
function parseYamlOrJson(fileContent, filePath) {
    const fileType = getFileType(filePath);
    switch (fileType) {
        case 'yaml':
        case 'yml':
            try {
                return YAML.parseAllDocuments(fileContent).map((doc) => {
                    if (doc.errors.length !== 0) {
                        throw doc.errors[0];
                    }
                    return doc.toJSON();
                });
            }
            catch (e) {
                debug('Failed to parse iac config as a YAML');
            }
            break;
        case 'json':
            try {
                const objectsArr = [];
                objectsArr.push(JSON.parse(fileContent));
                return objectsArr;
            }
            catch (e) {
                debug('Failed to parse iac config as a JSON');
            }
            break;
        default:
            debug(`Unsupported iac config file type (${fileType})`);
    }
    return undefined;
}
// This function validates that there is at least one valid doc with a k8s object kind.
// If there is a doc with a supported kind, but invalid, we should fail
// The function return true if the yaml is a valid k8s one, or false otherwise
function validateK8sFile(fileContent, filePath, fileName) {
    const k8sObjects = parseYamlOrJson(fileContent, filePath);
    if (!k8sObjects) {
        return { isValidFile: false, reason: errors_1.IllegalIacFileErrorMsg(fileName) };
    }
    let numOfSupportedKeyDocs = 0;
    for (let i = 0; i < k8sObjects.length; i++) {
        const k8sObject = k8sObjects[i];
        if (!k8sObject || !k8sObject.kind) {
            continue;
        }
        numOfSupportedKeyDocs++;
        for (const key of requiredK8SObjectFields) {
            if (!k8sObject[key]) {
                debug(`Missing required field (${key})`);
                return {
                    isValidFile: false,
                    reason: errors_1.IllegalIacFileErrorMsg(fileName),
                };
            }
        }
    }
    if (numOfSupportedKeyDocs === 0) {
        return {
            isValidFile: false,
            reason: errors_1.NotSupportedIacFileErrorMsg(fileName),
        };
    }
    debug(`k8s config found (${filePath})`);
    return { isValidFile: true, reason: '' };
}
exports.validateK8sFile = validateK8sFile;
async function makeValidateTerraformRequest(terraformFileContent) {
    var _a, _b;
    const response = await request({
        body: {
            contentBase64: Buffer.from(terraformFileContent).toString('base64'),
        },
        url: `${config.API}/iac-validate/terraform`,
        method: 'POST',
        json: true,
        headers: {
            Authorization: `token ${api_token_1.api()}`,
        },
    });
    // Token may have expired, so we need to ask the client to re-auth.
    if (response.res.statusCode === 401) {
        throw new errors_1.MissingApiTokenError();
    }
    if (!response.res.statusCode ||
        (response.res.statusCode < 200 && response.res.statusCode >= 300)) {
        debug(`internal server error - ${response.body}`);
        throw new errors_1.InternalServerError('Error occurred validating terraform file');
    }
    const { body } = response;
    return {
        isValidFile: (_a = body === null || body === void 0 ? void 0 : body.isValidTerraformFile) !== null && _a !== void 0 ? _a : false,
        reason: (_b = body === null || body === void 0 ? void 0 : body.reason) !== null && _b !== void 0 ? _b : '',
    };
}
exports.makeValidateTerraformRequest = makeValidateTerraformRequest;
//# sourceMappingURL=iac-parser.js.map