"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.streamToJson = exports.streamToSha1 = exports.streamToSha256 = exports.streamToBuffer = exports.streamToString = void 0;
const crypto = require("crypto");
const types_1 = require("./types");
const HASH_ENCODING = "hex";
async function streamToString(stream, streamSize, encoding = "utf8") {
    const chunks = [];
    return new Promise((resolve, reject) => {
        stream.on("end", () => {
            resolve(chunks.join(""));
        });
        stream.on("error", (error) => reject(error));
        stream.on("data", (chunk) => {
            chunks.push(chunk.toString(encoding));
        });
    });
}
exports.streamToString = streamToString;
async function streamToBuffer(stream) {
    const chunks = [];
    return new Promise((resolve, reject) => {
        stream.on("end", () => {
            resolve(Buffer.concat(chunks));
        });
        stream.on("error", (error) => reject(error));
        stream.on("data", (chunk) => {
            chunks.push(Buffer.from(chunk));
        });
    });
}
exports.streamToBuffer = streamToBuffer;
async function streamToHash(stream, hashAlgorithm) {
    return new Promise((resolve, reject) => {
        const hash = crypto.createHash(hashAlgorithm);
        hash.setEncoding(HASH_ENCODING);
        stream.on("end", () => {
            hash.end();
            resolve(hash.read().toString(HASH_ENCODING));
        });
        stream.on("error", (error) => reject(error));
        stream.pipe(hash);
    });
}
async function streamToSha256(stream) {
    return streamToHash(stream, types_1.HashAlgorithm.Sha256);
}
exports.streamToSha256 = streamToSha256;
async function streamToSha1(stream) {
    return streamToHash(stream, types_1.HashAlgorithm.Sha1);
}
exports.streamToSha1 = streamToSha1;
async function streamToJson(stream) {
    const file = await streamToString(stream);
    return JSON.parse(file);
}
exports.streamToJson = streamToJson;
//# sourceMappingURL=stream-utils.js.map