"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.assertIaCOptionsFlags = exports.FlagValueError = exports.FlagError = void 0;
const errors_1 = require("../../../../lib/errors");
const args_1 = require("../../../args");
const error_utils_1 = require("./error-utils");
const types_1 = require("./types");
const keys = [
    'org',
    'debug',
    'insecure',
    'detectionDepth',
    'severityThreshold',
    'rules',
    'json',
    'sarif',
    'json-file-output',
    'sarif-file-output',
    'v',
    'version',
    'h',
    'help',
    'q',
    'quiet',
    'scan',
    'legacy',
];
const allowed = new Set(keys);
function camelcaseToDash(key) {
    return key.replace(/[A-Z]/g, (m) => '-' + m.toLowerCase());
}
function getFlagName(key) {
    const dashes = key.length === 1 ? '-' : '--';
    const flag = camelcaseToDash(key);
    return `${dashes}${flag}`;
}
class FlagError extends errors_1.CustomError {
    constructor(key) {
        const flag = getFlagName(key);
        const msg = `Unsupported flag "${flag}" provided. Run snyk iac test --help for supported flags.`;
        super(msg);
        this.code = types_1.IaCErrorCodes.FlagError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = msg;
    }
}
exports.FlagError = FlagError;
class FlagValueError extends errors_1.CustomError {
    constructor(key, value) {
        const flag = getFlagName(key);
        const msg = `Unsupported value "${value}" provided to flag "${flag}".\nSupported values are: ${SUPPORTED_TF_PLAN_SCAN_MODES.join(', ')}`;
        super(msg);
        this.code = types_1.IaCErrorCodes.FlagValueError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = msg;
    }
}
exports.FlagValueError = FlagValueError;
/**
 * Validates the command line flags passed to the snyk iac test
 * command. The current argument parsing is very permissive and
 * allows unknown flags to be provided without valdiation.
 *
 * For snyk iac we need to explictly validate the flags to avoid
 * misconfigurations and typos. For example, if the --experimental
 * flag were to be mis-spelled we would end up sending the client
 * data to our backend rather than running it locally as intended.
 * @param argv command line args passed to the process
 */
function assertIaCOptionsFlags(argv) {
    // We process the process.argv so we don't get default values.
    const parsed = args_1.args(argv);
    for (const key of Object.keys(parsed.options)) {
        // The _ property is a special case that contains non
        // flag strings passed to the command line (usually files)
        // and `iac` is the command provided.
        if (key !== '_' && key !== 'iac' && !allowed.has(key)) {
            throw new FlagError(key);
        }
    }
    if (parsed.options.scan) {
        assertTerraformPlanModes(parsed.options.scan);
    }
}
exports.assertIaCOptionsFlags = assertIaCOptionsFlags;
const SUPPORTED_TF_PLAN_SCAN_MODES = [
    types_1.TerraformPlanScanMode.DeltaScan,
    types_1.TerraformPlanScanMode.FullScan,
];
function assertTerraformPlanModes(scanModeArgValue) {
    if (!SUPPORTED_TF_PLAN_SCAN_MODES.includes(scanModeArgValue)) {
        throw new FlagValueError('scan', scanModeArgValue);
    }
}
//# sourceMappingURL=assert-iac-options-flag.js.map