"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FailedToExtractCustomRulesError = exports.FailedToDownloadRulesError = exports.FailedToInitLocalCacheError = exports.cleanLocalCache = exports.initLocalCache = exports.getLocalCachePath = exports.assertNever = exports.LOCAL_POLICY_ENGINE_DIR = void 0;
const path = require("path");
const fs = require("fs");
const types_1 = require("./types");
const needle = require("needle");
const rimraf = require("rimraf");
const file_utils_1 = require("./file-utils");
const Debug = require("debug");
const errors_1 = require("../../../../lib/errors");
const analytics = require("../../../../lib/analytics");
const error_utils_1 = require("./error-utils");
const debug = Debug('iac-local-cache');
exports.LOCAL_POLICY_ENGINE_DIR = '.iac-data';
const KUBERNETES_POLICY_ENGINE_WASM_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'k8s_policy.wasm');
const KUBERNETES_POLICY_ENGINE_DATA_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'k8s_data.json');
const TERRAFORM_POLICY_ENGINE_WASM_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'tf_policy.wasm');
const TERRAFORM_POLICY_ENGINE_DATA_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'tf_data.json');
const CLOUDFORMATION_POLICY_ENGINE_WASM_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'cloudformation_policy.wasm');
const CLOUDFORMATION_POLICY_ENGINE_DATA_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'cloudformation_data.json');
// NOTE: The filenames used for the custom policy bundles match those output
// by the `opa` CLI tool, which is why they are very generic.
const CUSTOM_POLICY_ENGINE_WASM_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'policy.wasm');
const CUSTOM_POLICY_ENGINE_DATA_PATH = path.join(exports.LOCAL_POLICY_ENGINE_DIR, 'data.json');
function assertNever(value) {
    throw new Error(`Unhandled discriminated union member: ${JSON.stringify(value)}`);
}
exports.assertNever = assertNever;
function getLocalCachePath(engineType) {
    switch (engineType) {
        case types_1.EngineType.Kubernetes:
            return [
                `${process.cwd()}/${KUBERNETES_POLICY_ENGINE_WASM_PATH}`,
                `${process.cwd()}/${KUBERNETES_POLICY_ENGINE_DATA_PATH}`,
            ];
        case types_1.EngineType.Terraform:
            return [
                `${process.cwd()}/${TERRAFORM_POLICY_ENGINE_WASM_PATH}`,
                `${process.cwd()}/${TERRAFORM_POLICY_ENGINE_DATA_PATH}`,
            ];
        case types_1.EngineType.CloudFormation:
            return [
                `${process.cwd()}/${CLOUDFORMATION_POLICY_ENGINE_WASM_PATH}`,
                `${process.cwd()}/${CLOUDFORMATION_POLICY_ENGINE_DATA_PATH}`,
            ];
        case types_1.EngineType.Custom:
            return [
                `${process.cwd()}/${CUSTOM_POLICY_ENGINE_WASM_PATH}`,
                `${process.cwd()}/${CUSTOM_POLICY_ENGINE_DATA_PATH}`,
            ];
        default:
            assertNever(engineType);
    }
}
exports.getLocalCachePath = getLocalCachePath;
async function initLocalCache({ customRulesPath, } = {}) {
    try {
        file_utils_1.createIacDir();
    }
    catch (e) {
        throw new FailedToInitLocalCacheError();
    }
    // Attempt to extract the custom rules from the path provided.
    if (customRulesPath) {
        try {
            const response = fs.createReadStream(customRulesPath);
            await file_utils_1.extractBundle(response);
        }
        catch (e) {
            throw new FailedToExtractCustomRulesError(customRulesPath);
        }
    }
    // We extract the Snyk rules after the custom rules to ensure our files
    // always overwrite whatever might be there.
    try {
        const BUNDLE_URL = 'https://static.snyk.io/cli/wasm/bundle.tar.gz';
        const response = needle.get(BUNDLE_URL);
        await file_utils_1.extractBundle(response);
    }
    catch (e) {
        throw new FailedToDownloadRulesError();
    }
}
exports.initLocalCache = initLocalCache;
function cleanLocalCache() {
    // path to delete is hardcoded for now
    const iacPath = path.join(`${process.cwd()}`, '.iac-data');
    try {
        // when we support Node version >= 12.10.0 , we can replace rimraf
        // with the native fs.rmdirSync(path, {recursive: true})
        rimraf.sync(iacPath);
    }
    catch (e) {
        const err = new FailedToCleanLocalCacheError();
        analytics.add('error-code', err.code);
        debug('The local cache directory could not be deleted');
    }
}
exports.cleanLocalCache = cleanLocalCache;
class FailedToInitLocalCacheError extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Failed to initialize local cache');
        this.code = types_1.IaCErrorCodes.FailedToInitLocalCacheError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage =
            'We were unable to create a local directory to store the test assets, please ensure that the current working directory is writable';
    }
}
exports.FailedToInitLocalCacheError = FailedToInitLocalCacheError;
class FailedToDownloadRulesError extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Failed to download policies');
        this.code = types_1.IaCErrorCodes.FailedToDownloadRulesError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage =
            'We were unable to download the security rules, please ensure the network can access https://static.snyk.io';
    }
}
exports.FailedToDownloadRulesError = FailedToDownloadRulesError;
class FailedToExtractCustomRulesError extends errors_1.CustomError {
    constructor(path, message) {
        super(message || 'Failed to download policies');
        this.code = types_1.IaCErrorCodes.FailedToExtractCustomRulesError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to extract the rules provided at: ${path}`;
    }
}
exports.FailedToExtractCustomRulesError = FailedToExtractCustomRulesError;
class FailedToCleanLocalCacheError extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Failed to clean local cache');
        this.code = types_1.IaCErrorCodes.FailedToCleanLocalCacheError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = ''; // Not a user facing error.
    }
}
//# sourceMappingURL=local-cache.js.map