"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FailedToExtractResourcesInTerraformPlanError = exports.tryParsingTerraformPlan = exports.isTerraformPlan = void 0;
const types_1 = require("../types");
const errors_1 = require("../../../../../lib/errors");
const error_utils_1 = require("../error-utils");
const constants_1 = require("../../../../../lib/iac/constants");
function terraformPlanReducer(scanInput, resource) {
    // TODO: investigate if this reduction logic covers all edge-cases (nested modules, similar names, etc')
    const { type, name, mode, index, values } = resource;
    const inputKey = mode === 'data' ? 'data' : 'resource';
    if (scanInput[inputKey][type]) {
        // add new resources of the same type with different names
        scanInput[inputKey][type][index !== undefined ? `${name}_${index}` : name] =
            values || {};
    }
    else {
        // add a new resource type
        scanInput[inputKey][type] = { [name]: values };
    }
    return scanInput;
}
function resourceChangeReducer(scanInput, resource, isFullScan) {
    // TODO: investigate if we need to address also `after_unknown` field.
    const { actions, after } = resource.change || { actions: [], after: {} };
    if (isValidResourceActions(actions, isFullScan)) {
        const resourceForReduction = Object.assign(Object.assign({}, resource), { values: after || {} });
        return terraformPlanReducer(scanInput, resourceForReduction);
    }
    return scanInput;
}
function isValidResourceActions(action, isFullScan) {
    const VALID_ACTIONS = isFullScan
        ? types_1.VALID_RESOURCE_ACTIONS_FOR_FULL_SCAN
        : types_1.VALID_RESOURCE_ACTIONS_FOR_DELTA_SCAN;
    return VALID_ACTIONS.some((validAction) => {
        if (action.length !== validAction.length) {
            return false;
        }
        return validAction.every((field, idx) => action[idx] === field);
    });
}
function extractResourceChanges(terraformPlanJson) {
    return terraformPlanJson.resource_changes || [];
}
function extractResourcesForScan(terraformPlanJson, isFullScan = false) {
    const resourceChanges = extractResourceChanges(terraformPlanJson);
    return resourceChanges.reduce((memo, curr) => resourceChangeReducer(memo, curr, isFullScan), {
        resource: {},
        data: {},
    });
}
function isTerraformPlan(terraformPlanJson) {
    const missingRequiredFields = terraformPlanJson.resource_changes === undefined;
    return !missingRequiredFields;
}
exports.isTerraformPlan = isTerraformPlan;
function tryParsingTerraformPlan(terraformPlanFile, terraformPlanJson, { isFullScan } = { isFullScan: false }) {
    try {
        return [
            Object.assign(Object.assign({}, terraformPlanFile), { jsonContent: extractResourcesForScan(terraformPlanJson, isFullScan), engineType: types_1.EngineType.Terraform, projectType: constants_1.IacProjectType.TERRAFORM }),
        ];
    }
    catch (err) {
        throw new FailedToExtractResourcesInTerraformPlanError();
    }
}
exports.tryParsingTerraformPlan = tryParsingTerraformPlan;
// This error is due to the complex reduction logic, so it catches scenarios we might have not covered.
class FailedToExtractResourcesInTerraformPlanError extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Failed to extract resources from Terraform plan JSON file');
        this.code = types_1.IaCErrorCodes.FailedToExtractResourcesInTerraformPlanError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage =
            'We failed to extract resource changes from the Terraform plan file, please contact support@snyk.io, if possible with a redacted version of the file';
    }
}
exports.FailedToExtractResourcesInTerraformPlanError = FailedToExtractResourcesInTerraformPlanError;
//# sourceMappingURL=terraform-plan-parser.js.map