"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FailedToFormatResults = exports.filterPoliciesBySeverity = exports.formatScanResults = void 0;
const types_1 = require("./types");
const path = require("path");
const common_1 = require("../../../../lib/snyk-test/common");
const constants_1 = require("../../../../lib/iac/constants");
const errors_1 = require("../../../../lib/errors");
const extract_line_number_1 = require("./extract-line-number");
const error_utils_1 = require("./error-utils");
const SEVERITIES = [common_1.SEVERITY.LOW, common_1.SEVERITY.MEDIUM, common_1.SEVERITY.HIGH];
function formatScanResults(scanResults, options, meta) {
    try {
        // Relevant only for multi-doc yaml files
        const scannedResultsGroupedByDocId = groupMultiDocResults(scanResults);
        return scannedResultsGroupedByDocId.map((iacScanResult) => formatScanResult(iacScanResult, meta, options));
    }
    catch (e) {
        throw new FailedToFormatResults();
    }
}
exports.formatScanResults = formatScanResults;
const engineTypeToProjectType = {
    [types_1.EngineType.Kubernetes]: constants_1.IacProjectType.K8S,
    [types_1.EngineType.Terraform]: constants_1.IacProjectType.TERRAFORM,
    [types_1.EngineType.CloudFormation]: constants_1.IacProjectType.CLOUDFORMATION,
    [types_1.EngineType.Custom]: constants_1.IacProjectType.CUSTOM,
};
function formatScanResult(scanResult, meta, options) {
    const formattedIssues = scanResult.violatedPolicies.map((policy) => {
        const cloudConfigPath = scanResult.docId !== undefined
            ? [`[DocId:${scanResult.docId}]`].concat(policy.msg.split('.'))
            : policy.msg.split('.');
        const flagsRequiringLineNumber = [
            'json',
            'sarif',
            'json-file-output',
            'sarif-file-output',
        ];
        const shouldExtractLineNumber = flagsRequiringLineNumber.some((flag) => options[flag]);
        const lineNumber = shouldExtractLineNumber
            ? extract_line_number_1.extractLineNumber(scanResult, policy)
            : -1;
        return Object.assign(Object.assign({}, policy), { id: policy.publicId, name: policy.title, cloudConfigPath, isIgnored: false, iacDescription: {
                issue: policy.issue,
                impact: policy.impact,
                resolve: policy.resolve,
            }, severity: policy.severity, lineNumber, documentation: `https://snyk.io/security-rules/${policy.publicId}` });
    });
    const targetFilePath = path.resolve(scanResult.filePath, '.');
    return {
        result: {
            cloudConfigResults: filterPoliciesBySeverity(formattedIssues, options.severityThreshold),
            projectType: scanResult.projectType,
        },
        meta: Object.assign(Object.assign({}, meta), { projectId: '', policy: '' }),
        filesystemPolicy: false,
        vulnerabilities: [],
        dependencyCount: 0,
        licensesPolicy: null,
        ignoreSettings: null,
        targetFile: scanResult.filePath,
        projectName: path.basename(path.dirname(targetFilePath)),
        org: meta.org,
        policy: '',
        isPrivate: true,
        targetFilePath,
        packageManager: engineTypeToProjectType[scanResult.engineType],
    };
}
function groupMultiDocResults(scanResults) {
    const groupedData = scanResults.reduce((memo, result) => {
        if (memo[result.filePath]) {
            memo[result.filePath].violatedPolicies = memo[result.filePath].violatedPolicies.concat(result.violatedPolicies);
        }
        else {
            memo[result.filePath] = result;
        }
        return memo;
    }, {});
    return Object.values(groupedData);
}
function filterPoliciesBySeverity(violatedPolicies, severityThreshold) {
    if (!severityThreshold || severityThreshold === common_1.SEVERITY.LOW) {
        return violatedPolicies.filter((violatedPolicy) => {
            return violatedPolicy.severity !== 'none';
        });
    }
    const severitiesToInclude = SEVERITIES.slice(SEVERITIES.indexOf(severityThreshold));
    return violatedPolicies.filter((policy) => {
        return (policy.severity !== 'none' &&
            severitiesToInclude.includes(policy.severity));
    });
}
exports.filterPoliciesBySeverity = filterPoliciesBySeverity;
class FailedToFormatResults extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Failed to format results');
        this.code = types_1.IaCErrorCodes.FailedToFormatResults;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage =
            'We failed printing the results, please contact support@snyk.io';
    }
}
exports.FailedToFormatResults = FailedToFormatResults;
//# sourceMappingURL=results-formatter.js.map