"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.detectPackageManagerFromFile = exports.detectPackageFile = exports.isLocalFolder = exports.localFileSuppliedButNotFound = exports.detectPackageManager = exports.isPathToPackageFile = exports.AUTO_DETECTABLE_FILES = void 0;
const fs = require("fs");
const pathLib = require("path");
const debugLib = require("debug");
const errors_1 = require("./errors");
const package_managers_1 = require("./package-managers");
const debug = debugLib('snyk-detect');
const DETECTABLE_FILES = [
    'yarn.lock',
    'package-lock.json',
    'package.json',
    'Gemfile',
    'Gemfile.lock',
    'pom.xml',
    'build.gradle',
    'build.gradle.kts',
    'build.sbt',
    'Pipfile',
    'requirements.txt',
    'Gopkg.lock',
    'go.mod',
    'vendor/vendor.json',
    'obj/project.assets.json',
    'project.assets.json',
    'packages.config',
    'paket.dependencies',
    'composer.lock',
    'Podfile',
    'Podfile.lock',
    'poetry.lock',
    'mix.exs',
    'mix.lock',
];
exports.AUTO_DETECTABLE_FILES = [
    'package-lock.json',
    'yarn.lock',
    'package.json',
    'Gemfile',
    'Gemfile.lock',
    'pom.xml',
    'packages.config',
    'paket.dependencies',
    'project.json',
    'project.assets.json',
    'Podfile',
    'Podfile.lock',
    'composer.lock',
    'Gopkg.lock',
    'go.mod',
    'vendor.json',
    'Pipfile',
    'requirements.txt',
    'build.sbt',
    'build.gradle',
    'build.gradle.kts',
    'poetry.lock',
    'mix.exs',
    'mix.lock',
];
// when file is specified with --file, we look it up here
// this is also used when --all-projects flag is enabled and auto detection plugin is triggered
const DETECTABLE_PACKAGE_MANAGERS = {
    [package_managers_1.SUPPORTED_MANIFEST_FILES.GEMFILE]: 'rubygems',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.GEMFILE_LOCK]: 'rubygems',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.GEMSPEC]: 'rubygems',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PACKAGE_LOCK_JSON]: 'npm',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.POM_XML]: 'maven',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.JAR]: 'maven',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.WAR]: 'maven',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.BUILD_GRADLE]: 'gradle',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.BUILD_GRADLE_KTS]: 'gradle',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.BUILD_SBT]: 'sbt',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.YARN_LOCK]: 'yarn',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PACKAGE_JSON]: 'npm',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PIPFILE]: 'pip',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.SETUP_PY]: 'pip',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.REQUIREMENTS_TXT]: 'pip',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.GOPKG_LOCK]: 'golangdep',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.GO_MOD]: 'gomodules',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.VENDOR_JSON]: 'govendor',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PROJECT_ASSETS_JSON]: 'nuget',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PACKAGES_CONFIG]: 'nuget',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PROJECT_JSON]: 'nuget',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PAKET_DEPENDENCIES]: 'paket',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.COMPOSER_LOCK]: 'composer',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PODFILE_LOCK]: 'cocoapods',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.COCOAPODS_PODFILE_YAML]: 'cocoapods',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.COCOAPODS_PODFILE]: 'cocoapods',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.PODFILE]: 'cocoapods',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.POETRY_LOCK]: 'poetry',
    [package_managers_1.SUPPORTED_MANIFEST_FILES.MIX_EXS]: 'hex',
};
function isPathToPackageFile(path) {
    for (const fileName of DETECTABLE_FILES) {
        if (path.endsWith(fileName)) {
            return true;
        }
    }
    return false;
}
exports.isPathToPackageFile = isPathToPackageFile;
function detectPackageManager(root, options) {
    // If user specified a package manager let's use it.
    if (options.packageManager) {
        return options.packageManager;
    }
    // The package manager used by a docker container is not known ahead of time
    if (options.docker) {
        return undefined;
    }
    let packageManager;
    let file;
    if (isLocalFolder(root)) {
        if (options.file) {
            if (localFileSuppliedButNotFound(root, options.file)) {
                throw new Error('Could not find the specified file: ' +
                    options.file +
                    '\nPlease check that it exists and try again.');
            }
            file = options.file;
            packageManager = detectPackageManagerFromFile(file);
        }
        else if (options.scanAllUnmanaged) {
            packageManager = 'maven';
        }
        else {
            debug('no file specified. Trying to autodetect in base folder ' + root);
            file = detectPackageFile(root);
            if (file) {
                packageManager = detectPackageManagerFromFile(file);
            }
        }
    }
    else {
        debug('specified parameter is not a folder, trying to lookup as repo');
        const registry = options.registry || 'npm';
        packageManager = detectPackageManagerFromRegistry(registry);
    }
    if (!packageManager) {
        throw errors_1.NoSupportedManifestsFoundError([root]);
    }
    return packageManager;
}
exports.detectPackageManager = detectPackageManager;
// User supplied a "local" file, but that file doesn't exist
function localFileSuppliedButNotFound(root, file) {
    return (file && fs.existsSync(root) && !fs.existsSync(pathLib.resolve(root, file)));
}
exports.localFileSuppliedButNotFound = localFileSuppliedButNotFound;
function isLocalFolder(root) {
    try {
        return fs.lstatSync(root).isDirectory();
    }
    catch (e) {
        return false;
    }
}
exports.isLocalFolder = isLocalFolder;
function detectPackageFile(root) {
    for (const file of DETECTABLE_FILES) {
        if (fs.existsSync(pathLib.resolve(root, file))) {
            debug('found package file ' + file + ' in ' + root);
            return file;
        }
    }
    debug('no package file found in ' + root);
}
exports.detectPackageFile = detectPackageFile;
function detectPackageManagerFromFile(file) {
    let key = pathLib.basename(file);
    // TODO: fix this to use glob matching instead
    // like *.gemspec
    if (/\.gemspec$/.test(key)) {
        key = '.gemspec';
    }
    if (/\.jar$/.test(key)) {
        key = '.jar';
    }
    if (/\.war$/.test(key)) {
        key = '.war';
    }
    if (!(key in DETECTABLE_PACKAGE_MANAGERS)) {
        // we throw and error here because the file was specified by the user
        throw new Error('Could not detect package manager for file: ' + file);
    }
    return DETECTABLE_PACKAGE_MANAGERS[key];
}
exports.detectPackageManagerFromFile = detectPackageManagerFromFile;
function detectPackageManagerFromRegistry(registry) {
    return registry;
}
//# sourceMappingURL=detect.js.map