"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.testEcosystem = void 0;
const config = require("../config");
const is_ci_1 = require("../is-ci");
const promise_1 = require("../request/promise");
const types_1 = require("../../cli/commands/types");
const spinner = require("../../lib/spinner");
const plugins_1 = require("./plugins");
const common_1 = require("../snyk-test/common");
const api_token_1 = require("../api-token");
async function testEcosystem(ecosystem, paths, options) {
    const plugin = plugins_1.getPlugin(ecosystem);
    // TODO: this is an intermediate step before consolidating ecosystem plugins
    // to accept flows that act differently in the testDependencies step
    if (plugin.test) {
        const { readableResult: res } = await plugin.test(paths, options);
        return types_1.TestCommandResult.createHumanReadableTestCommandResult(res, '');
    }
    const scanResultsByPath = {};
    for (const path of paths) {
        await spinner(`Scanning dependencies in ${path}`);
        options.path = path;
        const pluginResponse = await plugin.scan(options);
        scanResultsByPath[path] = pluginResponse.scanResults;
    }
    spinner.clearAll();
    const [testResults, errors] = await testDependencies(scanResultsByPath, options);
    const stringifiedData = JSON.stringify(testResults, null, 2);
    if (options.json) {
        return types_1.TestCommandResult.createJsonTestCommandResult(stringifiedData);
    }
    const emptyResults = [];
    const scanResults = emptyResults.concat(...Object.values(scanResultsByPath));
    const readableResult = await plugin.display(scanResults, testResults, errors, options);
    return types_1.TestCommandResult.createHumanReadableTestCommandResult(readableResult, stringifiedData);
}
exports.testEcosystem = testEcosystem;
async function testDependencies(scans, options) {
    const results = [];
    const errors = [];
    for (const [path, scanResults] of Object.entries(scans)) {
        await spinner(`Testing dependencies in ${path}`);
        for (const scanResult of scanResults) {
            const payload = {
                method: 'POST',
                url: `${config.API}/test-dependencies`,
                json: true,
                headers: {
                    'x-is-ci': is_ci_1.isCI(),
                    authorization: api_token_1.getAuthHeader(),
                },
                body: {
                    scanResult,
                },
                qs: common_1.assembleQueryString(options),
            };
            try {
                const response = await promise_1.makeRequest(payload);
                results.push({
                    issues: response.result.issues,
                    issuesData: response.result.issuesData,
                    depGraphData: response.result.depGraphData,
                });
            }
            catch (error) {
                if (error.code >= 400 && error.code < 500) {
                    throw new Error(error.message);
                }
                errors.push('Could not test dependencies in ' + path);
            }
        }
    }
    spinner.clearAll();
    return [results, errors];
}
//# sourceMappingURL=test.js.map