"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateIssueSummary = exports.getSeveritiesColour = exports.defaultSeverityColor = exports.severitiesColourMapping = exports.formatIssueCountBySeverity = exports.calculateFailed = exports.calculateFixedIssues = exports.calculateFixed = exports.generateFixedAndFailedSummary = exports.generateUnresolvedSummary = exports.generateSuccessfulFixesSummary = exports.showResultsSummary = exports.PADDING_SPACE = void 0;
const chalk = require("chalk");
const common_1 = require("../errors/common");
const error_to_user_message_1 = require("../errors/error-to-user-message");
const fixable_issues_1 = require("../issues/fixable-issues");
const issues_by_severity_1 = require("../issues/issues-by-severity");
const total_issues_count_1 = require("../issues/total-issues-count");
const format_successful_item_1 = require("./format-successful-item");
const format_unresolved_item_1 = require("./format-unresolved-item");
exports.PADDING_SPACE = '  '; // 2 spaces
async function showResultsSummary(resultsByPlugin, exceptionsByScanType) {
    const successfulFixesSummary = generateSuccessfulFixesSummary(resultsByPlugin);
    const { summary: unresolvedSummary, count: unresolvedCount, } = generateUnresolvedSummary(resultsByPlugin, exceptionsByScanType);
    const { summary: overallSummary, count: changedCount, } = generateFixedAndFailedSummary(resultsByPlugin, exceptionsByScanType);
    const vulnsSummary = generateIssueSummary(resultsByPlugin, exceptionsByScanType);
    const fixedIssuesSummary = `${chalk.bold(calculateFixedIssues(resultsByPlugin))} fixed issues`;
    const getHelpText = chalk.red(`\n${common_1.reTryMessage}. ${common_1.contactSupportMessage}`);
    return `\n${successfulFixesSummary}${unresolvedSummary ? `\n\n${unresolvedSummary}` : ''}${unresolvedCount || changedCount
        ? `\n\n${overallSummary}\n${vulnsSummary}\n${exports.PADDING_SPACE}${fixedIssuesSummary}`
        : ''}${unresolvedSummary ? `\n\n${getHelpText}` : ''}`;
}
exports.showResultsSummary = showResultsSummary;
function generateSuccessfulFixesSummary(resultsByPlugin) {
    const sectionTitle = 'Successful fixes:';
    const formattedTitleHeader = `${chalk.bold(sectionTitle)}`;
    let summary = '';
    for (const plugin of Object.keys(resultsByPlugin)) {
        const fixedSuccessfully = resultsByPlugin[plugin].succeeded;
        if (fixedSuccessfully.length > 0) {
            summary +=
                '\n\n' +
                    fixedSuccessfully
                        .map((s) => format_successful_item_1.formatChangesSummary(s.original, s.changes))
                        .join('\n\n');
        }
    }
    if (summary) {
        return formattedTitleHeader + summary;
    }
    return chalk.red(' ✖ No successful fixes');
}
exports.generateSuccessfulFixesSummary = generateSuccessfulFixesSummary;
function generateUnresolvedSummary(resultsByPlugin, exceptionsByScanType) {
    const title = 'Unresolved items:';
    const formattedTitle = `${chalk.bold(title)}`;
    let summary = '';
    let count = 0;
    for (const plugin of Object.keys(resultsByPlugin)) {
        const skipped = resultsByPlugin[plugin].skipped;
        if (skipped.length > 0) {
            count += skipped.length;
            summary +=
                '\n\n' +
                    skipped
                        .map((s) => format_unresolved_item_1.formatUnresolved(s.original, s.userMessage))
                        .join('\n\n');
        }
        const failed = resultsByPlugin[plugin].failed;
        if (failed.length > 0) {
            count += failed.length;
            summary +=
                '\n\n' +
                    failed
                        .map((s) => format_unresolved_item_1.formatUnresolved(s.original, error_to_user_message_1.convertErrorToUserMessage(s.error), s.tip))
                        .join('\n\n');
        }
    }
    if (Object.keys(exceptionsByScanType).length) {
        for (const ecosystem of Object.keys(exceptionsByScanType)) {
            const unresolved = exceptionsByScanType[ecosystem];
            count += unresolved.originals.length;
            summary +=
                '\n\n' +
                    unresolved.originals
                        .map((s) => format_unresolved_item_1.formatUnresolved(s, unresolved.userMessage))
                        .join('\n\n');
        }
    }
    if (summary) {
        return { summary: formattedTitle + summary, count };
    }
    return { summary: '', count: 0 };
}
exports.generateUnresolvedSummary = generateUnresolvedSummary;
function generateFixedAndFailedSummary(resultsByPlugin, exceptionsByScanType) {
    const sectionTitle = 'Summary:';
    const formattedTitleHeader = `${chalk.bold(sectionTitle)}`;
    const fixed = calculateFixed(resultsByPlugin);
    const failed = calculateFailed(resultsByPlugin, exceptionsByScanType);
    return {
        summary: `${formattedTitleHeader}\n\n${exports.PADDING_SPACE}${chalk.bold.red(failed)} items were not fixed\n${exports.PADDING_SPACE}${chalk.green.bold(fixed)} items were successfully fixed`,
        count: fixed + failed,
    };
}
exports.generateFixedAndFailedSummary = generateFixedAndFailedSummary;
function calculateFixed(resultsByPlugin) {
    let fixed = 0;
    for (const plugin of Object.keys(resultsByPlugin)) {
        fixed += resultsByPlugin[plugin].succeeded.length;
    }
    return fixed;
}
exports.calculateFixed = calculateFixed;
function calculateFixedIssues(resultsByPlugin) {
    const fixedIssues = [];
    for (const plugin of Object.keys(resultsByPlugin)) {
        for (const entity of resultsByPlugin[plugin].succeeded) {
            // count unique vulns fixed per scanned entity
            // some fixed may need to be made in multiple places
            // and would count multiple times otherwise.
            const fixedPerEntity = new Set();
            entity.changes
                .filter((c) => c.success)
                .forEach((c) => {
                c.issueIds.map((i) => fixedPerEntity.add(i));
            });
            fixedIssues.push(...Array.from(fixedPerEntity));
        }
    }
    return fixedIssues.length;
}
exports.calculateFixedIssues = calculateFixedIssues;
function calculateFailed(resultsByPlugin, exceptionsByScanType) {
    let failed = 0;
    for (const plugin of Object.keys(resultsByPlugin)) {
        const results = resultsByPlugin[plugin];
        failed += results.failed.length + results.skipped.length;
    }
    if (Object.keys(exceptionsByScanType).length) {
        for (const ecosystem of Object.keys(exceptionsByScanType)) {
            const unresolved = exceptionsByScanType[ecosystem];
            failed += unresolved.originals.length;
        }
    }
    return failed;
}
exports.calculateFailed = calculateFailed;
function formatIssueCountBySeverity({ critical, high, medium, low, }) {
    const summary = [];
    if (critical && critical > 0) {
        summary.push(exports.severitiesColourMapping.critical.colorFunc(`${critical} Critical`));
    }
    if (high && high > 0) {
        summary.push(exports.severitiesColourMapping.high.colorFunc(`${high} High`));
    }
    if (medium && medium > 0) {
        summary.push(exports.severitiesColourMapping.medium.colorFunc(`${medium} Medium`));
    }
    if (low && low > 0) {
        summary.push(exports.severitiesColourMapping.low.colorFunc(`${low} Low`));
    }
    return summary.join(' | ');
}
exports.formatIssueCountBySeverity = formatIssueCountBySeverity;
exports.severitiesColourMapping = {
    low: {
        colorFunc(text) {
            return chalk.hex('#BCBBC8')(text);
        },
    },
    medium: {
        colorFunc(text) {
            return chalk.hex('#EDD55E')(text);
        },
    },
    high: {
        colorFunc(text) {
            return chalk.hex('#FF872F')(text);
        },
    },
    critical: {
        colorFunc(text) {
            return chalk.hex('#FF0B0B')(text);
        },
    },
};
exports.defaultSeverityColor = {
    colorFunc(text) {
        return chalk.grey(text);
    },
};
function getSeveritiesColour(severity) {
    var _a;
    return (_a = exports.severitiesColourMapping[severity]) !== null && _a !== void 0 ? _a : exports.defaultSeverityColor;
}
exports.getSeveritiesColour = getSeveritiesColour;
function generateIssueSummary(resultsByPlugin, exceptionsByScanType) {
    const testResults = getTestResults(resultsByPlugin, exceptionsByScanType);
    const issueData = testResults.map((i) => i.issuesData);
    const bySeverity = issues_by_severity_1.getIssueCountBySeverity(issueData);
    const issuesBySeverityMessage = formatIssueCountBySeverity({
        critical: bySeverity.critical.length,
        high: bySeverity.high.length,
        medium: bySeverity.medium.length,
        low: bySeverity.low.length,
    });
    // can't use .flat() or .flatMap() because it's not supported in Node 10
    const issues = [];
    for (const result of testResults) {
        issues.push(...result.issues);
    }
    let totalIssues = `${chalk.bold(total_issues_count_1.getTotalIssueCount(issueData))} total issues`;
    if (issuesBySeverityMessage) {
        totalIssues += `: ${issuesBySeverityMessage}`;
    }
    const { count: fixableCount } = fixable_issues_1.hasFixableIssues(testResults);
    const fixableIssues = `${chalk.bold(fixableCount)} fixable issues`;
    return `${exports.PADDING_SPACE}${totalIssues}\n${exports.PADDING_SPACE}${fixableIssues}`;
}
exports.generateIssueSummary = generateIssueSummary;
function getTestResults(resultsByPlugin, exceptionsByScanType) {
    const testResults = [];
    for (const plugin of Object.keys(resultsByPlugin)) {
        const { skipped, failed, succeeded } = resultsByPlugin[plugin];
        testResults.push(...skipped.map((i) => i.original.testResult));
        testResults.push(...failed.map((i) => i.original.testResult));
        testResults.push(...succeeded.map((i) => i.original.testResult));
    }
    if (Object.keys(exceptionsByScanType).length) {
        for (const ecosystem of Object.keys(exceptionsByScanType)) {
            const unresolved = exceptionsByScanType[ecosystem];
            testResults.push(...unresolved.originals.map((i) => i.testResult));
        }
    }
    return testResults;
}
//# sourceMappingURL=show-results-summary.js.map