"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.selectFileForPinning = exports.applyAllFixes = exports.fixIndividualRequirementsTxt = exports.pipRequirementsTxt = void 0;
const debugLib = require("debug");
const pathLib = require("path");
const sortBy = require('lodash.sortby');
const groupBy = require('lodash.groupby');
const update_dependencies_1 = require("./update-dependencies");
const is_supported_1 = require("./../is-supported");
const no_fixes_applied_1 = require("../../../../lib/errors/no-fixes-applied");
const extract_version_provenance_1 = require("./extract-version-provenance");
const requirements_file_parser_1 = require("./update-dependencies/requirements-file-parser");
const standardize_package_name_1 = require("./update-dependencies/standardize-package-name");
const contains_require_directive_1 = require("./contains-require-directive");
const validate_required_data_1 = require("../validate-required-data");
const format_display_name_1 = require("../../../../lib/output-formatters/format-display-name");
const debug = debugLib('snyk-fix:python:requirements.txt');
async function pipRequirementsTxt(entities, options) {
    debug(`Preparing to fix ${entities.length} Python requirements.txt projects`);
    const handlerResult = {
        succeeded: [],
        failed: [],
        skipped: [],
    };
    const { fixable, skipped: notFixable } = await is_supported_1.partitionByFixable(entities);
    handlerResult.skipped.push(...notFixable);
    const ordered = sortByDirectory(fixable);
    let fixedFilesCache = {};
    for (const dir of Object.keys(ordered)) {
        debug(`Fixing entities in directory ${dir}`);
        const entitiesPerDirectory = ordered[dir].map((e) => e.entity);
        const { failed, succeeded, skipped, fixedCache } = await fixAll(entitiesPerDirectory, options, fixedFilesCache);
        fixedFilesCache = Object.assign(Object.assign({}, fixedFilesCache), fixedCache);
        handlerResult.succeeded.push(...succeeded);
        handlerResult.failed.push(...failed);
        handlerResult.skipped.push(...skipped);
    }
    return handlerResult;
}
exports.pipRequirementsTxt = pipRequirementsTxt;
async function fixAll(entities, options, fixedCache) {
    const handlerResult = {
        succeeded: [],
        failed: [],
        skipped: [],
    };
    for (const entity of entities) {
        const targetFile = entity.scanResult.identity.targetFile;
        try {
            const { dir, base } = pathLib.parse(targetFile);
            // parse & join again to support correct separator
            const filePath = pathLib.normalize(pathLib.join(dir, base));
            if (Object.keys(fixedCache).includes(pathLib.normalize(pathLib.join(dir, base)))) {
                handlerResult.succeeded.push({
                    original: entity,
                    changes: [
                        {
                            success: true,
                            userMessage: `Fixed through ${format_display_name_1.formatDisplayName(entity.workspace.path, {
                                type: entity.scanResult.identity.type,
                                targetFile: fixedCache[filePath].fixedIn,
                            })}`,
                            issueIds: getFixedEntityIssues(fixedCache[filePath].issueIds, entity.testResult.issues),
                        },
                    ],
                });
                continue;
            }
            const { changes, fixedMeta } = await applyAllFixes(entity, options);
            if (!changes.length) {
                debug('Manifest has not changed!');
                throw new no_fixes_applied_1.NoFixesCouldBeAppliedError();
            }
            // keep fixed issues unique across files that are part of the same project
            // the test result is for 1 entry entity.
            const uniqueIssueIds = new Set();
            for (const c of changes) {
                c.issueIds.map((i) => uniqueIssueIds.add(i));
            }
            Object.keys(fixedMeta).forEach((f) => {
                fixedCache[f] = {
                    fixedIn: targetFile,
                    issueIds: Array.from(uniqueIssueIds),
                };
            });
            handlerResult.succeeded.push({ original: entity, changes });
        }
        catch (e) {
            debug(`Failed to fix ${targetFile}.\nERROR: ${e}`);
            handlerResult.failed.push({ original: entity, error: e });
        }
    }
    return Object.assign(Object.assign({}, handlerResult), { fixedCache });
}
// TODO: optionally verify the deps install
async function fixIndividualRequirementsTxt(workspace, dir, entryFileName, fileName, remediation, parsedRequirements, options, directUpgradesOnly) {
    const entryFilePath = pathLib.normalize(pathLib.join(dir, entryFileName));
    const fullFilePath = pathLib.normalize(pathLib.join(dir, fileName));
    const { updatedManifest, changes } = update_dependencies_1.updateDependencies(parsedRequirements, remediation.pin, directUpgradesOnly, entryFilePath !== fullFilePath
        ? format_display_name_1.formatDisplayName(workspace.path, {
            type: 'pip',
            targetFile: fullFilePath,
        })
        : undefined);
    if (!changes.length) {
        return { changes };
    }
    if (!options.dryRun) {
        debug('Writing changes to file');
        await workspace.writeFile(pathLib.join(dir, fileName), updatedManifest);
    }
    else {
        debug('Skipping writing changes to file in --dry-run mode');
    }
    return { changes };
}
exports.fixIndividualRequirementsTxt = fixIndividualRequirementsTxt;
async function applyAllFixes(entity, options) {
    const { remediation, targetFile: entryFileName, workspace, } = validate_required_data_1.validateRequiredData(entity);
    const fixedMeta = {};
    const { dir, base } = pathLib.parse(entryFileName);
    const provenance = await extract_version_provenance_1.extractProvenance(workspace, dir, base);
    const upgradeChanges = [];
    /* Apply all upgrades first across all files that are included */
    for (const fileName of Object.keys(provenance)) {
        const skipApplyingPins = true;
        const { changes } = await fixIndividualRequirementsTxt(workspace, dir, base, fileName, remediation, provenance[fileName], options, skipApplyingPins);
        upgradeChanges.push(...changes);
        fixedMeta[pathLib.normalize(pathLib.join(dir, fileName))] = upgradeChanges;
    }
    /* Apply all left over remediation as pins in the entry targetFile */
    const toPin = filterOutAppliedUpgrades(remediation, upgradeChanges);
    const directUpgradesOnly = false;
    const fileForPinning = await selectFileForPinning(entity);
    const { changes: pinnedChanges } = await fixIndividualRequirementsTxt(workspace, dir, base, fileForPinning.fileName, toPin, requirements_file_parser_1.parseRequirementsFile(fileForPinning.fileContent), options, directUpgradesOnly);
    return { changes: [...upgradeChanges, ...pinnedChanges], fixedMeta };
}
exports.applyAllFixes = applyAllFixes;
function filterOutAppliedUpgrades(remediation, upgradeChanges) {
    const pinRemediation = Object.assign(Object.assign({}, remediation), { pin: {} });
    const pins = remediation.pin;
    const normalizedAppliedRemediation = upgradeChanges
        .map((c) => {
        var _a;
        if (c.success && c.from) {
            const [pkgName, versionAndMore] = (_a = c.from) === null || _a === void 0 ? void 0 : _a.split('@');
            return `${standardize_package_name_1.standardizePackageName(pkgName)}@${versionAndMore}`;
        }
        return false;
    })
        .filter(Boolean);
    for (const pkgAtVersion of Object.keys(pins)) {
        const [pkgName, versionAndMore] = pkgAtVersion.split('@');
        if (!normalizedAppliedRemediation.includes(`${standardize_package_name_1.standardizePackageName(pkgName)}@${versionAndMore}`)) {
            pinRemediation.pin[pkgAtVersion] = pins[pkgAtVersion];
        }
    }
    return pinRemediation;
}
function sortByDirectory(entities) {
    const mapped = entities.map((e) => (Object.assign({ entity: e }, pathLib.parse(e.scanResult.identity.targetFile))));
    const sorted = sortBy(mapped, 'dir');
    return groupBy(sorted, 'dir');
}
async function selectFileForPinning(entity) {
    const targetFile = entity.scanResult.identity.targetFile;
    const { dir, base } = pathLib.parse(targetFile);
    const { workspace } = entity;
    // default to adding pins in the scanned file
    let fileName = base;
    let requirementsTxt = await workspace.readFile(targetFile);
    const { containsRequire, matches } = await contains_require_directive_1.containsRequireDirective(requirementsTxt);
    const constraintsMatch = matches.filter((m) => m.includes('c'));
    if (containsRequire && constraintsMatch[0]) {
        // prefer to pin in constraints file if present
        fileName = constraintsMatch[0][2];
        requirementsTxt = await workspace.readFile(pathLib.join(dir, fileName));
    }
    return { fileContent: requirementsTxt, fileName };
}
exports.selectFileForPinning = selectFileForPinning;
function getFixedEntityIssues(fixedIssueIds, issues) {
    const fixed = [];
    for (const { issueId } of issues) {
        if (fixedIssueIds.includes(issueId)) {
            fixed.push(issueId);
        }
    }
    return fixed;
}
//# sourceMappingURL=index.js.map