"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateUpgrades = void 0;
const calculate_relevant_fixes_1 = require("./calculate-relevant-fixes");
const standardize_package_name_1 = require("./standardize-package-name");
function generateUpgrades(requirements, updates, referenceFileInChanges) {
    // Lowercase the upgrades object. This might be overly defensive, given that
    // we control this input internally, but its a low cost guard rail. Outputs a
    // mapping of upgrade to -> from, instead of the nested upgradeTo object.
    const normalizedUpgrades = calculate_relevant_fixes_1.calculateRelevantFixes(requirements, updates, 'direct-upgrades');
    if (Object.keys(normalizedUpgrades).length === 0) {
        return {
            updatedRequirements: {},
            changes: [],
        };
    }
    const changes = [];
    const updatedRequirements = {};
    requirements.map(({ name, originalName, versionComparator, version, originalText, extras, }) => {
        // Defensive patching; if any of these are undefined, return
        if (typeof name === 'undefined' ||
            typeof versionComparator === 'undefined' ||
            typeof version === 'undefined' ||
            originalText === '') {
            return;
        }
        // Check if we have an upgrade; if we do, replace the version string with
        // the upgrade, but keep the rest of the content
        const upgrade = Object.keys(normalizedUpgrades).filter((packageVersionUpgrade) => {
            const [pkgName, versionAndMore] = packageVersionUpgrade.split('@');
            return `${standardize_package_name_1.standardizePackageName(pkgName)}@${versionAndMore}`.startsWith(`${standardize_package_name_1.standardizePackageName(name)}@${version}`);
        })[0];
        if (!upgrade) {
            return;
        }
        const newVersion = normalizedUpgrades[upgrade].upgradeTo.split('@')[1];
        const updatedRequirement = `${originalName}${versionComparator}${newVersion}`;
        changes.push({
            issueIds: normalizedUpgrades[upgrade].vulns,
            from: `${originalName}@${version}`,
            to: `${originalName}@${newVersion}`,
            success: true,
            userMessage: `Upgraded ${originalName} from ${version} to ${newVersion}${referenceFileInChanges
                ? ` (upgraded in ${referenceFileInChanges})`
                : ''}`,
        });
        updatedRequirements[originalText] = `${updatedRequirement}${extras ? extras : ''}`;
    });
    return {
        updatedRequirements,
        changes,
    };
}
exports.generateUpgrades = generateUpgrades;
//# sourceMappingURL=generate-upgrades.js.map