"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.pythonFix = void 0;
const debugLib = require("debug");
const pMap = require("p-map");
const ora = require("ora");
const load_handler_1 = require("./load-handler");
const map_entities_per_handler_type_1 = require("./map-entities-per-handler-type");
const chalk = require("chalk");
const debug = debugLib('snyk-fix:python');
async function pythonFix(entities, options) {
    const spinner = ora({ isSilent: options.quiet, stream: process.stdout });
    const spinnerMessage = 'Looking for supported Python items';
    spinner.text = spinnerMessage;
    spinner.start();
    const handlerResult = {
        python: {
            succeeded: [],
            failed: [],
            skipped: [],
        },
    };
    const results = handlerResult.python;
    const { entitiesPerType, skipped: notSupported } = map_entities_per_handler_type_1.mapEntitiesPerHandlerType(entities);
    results.skipped.push(...notSupported);
    spinner.stopAndPersist({
        text: spinnerMessage,
        symbol: chalk.green('\n✔'),
    });
    await pMap(Object.keys(entitiesPerType), async (projectType) => {
        const projectsToFix = entitiesPerType[projectType];
        if (!projectsToFix.length) {
            return;
        }
        const processingMessage = `Processing ${projectsToFix.length} ${projectType} items`;
        spinner.text = processingMessage;
        spinner.render();
        try {
            const handler = load_handler_1.loadHandler(projectType);
            const { failed, skipped, succeeded } = await handler(projectsToFix, options);
            results.failed.push(...failed);
            results.skipped.push(...skipped);
            results.succeeded.push(...succeeded);
        }
        catch (e) {
            debug(`Failed to fix ${projectsToFix.length} ${projectType} projects.\nError: ${e.message}`);
            results.failed.push(...projectsToFix.map((p) => ({ original: p, error: e })));
        }
        spinner.stopAndPersist({
            text: processingMessage,
            symbol: chalk.green('✔'),
        });
    }, {
        concurrency: 5,
    });
    return handlerResult;
}
exports.pythonFix = pythonFix;
//# sourceMappingURL=index.js.map