package business

import (
	"testing"

	jaegerModels "github.com/jaegertracing/jaeger/model/json"
	"github.com/stretchr/testify/assert"

	"github.com/kiali/kiali/jaeger"
)

var trace1 = jaegerModels.Trace{
	Spans: []jaegerModels.Span{{
		ProcessID:     "t1_process_1",
		OperationName: "reviews.default.svc.cluster.local:8000/*",
		Tags: []jaegerModels.KeyValue{{
			Key:   "dummy",
			Value: "dummy",
		}, {
			Key:   "node_id",
			Value: "sidecar~172.17.0.20~reviews-6d8996bff-ztg6z.default~default.svc.cluster.local",
		}},
	}, {
		ProcessID:     "t1_process_2",
		OperationName: "my-operation",
		Tags: []jaegerModels.KeyValue{{
			Key:   "dummy",
			Value: "dummy",
		}},
	}},
	Processes: map[jaegerModels.ProcessID]jaegerModels.Process{
		"t1_process_1": {
			ServiceName: "reviews.default",
			Tags: []jaegerModels.KeyValue{{
				Key:   "dummy",
				Value: "dummy",
			}},
		},
		"t1_process_2": {
			ServiceName: "my-pod",
			Tags: []jaegerModels.KeyValue{{
				Key:   "dummy",
				Value: "dummy",
			}, {
				Key:   "hostname",
				Value: "my-pod-123456-789abc",
			}},
		},
	},
}
var trace2 = jaegerModels.Trace{
	Spans: []jaegerModels.Span{{
		ProcessID:     "t2_process_1",
		OperationName: "reviews.default.svc.cluster.local:8000/*",
		Tags: []jaegerModels.KeyValue{{
			Key:   "node_id",
			Value: "sidecar~172.17.0.20~reviews-6d8996bff-ztg6z.default~default.svc.cluster.local",
		}},
	}, {
		ProcessID:     "t2_process_2",
		OperationName: "rating.default.svc.cluster.local:8000/*",
		Tags: []jaegerModels.KeyValue{{
			Key:   "node_id",
			Value: "sidecar~172.17.0.20~rating-v2-6d8996bff-ztg6z.default~default.svc.cluster.local",
		}},
	}, {
		ProcessID:     "t2_process_3",
		OperationName: "my-operation",
		Tags:          []jaegerModels.KeyValue{},
	}},
	Processes: map[jaegerModels.ProcessID]jaegerModels.Process{
		"t2_process_1": {
			ServiceName: "reviews.default",
			Tags:        []jaegerModels.KeyValue{},
		},
		"t2_process_2": {
			ServiceName: "rating.default",
			Tags:        []jaegerModels.KeyValue{},
		},
		"t2_process_3": {
			ServiceName: "rating",
			Tags: []jaegerModels.KeyValue{{
				Key:   "hostname",
				Value: "rating-v2-6d8996bff-ztg6z",
			}},
		},
	},
}

func TestMatchingWorkload(t *testing.T) {
	assert := assert.New(t)

	assert.False(matchesWorkload(&trace1, "default", "some-workload"))
	assert.True(matchesWorkload(&trace1, "default", "reviews"))
	assert.True(matchesWorkload(&trace1, "default", "my-pod"))
}

func TestTracesToSpanWithoutFilter(t *testing.T) {
	assert := assert.New(t)

	r := jaeger.JaegerResponse{
		Data:              []jaegerModels.Trace{trace1, trace2},
		JaegerServiceName: "reviews.default",
	}
	spans := tracesToSpans("reviews", &r, nil)
	assert.Len(spans, 2)
	assert.Equal("t1_process_1", string(spans[0].ProcessID))
	assert.Equal("t2_process_1", string(spans[1].ProcessID))

	r = jaeger.JaegerResponse{
		Data:              []jaegerModels.Trace{trace1, trace2},
		JaegerServiceName: "rating.default",
	}
	spans = tracesToSpans("rating", &r, nil)
	assert.Len(spans, 2)
	assert.Equal("t2_process_2", string(spans[0].ProcessID))
	assert.Equal("t2_process_3", string(spans[1].ProcessID))
}

func TestTracesToSpanWithServiceFilter(t *testing.T) {
	assert := assert.New(t)

	r := jaeger.JaegerResponse{
		Data:              []jaegerModels.Trace{trace1, trace2},
		JaegerServiceName: "reviews.default",
	}
	spans := tracesToSpans("reviews", &r, operationSpanFilter("default", "reviews"))
	assert.Len(spans, 2)
	assert.Equal("t1_process_1", string(spans[0].ProcessID))
	assert.Equal("t2_process_1", string(spans[1].ProcessID))

	r = jaeger.JaegerResponse{
		Data:              []jaegerModels.Trace{trace1, trace2},
		JaegerServiceName: "rating.default",
	}
	spans = tracesToSpans("rating", &r, operationSpanFilter("default", "rating"))
	assert.Len(spans, 1)
	assert.Equal("t2_process_2", string(spans[0].ProcessID))
}

func TestTracesToSpanWithWorkloadFilter(t *testing.T) {
	assert := assert.New(t)

	r := jaeger.JaegerResponse{
		Data:              []jaegerModels.Trace{trace1, trace2},
		JaegerServiceName: "reviews.default",
	}
	spans := tracesToSpans("reviews", &r, wkdSpanFilter("default", "reviews"))
	assert.Len(spans, 2)
	assert.Equal("t1_process_1", string(spans[0].ProcessID))
	assert.Equal("t2_process_1", string(spans[1].ProcessID))

	r = jaeger.JaegerResponse{
		Data:              []jaegerModels.Trace{trace1, trace2},
		JaegerServiceName: "rating.default",
	}
	spans = tracesToSpans("rating", &r, wkdSpanFilter("default", "rating-v2"))
	assert.Len(spans, 2)
	assert.Equal("t2_process_2", string(spans[0].ProcessID))
	assert.Equal("t2_process_3", string(spans[1].ProcessID))
}
