"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Yarn2LockParser = void 0;
const yaml = require("yaml");
const yarnCore = require("@yarnpkg/core");
const lock_parser_base_1 = require("./lock-parser-base");
const _1 = require(".");
const config_1 = require("../config");
const errors_1 = require("../errors");
const yarn_utils_1 = require("./yarn-utils");
class Yarn2LockParser extends lock_parser_base_1.LockParserBase {
    constructor() {
        super(_1.LockfileType.yarn2, config_1.config.YARN_TREE_SIZE_LIMIT);
    }
    parseLockFile(lockFileContents) {
        try {
            const rawYarnLock = yaml.parse(lockFileContents);
            delete rawYarnLock.__metadata;
            const dependencies = {};
            const structUtils = yarnCore.structUtils;
            const parseDescriptor = structUtils.parseDescriptor;
            const parseRange = structUtils.parseRange;
            const keyNormalizer = yarn_utils_1.yarnLockFileKeyNormalizer(parseDescriptor, parseRange);
            Object.entries(rawYarnLock).forEach(([fullDescriptor, versionData]) => {
                keyNormalizer(fullDescriptor).forEach((descriptor) => {
                    dependencies[descriptor] = versionData;
                });
            });
            return {
                dependencies,
                lockfileType: _1.LockfileType.yarn2,
                object: dependencies,
                type: _1.LockfileType.yarn2,
            };
        }
        catch (e) {
            throw new errors_1.InvalidUserInputError(`yarn.lock parsing failed with an error: ${e.message}`);
        }
    }
    async getDependencyTree(manifestFile, lockfile, includeDev = false, strict = true) {
        const depTree = await super.getDependencyTree(manifestFile, lockfile, includeDev, strict);
        const meta = { lockfileVersion: 2, packageManager: 'yarn' };
        const depTreeWithMeta = Object.assign(Object.assign({}, depTree), { meta: Object.assign(Object.assign({}, depTree.meta), meta) });
        return depTreeWithMeta;
    }
    getDepMap(lockfile) {
        const yarnLockfile = lockfile;
        const depMap = {};
        for (const [depName, dep] of Object.entries(yarnLockfile.object)) {
            const subDependencies = Object.entries(Object.assign(Object.assign({}, (dep.dependencies || {})), (dep.optionalDependencies || {})));
            depMap[depName] = {
                labels: {
                    scope: _1.Scope.prod,
                },
                name: getName(depName),
                requires: subDependencies.map(([key, ver]) => `${key}@${ver}`),
                version: dep.version,
            };
        }
        return depMap;
    }
    getDepTreeKey(dep) {
        return `${dep.name}@${dep.version}`;
    }
}
exports.Yarn2LockParser = Yarn2LockParser;
function getName(depName) {
    return depName.slice(0, depName.indexOf('@', 1));
}
//# sourceMappingURL=yarn2-lock-parser.js.map