"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var esprima_1 = require("esprima");
var JsonIterator = /** @class */ (function () {
    function JsonIterator(fileContent) {
        this.tokens = [];
        this.i = -1;
        this.currentToken = {
            type: '',
            value: '',
            loc: undefined,
        };
        this.done = false;
        this.previousProps = [];
        this.lastProp = undefined;
        this.tokens = esprima_1.tokenize(fileContent, {
            comment: true,
            loc: true,
        });
        if (!this.tokens.length) {
            throw new SyntaxError('unexpected end of JSON input');
        }
    }
    JsonIterator.prototype[Symbol.iterator] = function () {
        return this;
    };
    JsonIterator.prototype.isCurrentType = function (type) {
        if (!this.currentToken) {
            throw new SyntaxError('unexpected end of JSON input');
        }
        return ((this.currentToken.type === 'Punctuator'
            ? this.currentToken.value
            : this.currentToken.type) === type);
    };
    JsonIterator.prototype.isCurrentValue = function (value) {
        if (!this.currentToken) {
            throw new SyntaxError('unexpected end of JSON input');
        }
        return (this.currentToken.type === 'Punctuator' &&
            this.currentToken.value === value);
    };
    JsonIterator.prototype.expectType = function (type) {
        if (this.isCurrentType(type)) {
            return;
        }
        var error = new SyntaxError("Unexpected type " + this.currentToken.type);
        Object.assign(error, this.currentToken.loc.start);
        throw error;
    };
    JsonIterator.prototype.expectValue = function (value) {
        if (this.isCurrentValue(value)) {
            return;
        }
        var error = new SyntaxError("Unexpected value " + this.currentToken.value);
        Object.assign(error, this.currentToken.loc.start);
        throw error;
    };
    JsonIterator.prototype.skipComments = function () {
        while (this.currentToken &&
            (this.isCurrentType('LineComment') || this.isCurrentType('BlockComment'))) {
            this.next();
        }
    };
    JsonIterator.prototype.next = function () {
        if (this.done) {
            throw new SyntaxError('Unexpected EOF');
        }
        var newToken = this.tokens[++this.i];
        this.currentToken = newToken;
        if (!this.currentToken) {
            this.done = true;
        }
    };
    JsonIterator.prototype.current = function () {
        return this.currentToken;
    };
    JsonIterator.prototype.getCurrentLocation = function () {
        return {
            line: this.currentToken ? this.currentToken.loc.start.line : 0,
            columnStart: this.currentToken ? this.currentToken.loc.start.column : 0,
            columnEnd: this.currentToken ? this.currentToken.loc.end.column : 0,
        };
    };
    //--------------- Prop Handling
    JsonIterator.prototype.setLastProp = function (prop) {
        this.lastProp = prop;
    };
    JsonIterator.prototype.pushLastProp = function () {
        this.previousProps.push(this.lastProp);
        this.lastProp = undefined;
    };
    JsonIterator.prototype.restoreProp = function () {
        this.lastProp = this.previousProps.pop();
    };
    JsonIterator.prototype.getLastProp = function () {
        return this.lastProp !== undefined ? this.lastProp.toString() : '';
    };
    return JsonIterator;
}());
exports.default = JsonIterator;
//# sourceMappingURL=iterator.js.map