"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildJsonTreeMap = void 0;
var tslib_1 = require("tslib");
var iterator_1 = require("./iterator");
var COLON = ':';
var BRACKET_OPEN = '[';
var BRACKET_CLOSE = ']';
var CURLY_BRACKET_OPEN = '{';
var CURLY_BRACKET_CLOSE = '}';
var COMMA = ',';
var MINUS = '-';
var possibleValueTypes = ['String', 'Boolean', 'Null', 'Numeric'];
function buildJsonTreeMap(jsonContent) {
    var iter = new iterator_1.default(jsonContent);
    iter.next();
    iter.skipComments();
    var singleWalk = walk(iter);
    var tree = {
        nodes: singleWalk[0].values,
    };
    // JSON are always single doc
    return {
        0: tree,
    };
}
exports.buildJsonTreeMap = buildJsonTreeMap;
function skipCommentsAndParseObjectOrArray(iter, shouldParseObject) {
    iter.pushLastProp();
    iter.skipComments();
    var nodes = [];
    if (shouldParseObject) {
        nodes = parseObject(iter);
    }
    else {
        nodes = parseArray(iter);
    }
    // bypass }
    iter.next();
    iter.restoreProp();
    return nodes;
}
function parseObject(iter) {
    var nodes = [];
    var firstLoop = true;
    var name;
    while (!iter.isCurrentValue(CURLY_BRACKET_CLOSE)) {
        if (!firstLoop) {
            // key-value pair delimiter
            iter.expectValue(COMMA);
            iter.next();
            iter.skipComments();
            // If there is a trailing comma, we might reach the end
            // ```
            // {
            //   "a": 1,
            // }
            // ```
            if (iter.isCurrentValue(CURLY_BRACKET_CLOSE)) {
                break;
            }
        }
        firstLoop = false;
        iter.expectType('String');
        name = JSON.parse(iter.current().value);
        iter.setLastProp(name);
        iter.next();
        iter.expectValue(COLON);
        iter.next();
        nodes.push.apply(nodes, walk(iter));
    }
    return nodes;
}
function parseArray(iter) {
    var nodes = [];
    var firstLoop = true;
    var i = 0;
    while (!iter.isCurrentValue(BRACKET_CLOSE)) {
        if (!firstLoop) {
            iter.expectValue(COMMA);
            iter.next();
            iter.skipComments();
            if (iter.isCurrentValue(BRACKET_CLOSE)) {
                break;
            }
        }
        firstLoop = false;
        iter.setLastProp(i);
        nodes.push.apply(nodes, walk(iter));
        iter.skipComments();
        i++;
    }
    return nodes;
}
function handleNativeCase(iter) {
    if (!possibleValueTypes.includes(iter.current().type)) {
        throw new Error('failed to find type ' + iter.current().type);
    }
    // turn "apiVersion" -> apiVersion
    // turn 'apiVersion' -> apiVersion
    var currentValue = iter.current().value;
    if ((currentValue.startsWith('"') && currentValue.endsWith('"')) ||
        (currentValue.startsWith("'") && currentValue.endsWith("'"))) {
        currentValue = currentValue.substring(1, currentValue.length - 1);
    }
    var nativeNode = {
        key: iter.getLastProp(),
        lineLocation: iter.getCurrentLocation(),
        values: currentValue,
    };
    iter.next();
    return [nativeNode];
}
function walk(iter) {
    var currentNode = {
        key: iter.getLastProp(),
        lineLocation: iter.getCurrentLocation(),
        values: [],
    };
    switch (iter.current().value) {
        case CURLY_BRACKET_OPEN:
            iter.next();
            return [
                tslib_1.__assign(tslib_1.__assign({}, currentNode), { values: skipCommentsAndParseObjectOrArray(iter, true) }),
            ];
        case BRACKET_OPEN:
            iter.next();
            return skipCommentsAndParseObjectOrArray(iter, false).map(function (value, i) {
                return {
                    key: iter.getLastProp() + "[" + i + "]",
                    lineLocation: value.lineLocation,
                    values: value.values,
                };
            });
        case MINUS:
            iter.next();
            break;
    }
    return handleNativeCase(iter);
}
//# sourceMappingURL=parser.js.map