"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTfTreeMap = void 0;
var types_1 = require("./types");
var utils_1 = require("./utils");
function buildTfTreeMap(tfContent) {
    var multiLineComment = false;
    var multiLinePhrase = { phrase: null };
    var nodes = [];
    var currNode = null;
    var stateQueue = [];
    var duringTypeParsing = false;
    var tfFileLines = tfContent.split('\n');
    for (var i = 0; i < tfFileLines.length; i++) {
        var line = { content: tfFileLines[i].trim(), number: i };
        var lineState = utils_1.getLineState(line, multiLineComment, multiLinePhrase);
        multiLineComment = lineState.multiCommentLine;
        if (lineState.ignoredLine) {
            continue;
        }
        if (multiLinePhrase.phrase) {
            //Multi-line phrase ended - line is not ignored any more
            multiLinePhrase.phrase = null;
            continue;
        }
        var topType = void 0;
        if (stateQueue.length > 0) {
            topType = stateQueue[stateQueue.length - 1].type;
        }
        var lineType = utils_1.getLineType(line, topType);
        // In case of array value where is multiline and not completed yet
        // or
        // Object inside array
        //   -> can skip this line
        if (topType === types_1.TFLineTypes.ARRAY_START &&
            ![
                types_1.TFLineTypes.ARRAY_END,
                types_1.TFLineTypes.OBJECT_START,
                types_1.TFLineTypes.OBJECT_START_AND_END,
            ].includes(lineType)) {
            continue;
        }
        switch (lineType) {
            case types_1.TFLineTypes.IGNORE:
                continue;
            case types_1.TFLineTypes.TYPE_START:
                currNode = getTypeDetailsFromLine(line, nodes, stateQueue);
                duringTypeParsing = true;
                continue;
            case types_1.TFLineTypes.TYPE_START_AND_END:
                currNode = getTypeDetailsFromLine(line, nodes, stateQueue);
                stateQueue = [];
                duringTypeParsing = false;
                continue;
            case types_1.TFLineTypes.TYPE_END:
                if (topType !== types_1.TFLineTypes.SUB_TYPE) {
                    throw new SyntaxError('Invalid TF Input - End of type object without sub type');
                }
                stateQueue = [];
                duringTypeParsing = false;
                continue;
            case types_1.TFLineTypes.OBJECT_START:
                currNode = getComplexObjectNode(line, stateQueue, utils_1.Charts.openBracketsObject, lineType);
                continue;
            case types_1.TFLineTypes.OBJECT_START_AND_END:
                getComplexObjectNode(line, stateQueue, utils_1.Charts.openBracketsObject, lineType);
                stateQueue.pop();
                continue;
            case types_1.TFLineTypes.OBJECT_END: {
                currNode = handleComplexObjectEnd(currNode, stateQueue, types_1.TFLineTypes.OBJECT_START);
                continue;
            }
            case types_1.TFLineTypes.FUNCTION_START:
                currNode = getComplexObjectNode(line, stateQueue, utils_1.Charts.openFunction, lineType);
                continue;
            case types_1.TFLineTypes.FUNCTION_START_AND_END:
                getComplexObjectNode(line, stateQueue, utils_1.Charts.openFunction, lineType);
                stateQueue.pop();
                continue;
            case types_1.TFLineTypes.FUNCTION_END:
                currNode = handleComplexObjectEnd(currNode, stateQueue, types_1.TFLineTypes.FUNCTION_START);
                continue;
            case types_1.TFLineTypes.STRING:
            case types_1.TFLineTypes.MULTILINE_STRING:
            case types_1.TFLineTypes.ARRAY_START_AND_END: {
                if (!currNode) {
                    throw new SyntaxError('Unexpected TF input - Simple object without parent node');
                }
                var simpleNode = getSimpleNode(line);
                currNode.values.push(simpleNode);
                if (lineType === types_1.TFLineTypes.MULTILINE_STRING) {
                    multiLinePhrase = utils_1.getMultiLinePhrase(line);
                }
                continue;
            }
            case types_1.TFLineTypes.ARRAY_START: {
                if (!currNode) {
                    throw new SyntaxError('Unexpected TF input - Simple object without parent node');
                }
                var simpleNode = getSimpleNode(line);
                if (simpleNode.values === utils_1.Charts.openBracketsArray) {
                    simpleNode.values = [];
                }
                currNode.values.push(simpleNode);
                stateQueue.push({
                    structure: simpleNode,
                    type: lineType,
                });
                continue;
            }
            case types_1.TFLineTypes.ARRAY_END: {
                stateQueue.pop();
                continue;
            }
            default:
                throw new SyntaxError("Invalid TF input - Unhandled line type " + types_1.TFLineTypes[lineType]);
        }
    }
    if (duringTypeParsing || stateQueue.length !== 0) {
        throw new SyntaxError('Invalid TF input - Broken file');
    }
    if (nodes.length === 0) {
        throw new SyntaxError('Invalid TF input - No nodes were parsed');
    }
    // TF are always single doc
    return {
        0: { nodes: nodes },
    };
}
exports.buildTfTreeMap = buildTfTreeMap;
function getTypeDetailsFromLine(currentLine, nodes, stateQueue) {
    var lineContent = currentLine.content.split(utils_1.Charts.space);
    var resourceType = lineContent[1].replace(/"/g, '');
    var objectType = lineContent[0];
    if (resourceType === utils_1.Charts.openBracketsObject) {
        if (types_1.TerraformValidConfigurationTypes.includes(objectType)) {
            //Support Terraform configurations settings object
            resourceType = '';
        }
        else {
            throw new SyntaxError('Invalid TF input - Type object without sub type');
        }
    }
    var headNode = getTypeNode(objectType, currentLine, nodes);
    if (lineContent[2] &&
        lineContent[2] !== null &&
        lineContent[2] !== utils_1.Charts.openBracketsObject) {
        var resourceName = lineContent[2].replace(/"/g, '');
        resourceType = resourceType + "[" + resourceName + "]";
    }
    var subHeadNode = getSubTypeNode(headNode, resourceType, currentLine);
    stateQueue.push({ structure: headNode, type: types_1.TFLineTypes.TYPE_START });
    stateQueue.push({ structure: subHeadNode, type: types_1.TFLineTypes.SUB_TYPE });
    return subHeadNode;
}
function getTypeNode(objectType, line, nodes) {
    var headNode = nodes.find(function (node) { return node.key === objectType; });
    if (!headNode) {
        headNode = utils_1.getNode(objectType, line, []);
        nodes.push(headNode);
    }
    return headNode;
}
function getSubTypeNode(headNode, resourceType, line) {
    var headerSubTypes = headNode.values;
    var subHeadNode = headerSubTypes.find(function (node) { return node.key === resourceType; });
    if (!subHeadNode) {
        subHeadNode = utils_1.getNode(resourceType, line);
        headNode.values.push(subHeadNode);
    }
    return subHeadNode;
}
function getComplexObjectNode(line, stateQueue, splitByChart, lineType) {
    var key = line.content
        .split(splitByChart)[0]
        .split(utils_1.Charts.equal)[0]
        .trim();
    var objectNode = utils_1.getNode(key, line);
    stateQueue.push({ structure: objectNode, type: lineType });
    return objectNode;
}
function getSimpleNode(line) {
    var _a = line.content.split(utils_1.Charts.equal), key = _a[0], value = _a[1];
    return utils_1.getNode(key.trim(), line, value.trim().replace(/"/g, ''));
}
function handleComplexObjectEnd(currNode, stateQueue, startLineType) {
    var topState = stateQueue[stateQueue.length - 1];
    if (topState.type !== startLineType || stateQueue.length === 0) {
        throw new SyntaxError('Invalid TF Input - Object end without start');
    }
    if (!currNode) {
        throw new SyntaxError('Invalid TF input - Object without parent');
    }
    stateQueue.pop();
    topState = stateQueue[stateQueue.length - 1];
    var topNode = topState.structure;
    topNode.values.push(currNode);
    return topNode;
}
//# sourceMappingURL=parser.js.map