"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractMeta = exports.groupEntitiesPerScanType = exports.fix = void 0;
const debugLib = require("debug");
const pMap = require("p-map");
const ora = require("ora");
const chalk = require("chalk");
const stripAnsi = require("strip-ansi");
const outputFormatter = require("./lib/output-formatters/show-results-summary");
const load_plugin_1 = require("./plugins/load-plugin");
const error_to_user_message_1 = require("./lib/errors/error-to-user-message");
const debug = debugLib('snyk-fix:main');
async function fix(entities, options = {
    dryRun: false,
    quiet: false,
    stripAnsi: false,
}) {
    const spinner = ora({ isSilent: options.quiet, stream: process.stdout });
    let resultsByPlugin = {};
    const entitiesPerType = groupEntitiesPerScanType(entities);
    const exceptionsByScanType = {};
    await pMap(Object.keys(entitiesPerType), async (scanType) => {
        try {
            const fixPlugin = load_plugin_1.loadPlugin(scanType);
            const results = await fixPlugin(entitiesPerType[scanType], options);
            resultsByPlugin = Object.assign(Object.assign({}, resultsByPlugin), results);
        }
        catch (e) {
            debug(`Failed to processes ${scanType}`, e);
            exceptionsByScanType[scanType] = {
                originals: entitiesPerType[scanType],
                userMessage: error_to_user_message_1.convertErrorToUserMessage(e),
            };
        }
    }, {
        concurrency: 3,
    });
    const fixSummary = await outputFormatter.showResultsSummary(resultsByPlugin, exceptionsByScanType);
    const meta = extractMeta(resultsByPlugin, exceptionsByScanType);
    spinner.start();
    if (meta.fixed > 0) {
        spinner.stopAndPersist({
            text: 'Done',
            symbol: chalk.green('✔'),
        });
    }
    else {
        spinner.stop();
    }
    return {
        results: resultsByPlugin,
        exceptions: exceptionsByScanType,
        fixSummary: options.stripAnsi ? stripAnsi(fixSummary) : fixSummary,
        meta,
    };
}
exports.fix = fix;
function groupEntitiesPerScanType(entities) {
    var _a, _b, _c;
    const entitiesPerType = {};
    for (const entity of entities) {
        // TODO: group all node
        const type = (_c = (_b = (_a = entity.scanResult) === null || _a === void 0 ? void 0 : _a.identity) === null || _b === void 0 ? void 0 : _b.type) !== null && _c !== void 0 ? _c : 'missing-type';
        if (entitiesPerType[type]) {
            entitiesPerType[type].push(entity);
            continue;
        }
        entitiesPerType[type] = [entity];
    }
    return entitiesPerType;
}
exports.groupEntitiesPerScanType = groupEntitiesPerScanType;
function extractMeta(resultsByPlugin, exceptionsByScanType) {
    const failed = outputFormatter.calculateFailed(resultsByPlugin, exceptionsByScanType);
    const fixed = outputFormatter.calculateFixed(resultsByPlugin);
    return { fixed, failed };
}
exports.extractMeta = extractMeta;
//# sourceMappingURL=index.js.map